/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "IfoResponse.hh"
#include "LscCalib.hh"
#include <fstream>
#include <stdexcept>

using namespace std;

//======================================  Default constructor
IfoResponse::IfoResponse(void) 
  : mDf(0.0), mLastAlpha(0.0), mLastAlphaBet(0.0)
{}

//======================================  LscCalib input file
IfoResponse::IfoResponse(const std::string& title, const std::string& file) 
  : mDf(0.0), mLastAlpha(0.0), mLastAlphaBet(0.0)
{
    read(title, file);
}

//======================================  Clone and IfoResponse filter
IfoResponse* 
IfoResponse::clone(void) const {
    return new IfoResponse(*this);
}

//======================================  Read in the calibration info
void 
IfoResponse::read(const std::string& title, const std::string& file) {
    // ifstream xin(file.c_str(), ios::in);
    if (!file.empty()) readXml(title, file);
}

//======================================  Read a calibration xml file
void 
IfoResponse::readXml(const std::string& title, const std::string& istr) {
    LscCalib c(title, istr);
    if (mDf == 0.0) {
        mSensing      = c.refSensingFunction();
	mOpenLoopGain = c.refOpenLoopGain();
    } else {
        mSensing      = c.refSensingFunction().interpolate(mFmin, mFmax, mDf);
	mOpenLoopGain = c.refOpenLoopGain().interpolate(mFmin, mFmax, mDf);
    }
    mAlpha        = c.refAlpha();
    mAlphaBet     = c.refAlphaBeta();
}

//======================================  Read a  calibration frame
void 
IfoResponse::readFrame(const std::string& title, const std::string& istr) {
    throw runtime_error("IfoResponse::readFrame not implemented");
}

//======================================  Set the frequenxy range.
void 
IfoResponse::setFreq(double fMin, double fMax, double dF) {
    if (dF <= 0) {
       throw runtime_error("IfoResponse::setFreq: Invalid delta-F in FSeries");
    }
    if (!mSensing.empty() && !mOpenLoopGain.empty()) {
        mSensing      = mSensing.interpolate(fMin, fMax, dF);
        mOpenLoopGain = mOpenLoopGain.interpolate(fMin, fMax, dF);
    }
    mFmin = fMin;
    mFmax = fMax;
    mDf   = dF;
}

//======================================  Filter an FSeries.
void 
IfoResponse::Apply(const FSeries& fIn, FSeries& fOut) {
    if (fIn.empty()) {
        fOut.clear();
	return;
    }
    double dF = fIn.getFStep();
    if (dF <= 0) throw runtime_error("Invalid delta-F in FSeries");
    if (mDf <= 0) {
        setFreq(0.0, fIn.getHighFreq(), dF);
    } else if (dF != mDf) {
        throw runtime_error("Invalid delta-F in FSeries");
    }
    setResponse(fIn.getStartTime());
    fOut  = fIn.extract(mFmin, mFmax-mFmin);
    fOut *= mResponse;
}

//======================================  Filter a DFT
void 
IfoResponse::Apply(const containers::DFT& fIn, containers::DFT& fOut) {
    if (fIn.empty()) {
        fOut.clear();
	return;
    }
    double dF = fIn.getFStep();
    if (dF <= 0) throw runtime_error("Invalid delta-F in DFT");
    if (mDf <= 0) {
        setFreq(0.0, fIn.getHighFreq(), dF);
    } else if (dF != mDf) {
        throw runtime_error("Invalid delta-F in DFT");
    }
    setResponse(fIn.getStartTime());
    fOut  = fIn.extract_dft(mFmin, mFmax-mFmin);
    fOut.refDVect() *= *mResponse.refDVect();
}

//======================================  Filter an FSpectrum.
void 
IfoResponse::Apply(const FSpectrum& fIn, FSpectrum& fOut) {
    throw runtime_error("IfoResponse::Apply(FSpectrum) Not implemented");
}

//======================================  Filter an FSpectrum.
void 
IfoResponse::Apply(const containers::PSD& fIn, containers::PSD& fOut) {
    throw runtime_error("IfoResponse::Apply(PSD) Not implemented");
}

//======================================  Get a best guess value from a TSeries
inline double
guessTVal(const TSeries& ts, const Time& t) {
    Interval dT = t - ts.getStartTime();
    long inx = long(dT/ts.getTStep() + 0.5);
    if (inx < 0 || inx >= long(ts.getNSample())) {
         throw runtime_error("Time not covered by TSeries");
    }
    return ts.getDouble(inx);
}

//======================================  Calculate the response fc.
void 
IfoResponse::setResponse(const Time& t) const {
    if (t < mAlpha.getStartTime() || t >= mAlpha.getEndTime()) {
        throw runtime_error("Alpha, beta not available for data");
    }
    float alpha = guessTVal(mAlpha, t);
    if (alpha == 0.0) alpha = 1.0;
    float alphabet = guessTVal(mAlphaBet, t);
    if (alphabet == 0.0) alphabet = 1.0;
    if (alpha == mLastAlpha && alphabet == mLastAlphaBet) return;
    //cout << "Setting response function for t=" << t << " alpha=" << alpha 
    //	 << ", alpha*beta=" << alphabet << endl;
    FSeries temp = mOpenLoopGain.extract(mFmin, mFmax-mFmin);
    temp *= alphabet / alpha;
    temp += 1.0 / alpha;
    mResponse  = mSensing.extract(mFmin, mFmax-mFmin);
    mResponse /= temp;
    mLastAlpha    = alpha;
    mLastAlphaBet = alphabet;
}
