/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef DMTSERVER_HH
#define DMTSERVER_HH

#include "MonServer.hh"
#include "SigFlag.hh"
#include "Interval.hh"
#include "Time.hh"

/**  <h2> Introduction </h2>
  *  The %DMTServer class is a base class for dataless processes that produce 
  *  objects for the DMT viewer. The %DMTServer class is designed to provide 
  *  a similar environment whether running in the foreground (\e i.e. under 
  *  root) or in the background. The functions provided by the environment are:
  *  <ul>
  *    <li> Parse common command line arguments.
  *    <li> Dispatch a display calculation method.
  *    <li> Handle dmtviewer requests
  *    <li> Handling attention interrupts.
  *    <li> Clean up on termination.
  *  </ul>
  *
  *  <h2> Writing a %DMTServer Monitor</h2>
  *  The strategy for writing a monitor program is to create a class 
  *  (MyClass) based on %DMTServer including a constructor, destructor and
  *  ProcessData method. An attention handler may also be provided if
  *  desired. The operations to be performed by each of these methods
  *  are:
  *
  *  <b> Constructor (MyClass(int argc, const char* argv[])) </b>
  *  <ul>
  *    <li> %Process user specific arguments
  *    <li> Define parameters (see TrigPar class).
  *    <li> Configure the process.
  *  </ul>
  *
  *  <b> Destructor (~MyClass(void)) </b>
  *  <ul>
  *    <li> Save modified parameters
  *    <li> Print results and statistics.
  *  </ul>
  *
  *  <b> Epoch processing (void ProcessData(void)) </b>
  *  <ul>
  *    <li> %Process data from one epoch.
  *    <li> Keep statistics.
  *  </ul>
  *
  *  <b> Interrupt handler (void Attention(void)) </b>
  *  <ul>
  *    <li> Monitor dependent interrupt processing.
  *  </ul>
  *
  *  The \c EXECSVR(myclass) macro is used to generate the main
  *  function. Definition and implementation templates are available 
  *  from DatTemplate.hh and DatTemplate.cc.
  *
  *  The \c EXEROOT(myclass) macro is used to generate a main function
  *  with root graphics. Note that the "TROOT.h" and "TApplication.h"
  *  header files must be included in your monitor implementation file 
  *  if \c EXEROOT is used.
  *
  *  <h2> Running the Monitor</h2>
  *  The base class constructor recognizes the following command line 
  *  arguments: 
  *  <table>
  *  <tr><td>-debug \<lvl\>     </td>
  *      <td>Set debug level to \<lvl\> </td></tr>
  *  <tr><td>-dt \<delta-t\>    </td>
  *      <td>Set reschedule time to \<delta-t\> </td></tr>
  *  <tr><td>-name \<serverid\></td>
  *      <td>Set server name </td></tr>
  *  </table>
  *
  *  \brief Data processing monitor base class.
  *  @author John G. Zweizig
  *  @version $id:$
  */
class DMTServer : public MonServer {
public:
    /**  The class constructor gets all the command line arguments in the 
      *  usual Unix/C argument list format. It sets up the environment 
      *  for processing and opens the frame input stream as directed by 
      *  the command line arguments.
      *  \brief Class constructor.
      *  @param argc Number of arguments specified.
      *  @param argv Pointers to arguments.
      */
  DMTServer(int argc, const char* argv[]);

    /**  Close the input stream and go away.
      *  \brief Class destructor.
      */
  virtual ~DMTServer(void);


    /**  finish stops processing by setting a flag that breaks the main 
      *  loop. Note that finish returns to the calling function (generally 
      *  the user's ProcessData) which must exit normally for the process 
      *  to terminate.
      *  \brief Stop data processing and go away cleanly.
      */
  void finish(void);

    /**  Debug returns the debug level as specified by the "-debug nnn" 
      *  argument. If "-debug" isn't specified in the command line the 
      *  debug level defaults to 0.
      *  \brief Get the debug level.
      *  \return Debug print level
      */
  int Debug(void) const;

  /**  Get the current rescheduling interval.
    *  \brief Get rescheduling interval
    *  \return Rescheduling interval.
    */ 
  Interval getReschedule(void) const;

  /**  Get the scheduled wake-up time.
    *  \brief Get scheduled wake-up time
    *  \return Scheduled wake-up time.
    */ 
  Time getWakeup(void) const;

    /**  Test for a %DMTServer command line argument. All %DMTServer command 
      *  line arguments take a value in the following argument position.
      *  \brief Test if string is a %DMTServer command line argument.
      *  \param arg Argument string to be tested.
      *  \return true if the specified string is a %DMTServer argument.
      */
  bool isServerArg(const std::string& arg) const;

    /**  MainLoop redispatches ProcessData for each epoch. MainLoop
      *  exits when a \c SIGINT or \c SIGTERM signal is caught, or when finish()
      *  is called by a subsidiary function (usually by ProcessData()). This 
      *  function is provided by the base class.
      *  \brief Loop over data epochs.
      *  \return Zero for normal operation
      */
  int MainLoop(void);

  /**  Set the rescheduling interval to the specified value.
    *  \brief Set reschedule interval.
    *  \param dt Rescheduling interval.
    */
  void setReschedule(Interval dt);

  /**  Set the next scheduled wake-up time.
    *  \brief Set the wake-up time.
    *  \param t Scheduled wake-up time.
    */
  void setWakeup(const Time& t);

  /**  Sleep until the specified time.
    *  \brief Sleep.
    *  \param t Target wake-up time
    */
  void sleep_til(const Time& t);

    /**  ProcessData performs the monitor specific processing data from
      *  a single epoch. It must be defined as part of the derived monitor 
      *  class implementation.
      *  \brief %Process one epoch (Supplied by user).
      *  \param t Current time.
      */
  virtual void ProcessData(const Time& t) = 0;

    /**  Attention performs the monitor specific attention interrupt 
      *  processing. Its implementation is optional. Execution of 
      *  the attention handler is deferred until the processing of the 
      *  current epoch has completed. This means that there may be a 
      *  significant delay between the delivery of the \c SIGUSR1 signal and
      *  the invocation of the handler. The ROOT version uses the \c ROOT 
      *  event handler as a source of When used with \c ROOT, the default 
      *  action is to terminate the monitor. When used in the background, 
      *  it defaults to no action. The execution of the attention handler is 
      *  synchronized to avoid interrupting the ProcessData function.
      *  \brief Attention interrupt handler (Supplied by user).
      */
  virtual void Attention(void);
private:
    bool     mActive;
    Interval mReschedule;
    Time     mWakeup;
    SigFlag mTerm;
    SigFlag mAttn;
    int     mDebug;
};

//
//   The EXECSVR macro generates the main function to execute a server
//   based on DMTServer. The argument is whatever name you have chosen
//   for your class.
//
#define EXECSVR(myclass) \
int main(int argc, const char *argv[]) { \
    myclass MyObject(argc,argv); \
    MyObject.MainLoop(); \
    return 0;}

//======================================  inline methods.
inline int
DMTServer::Debug(void) const {
    return mDebug;
}

inline Time
DMTServer::getWakeup(void) const {
    return mWakeup;
}

inline Interval
DMTServer::getReschedule(void) const {
    return mReschedule;
}

#endif   // !defined(DMTSERVER_HH)
