#ifndef SQUAREWAVE_HH
#define SQUAREWAVE_HH
#include "Chirp.hh"
#include "Interval.hh"

/**  Square wave template generator.
  *  SquareWave is a template generator based on the Chirp class. It generates 
  *  a sqaue wave.
  */
class SquareWave : public Chirp {
public:

  /**  SquareWave Constructor. 
    *  Generate a square wave template of frequency f, amplitude A and 
    *  starting phase phi0.
    *  The template extends from T0 for a time dT.
    */
  explicit SquareWave(double f, double ampl = 1.0, double phi0=0.0, 
                const Interval& dT=Interval(60.0), 
		const Time& t0=Time(0));

  /**  SquareWave destructor.
    */
  ~SquareWave(void) {}

  /**  Frequency domain template bin.
    *  Complex Frequency domain representation of the template at frequency
    *  Freq.
    */
  fComplex Fspace(double Freq, double dF=0) const;

  /**  Time domain template bin.
    *  Tspace returns the waveform value at time 't0'.
    */ 
  double Tspace(const Time& t0) const;

  /**  Get the t0.
    *  Return the template start time. 
    */
  Time getT0(void)   const;

  /**  Get the End time.
    *  Return the template end time. The end-time is defined to be the 
    *  time at which the template frequency reaches a maximum.
    */
  Time getTEnd(void) const;

  /**  Get the Critical time.
    *  Return the template critical time. The critical time is defined 
    *  explicitly or implicitly when the object is constructed.
    */
  Time getTc(void) const;

  /**  Time versus Frequency.
    *  TvsF(f) returns the absolute (GPS) time at which the in-spiral 
    *  frequency is exactly 'f'.
    */
  Time   TvsF(double f) const;

  /**  Waveform Frequency.
    *  freq(t) returns the in-spiral frequency at Absolute time t.
    */
  double freq(const Time& t) const;

  /**  Waveform Phase angle.
    *  phi(t) returns the phase angle at absolute time t.
    */
  double  phi(const Time& t) const;

  /**  Amplitude.
    *  Ampl(t) returns the wave-form amplitude.
    */
  double Ampl(const Time& t) const;

  /**  Set the fnunction amplitude.
    *  @memo set the amplitude.
    */
  void setAmp(double amp);

private:
  //---------------------------  External Parameters
  /**  Frequency.
    *  The sinusoidal frequency.
    */
  double mOmega;

  /**  Template start time.
    *  Absolute time of the start of template generation.
    */
  Time   mT0;

  /**  Template limit time.
    *  The template limit time is defined to be the absolute (GPS) time at 
    *  which the template ends.
    */
  Time   mTl;

  /**  Critical time.
    *  The critical time is the absolute (GPS) time at which the Phi_c is 
    *   defined.
    */
  Time   mTc;

  /**  Phase offset.
    *  This is the phase at Tc.
    */
  double mPhic;

  /**  SquareWave amplitude.
    */
  double mAmpl;
};

#endif  //  SQUAREWAVE_HH
