/**  @name trendtable
  *  { \Large { \bf Introduction } }
  * 
  *  Trendtable is a utility program to extract trend data into a  text file.
  *  one channel is extracted at a time and all times are written to the 
  *  output file in six columns containing.
  *  \begin{enumerate}
  *  \item GPS time
  *  \item Mean value
  *  \item Standard deviation (sigma)
  *  \item Minimum
  *  \item Maximum
  *  \item Number of items in average
  *  \end{enumerate}
  *  By default all time bins from the earliest to the latest trended times 
  *  are written to the output file. A flag is provided to suppress trend 
  *  time bins that have a zero count.
  *
  *  { \Large { \bf Running trendtable } }
  *
  *  The #trendtable# command has the following syntax:
  *
  *  \begin{verbatim}
      trendtable [--startgps | -s <start>] [--endgps | -e <end>] \
                 [--duration | -d <delta-t>] [--outfile | -o <ofile>] \
                 [--noempty | -n] <channel> <dirs>
     \end{verbatim}
  *
  *  Where #<channel># is the root name of the trend channel and #<dirs># is
  *  a wild-carded list of directories containing the trend frame files. If 
  *  the start and stop times are not specified, the trends are written from
  *  GPS 700000000 (~Mar 13 2002) for 999999999 seconds. If #--outfile# is 
  *  omitted or set to #"-"#, the table is written to the standard output. 
  *  The #--noempty# specifier suppresses all rows with zero count fields.
  *
  *  @memo Trend frame to table conversion utility
  *  @author J. Zweizig
  *  @version 1.1; last modified June  4, 2006
  */
//@{
//@}

//======================================  Include header files
#include "ReadTrend.hh"
#include <string>
#include <iostream>
#include <fstream>

using namespace std;

//======================================  TrendTable main program
int 
main(int argc, const char* argv[]) {
    Time tStart(0), tEnd(0);
    Interval dT(0);
    string ofile("-");
    bool noempty(false);

    //----------------------------------  Get argumenat
    bool syntax(argc < 3);
    for (int i=1; i<argc-2; ++i) {
        string argi(argv[i]);
	if (argi == "--startgps" || argi == "-s") {
	    tStart = Time(strtoul(argv[++i], 0, 0));
	} else if (argi == "--endgps" || argi == "-e") {
	    tEnd   = Time(strtoul(argv[++i], 0, 0));
	} else if (argi == "--duration" || argi == "-d") {
	    dT     = Interval(strtod(argv[++i], 0));
	} else if (argi == "--outfile" || argi == "-o") {
	    ofile  = argv[++i];
	} else if (argi == "--noempty" || argi == "-n") {
	    noempty  = true;
	} else {
	    cerr << "Unrecognized argument: " << argi << endl;
	    syntax = true;
	    break;
	}
    }

    //----------------------------------  Bail if syntax error
    if (syntax) {
        cerr << "Command syntax error. Syntax:" << endl;
	cerr << "trendtable [--startgps | -s <start>] [--endgps | -e <end>] \\"
	     << endl;
	cerr << "           [--duration | -d <delta-t>] [--noempty | -n] \\"
	     << endl;
	cerr << "           [--outfile | -o <file>] <channel> <directories>" 
	     << endl;
	return 1;
    }

    //----------------------------------  Fill in arguments
    string chan(argv[argc-2]);
    string direc(argv[argc-1]);
    if (!tStart) {
        if (!tEnd || !dT) tStart = Time(700000000);
	else              tStart = tEnd - dT;
    }
    if (!dT) {
        if (!tEnd) dT = Interval(999999999);
	else       dT = tEnd - tStart;
    }

    //-----------------------------------  Open output file.
    ostream* out(0);
    if (ofile == "-") {
        out = &cout;
    } else {
        out = new ofstream(ofile.c_str());
	if (!dynamic_cast<ofstream*>(out)->is_open()) {
	    cerr << "Unable to open file :" << ofile << endl; 
	    return 2;
	}
    }

    //----------------------------------  Construct trend reader
    ReadTrend rt(direc.c_str());

    //----------------------------------  Build I/O vectors & read trends
    ReadTrend::string_vect  chanv(5);
    for (int i=0; i<5; i++) chanv[i] = chan;
    ReadTrend::type_vect    typev(5);
    typev[0] = ReadTrend::kMean;
    typev[1] = ReadTrend::kSigma;
    typev[2] = ReadTrend::kMin;
    typev[3] = ReadTrend::kMax;
    typev[4] = ReadTrend::kCount;
    ReadTrend::tseries_vect teaser(5);
    rt.getSeries(chanv, typev, tStart, dT, teaser);

    //-----------------------------------  Print out the trend data.
    long N = teaser[1].getNSample();
    *out << "#   Trends for channel: " << chan << endl;
    *out << "#  GPS        Mean       Sigma     Min     Max     Count"
	 << endl;
    for (int i=0; i<N; i++) {
        if (noempty && teaser[4].getDouble(i) == 0) continue;
        *out << teaser[0].getBinT(i).getS();
	for (int j=0; j<5; j++) *out << "  " << teaser[j].getDouble(i);
	*out << endl;
    }

    //-----------------------------------  Close output file.
    if (ofile != "-") dynamic_cast<ofstream*>(out)->close();
}
