/* -*- mode: c++; c-basic-offset: 4; -*- */
#include <iostream>
#include <fstream>
#include <cstring>
#include "ParseLine.hh"
#include "xsil/MetaIO.hh"
#include "xsil/ligolw.hh"
#include "xsil/table.hh"
#include "xsil/Xwriter.hh"
#include "html/document.hh"
#include "html/table.hh"
#include "html/writer.hh"
#ifdef HAVE_HDF5
#include "tableIO_hdf.hh"
#endif

//======================================  Table classes
#include "table.hh"

using namespace std;
using namespace table;

//======================================  Default name table.
static const char* defNames[] = {
  "A",  "B",  "C",  "D",  "E",  "F",  "G",  "H",  "I",  "J",
  "K",  "L",  "M",  "N",  "O",  "P",  "Q",  "R",  "S",  "T",
  "U",  "V",  "W",  "X",  "Y",  "Z",  "AA", "AB", "AC", "AD",
  "AE", "AF", "AG", "AH", "AI", "AJ", "AK", "AL", "AM", "AN",
  "AO", "AP", "AQ", "AR", "AS", "AT", "AU", "AV", "AW", "AX"
};

//======================================  Table constructor
Table::Table(const string& title, const string& in) 
  : mRow(0), mTitle(title), mDebug(0)
{
    if (!in.empty()) read(in);
}

//======================================   Add a row to each column
void
Table::addRow(row_index b4, const Cell& cell) {
    col_index N = size();
    for (col_index i=0; i<N; i++) {
	mTable[i].insertRow(b4, cell);
    }
    mRow++;
}

//======================================   Find a column
Table::col_index
Table::find(const string& cName) const {
    col_index N=size();
    for (col_index i=0; i<N; ++i) if (mTable[i].titleq(cName)) return i;
    return N;
}

//======================================  Reference a named column (const)
const Column&
Table::findColumn(const string& cName) const {
    col_index i = find(cName);
    if (i == size()) throw runtime_error(string("Undefined column: ")+cName);
    return mTable[i];
}

//======================================  Reference a named column.
Column&
Table::findColumn(const string& cName) {
    col_index i = find(cName);
    if (i == size()) throw runtime_error(string("Undefined column: ")+cName);
    return mTable[i];
}


//======================================   Add a column
void 
Table::merge(col_index iCol, col_index jCol, const std::string& join) {
    for (Column::row_index i=0; i<mRow; ++i) {
        if (!join.empty()) mTable[iCol].refCell(i).append(join);
        mTable[iCol].refCell(i).append(mTable[jCol].refCell(i).getString());
    }
}

//======================================   Add a column
void 
Table::insertColumn(col_index inx, const std::string& title) {
    col_index nCol = mTable.size();
    if (inx > nCol) throw std::range_error("Column number out of range");
    string name = title;
    Column tCol(title);
    if (title.empty()) tCol.setTitle(defNames[nCol]);
    if (inx == nCol) {
        mTable.push_back(tCol);
    } else {
        mTable.push_back(mTable[nCol-1]);
	for (col_index j=nCol-1; j>inx; --j) mTable[j]=mTable[j-1];
	mTable[inx] = tCol;
    }
    for (Column::row_index j=0; j<mRow; j++) mTable[inx].insertRow(j, Cell());
}

//======================================   Copy rows from another table
void 
Table::insertRows(Column::row_index b4, const Table& tab) {
    //----------------------------------  Check column structure
    col_index nCol = size();
    for (col_index i=0; i<nCol; ++i) {
        if (!tab.exists(mTable[i].getTitle())) 
	    throw runtime_error("Invalid column structure");
    }

    //----------------------------------   Loop over Columns.
    for (col_index i=0; i<nCol; ++i) {
	mTable[i].insertRow(b4, tab[mTable[i].refTitle()], 0, tab.mRow);
    }
    mRow += tab.mRow;
}

//======================================  Move a row in a table
void 
Table::move_row(Column::row_index irow, Column::row_index jrow) {
    col_index N = size();
    for (col_index i=0; i<N; i++) mTable[i].move(irow, jrow);
}

//======================================  Read a table from an ascii file
void
Table::read(const string& in) {
    string::size_type l = in.rfind(".");
    string intype;
    if (l < in.size()) intype = in.substr(l);
    if (intype == ".gz" && l > 0) {
	string::size_type k = in.rfind(".", l-1);
	if (l < k) k = l;
	intype = in.substr(k);
    }
    if (intype == ".txt" || intype == ".txt.gz") {
        readtxt(in);
    } 
    else if (intype == ".h5") {
#if HAVE_HDF5
	tableIO_hdf inhdf;
	inhdf.set_debug(mDebug);
        inhdf.readTable(*this, in, mTitle);
#else
	throw runtime_error("HDF5 table I/O not implemented");
#endif
    }
    else if (intype == ".xml" || intype == ".xml.gz") {
        readxml(in);
    } else {
        readtxt(in);
    }
}

//======================================  Read a table from an ascii file
void
Table::readtxt(const string& in) {

    //----------------------------------  Get the input stream.
    istream* inf;
    if (in == "-") {
	inf = &cin;
    } else {
	inf = new ifstream(in.c_str());
    }
    if (!inf || !*inf) {
	throw runtime_error(string("Unable to open file: ")+in);
    }

    //----------------------------------  Open a line parser.
    ParseLine pl(*inf);
    pl.setDelim(", \t");
    while (pl.getLine() >= 0) {
        col_index nArgs = pl.getCount();
	if (!nArgs) continue;
	col_index nDim = mTable.size();

	//------------------------------  Loop over data cells in this row
	for (col_index i=0; i<nArgs; i++) {
	    if (nDim == i) {
	        insertColumn(nDim, "");
		nDim++;
	    }
	    mTable[i].addRow(Cell(pl[i]));
	}

	//------------------------------  Pad extra columns
	for (col_index i=nArgs; i<nDim; i++) mTable[i].addRow(Cell(""));
	mRow++;
	if (mDebug && (mRow%100 == 0)) cerr << "\rRow: " << mRow << flush;
    }
    if (inf != &cin) delete inf;
    if (mDebug) cerr << "\rRow: " << mRow << endl;
}

//======================================  Read a table from an ascii file
void
Table::readxml(const string& in) {
    const char* name = mTitle.c_str();
    if (name && !*name) name = 0;
    xsil::MetaIO meta(in.c_str(), name);
    if (!meta.is_open()) return;

    //----------------------------------  Initialize column names
    col_index nDim = mTable.size();
    if (!nDim) {
        try {
	    nDim = meta.getNColumn();
	    for (col_index i=0; i<nDim; ++i) {
	        const char* colname = meta.getColumnName(i);
		insertColumn(i, colname);
		mTable[i].setXType(meta.getTypeName(i));
	    }
	} catch (...) {
	    cerr << "Caught exception while reading column headers" << endl;
	    throw;
	}
    }
    mTitle = meta.getTableName();

    //----------------------------------  Loop over rows
    while (meta.getRow() == 1) {

	//------------------------------  Loop over data cells in this row
	for (col_index i=0; i<nDim; ++i) {
	    string cell = meta.getString(mTable[i].getTitle());
	    try {
	        mTable[i].addRow(Cell(cell));
	    } catch (...) {
	        cerr << "Exception in row " << mRow << " column: " << cell
		     << endl;
		throw;
	    }
	}
	mRow++;
    }
}

//======================================  Remove a specified column
void 
Table::removeColumn(col_index col) {
    mTable.erase(mTable.begin()+col);
}

//======================================  Remove unselected rows
void 
Table::selectRows(const std::vector<bool>& mask) {

    //----------------------------------  Select rows in each column
    col_index N=mTable.size();
    for (col_index i=0; i<N; ++i) {
        mTable[i].selectRows(mask);
    }

    //----------------------------------  Count the remaining rows
    N=mask.size();
    mRow = 0;
    for (col_index i=0; i<N; ++i) if (mask[i]) ++mRow;
}

//======================================  Set the debug printout level
void 
Table::setDebug(int i) {
    mDebug = i;
}

//======================================  Sort a table
void 
Table::sort(col_index col, bool decrease, sort_enum t) {
    if (col >= size()) throw  range_error("Column range error");

    if (t == kAuto) t = (mRow<1024) ? kBubble: kMerge;

    //----------------------------------  Numeric key - use index based sort.
    if (mTable[col].getType() == Cell::kNumeric) {
        const Column& keyCol(mTable[col]);
        std::vector<double> x(mRow);
        for (Column::row_index i=0; i<mRow; i++) x[i] = keyCol[i].getNumeric();
	switch (t) {
	case kBubble:
	    bsort(x, decrease);
	    break;
	case kMerge:
	default:
	    msort(x, decrease);
	}

    //----------------------------------  Use table sort.
    } else {
	switch (t) {
	case kBubble:
	    bsort(col, decrease);
	    break;
	case kMerge:
	default:
	    msort(col, decrease);
	}
    }

    //----------------------------------  Save the column ordering information
    col_index N=mTable.size();
    for (col_index i=0; i<N; ++i) {
        mTable[i].setOrdering(Column::kUnordered);
    }
    if (decrease) mTable[col].setOrdering(Column::kDecreasing);
    else          mTable[col].setOrdering(Column::kIncreasing);
}

//======================================  Sort algorithms
void 
Table::bsort(col_index col, bool decrease) {
    for (Column::row_index i=0; i<mRow; i++) {
        col_index imin = i;
	if (decrease) {
	    for (Column::row_index j=i+1; j<mRow; j++) {
	        if (mTable[col].less(imin, j)) imin = j;
	    }
	} else {
	    for (Column::row_index j=i+1; j<mRow; j++) {
	        if (mTable[col].less(j, imin)) imin = j;
	    }
	}
	if (i != imin) move_row(imin, i);
    }
}

void  
Table::bsort(vector<double> index, bool decrease) {
    for (Column::row_index i=0; i<mRow; i++) {
        col_index imin = i;
	if (decrease) {
	    for (Column::row_index j=i+1; j<mRow; j++) {
	        if (index[j] > index[imin]) imin = j;
	    }
	} else  {
	    for (Column::row_index j=i+1; j<mRow; j++) {
	        if (index[j] < index[imin]) imin = j;
	    }
	}
	if (i != imin) {
	    move_row(imin, i);
	    double t = index[imin];
	    for (Column::row_index j=imin; j>i; j--) index[j] = index[j-1];
	    index[i] = t;
	}
    }
}

//======================================  Merge sorts
//     Merge sorts are  O(N logN) sorts that work by first sorting all pairs
//     pairs entries and then iterating over pairs of sorted blocks.
//
void  
Table::msort(col_index col, bool decrease) {
    const Column& key(mTable[col]);

    //----------------------------------  Sort pairs of rows
    if (decrease) {
        for (Column::row_index i=0; i<mRow-1; i+=2) {
	    if (key.less(i, i+1)) swap_rows(i, i+1);
	}
    } else  {
        for (Column::row_index i=0; i<mRow-1; i+=2) {
	    if (key.less(i+1, i)) swap_rows(i, i+1);
	}
    }

    //----------------------------------  Loop over increasing group size
    std::vector<Column::row_index> inx(mRow);
    for (Column::row_index rank=2; rank<mRow; rank+=rank) {

        //------------------------------  Loop over groups
        for (Column::row_index ioff=0; ioff<mRow; ioff+=2*rank) {
	    Column::row_index imax = 2*rank < mRow-ioff ?  2*rank: mRow-ioff;
	    if (imax <= rank) continue;

	    for (Column::row_index i=0; i<rank; ++i) inx[i] = i;
	    Column::row_index jnx = rank;
	    Column::row_index ii  = 0;

	    if (decrease) {
	        //----------------------  Merge two blocks.
	        //  The inx[i] vector contains the current location of the
	        //  ith element for 0 <= i < rank, and the original location
	        //  of the ith element for rank <= i < imax. ii points to the 
	        //  original location of the next element in the first list and
	        //  jnx points to the next element to be merged from the second
	        //  list. With each pass through this loop, the ith element is 
	        //  exchanged with the greater (lesser) of the next elements 
	        //  from the two lists. If an element from the first list is 
	        //  kicked out from its position in the list, its current 
	        //  position and original location are recoded in inx.
	        for (Column::row_index i=0; i<jnx; i++) {
		    Column::row_index inc=inx[ii];
		    if (jnx<imax && key.less(ioff+inc, ioff+jnx)) {
		        swap_rows(ioff+i, ioff+jnx);
			inx[jnx]    = inx[i];
			inx[inx[i]] = jnx;
			jnx++;
		    } else {
		        if (inc != i) {
			    swap_rows(ioff+i, ioff+inc);
			    inx[inc] = inx[i];
			    inx[inx[i]] = inc;
			}
			++ii;
		    }
		}
	    } else  {
	        //----------------------  Merge two blocks (see above)
	        for (Column::row_index i=0; i<jnx; i++) {
		    Column::row_index inc=inx[ii];
		    if (jnx<imax && key.less(ioff+jnx, ioff+inc)) {
		        swap_rows(ioff+i, ioff+jnx);
			inx[jnx]    = inx[i];
			inx[inx[i]] = jnx;
			jnx++;
		    } else {
		        if (inc != i) {
			    swap_rows(ioff+i, ioff+inc);
			    inx[inc] = inx[i];
			    inx[inx[i]] = inc;
			}
			++ii;
		    }
		}
	    }
	} // loop over segment
    }  // loop over rank
}

void  
Table::msort(vector<double> x, bool decrease) {
    //----------------------------------  Sort pairs of rows
    if (decrease) {
        for (Column::row_index i=0; i<mRow-1; i+=2) {
	    if (x[i] < x[i+1]) {
	        swap_rows(i, i+1);
		double t = x[i];
		x[i] = x[i+1];
		x[i+1] = t;
	    }
	}
    } else  {
        for (Column::row_index i=0; i<mRow-1; i+=2) {
	    if (x[i] > x[i+1]) {
	        swap_rows(i, i+1);
		double t = x[i];
		x[i] = x[i+1];
		x[i+1] = t;
	    }
	}
    }

    //----------------------------------  Loop over increasing group size
    std::vector<Column::row_index> inx(mRow);
    for (Column::row_index rank=2; rank<mRow; rank+=rank) {

        //------------------------------  Loop over groups
        for (Column::row_index ioff=0; ioff<mRow; ioff+=2*rank) {
	    Column::row_index imax = (2*rank < mRow-ioff) ? 2*rank: mRow-ioff;
	    if (imax <= rank) continue;

	    for (Column::row_index i=0; i<rank; ++i) inx[i] = i;
	    Column::row_index jnx = rank;
	    Column::row_index ii  = 0;

	    if (decrease) {
	        //----------------------  Merge two blocks.
	        for (Column::row_index i=0; i<jnx; i++) {
		    Column::row_index inc=inx[ii];
		    if (jnx<imax && x[ioff+jnx] > x[ioff+inc]) {
		        swap_rows(ioff+i, ioff+jnx);
			double t    = x[ioff+i];
			x[ioff+i]   = x[ioff+jnx];
			x[ioff+jnx] = t;
			inx[jnx]    = inx[i];
			inx[inx[i]] = jnx;
			jnx++;
		    } else {
		        if (inc != i) {
			    swap_rows(ioff+i, ioff+inc);
			    double t = x[ioff+i];
			    x[ioff+i] = x[ioff+inc];
			    x[ioff+inc] = t;
			    inx[inc] = inx[i];
			    inx[inx[i]] = inc;
			}
			++ii;
		    }
		}
		if (mDebug) {
		    Column::row_index ierr=0;
		    for (Column::row_index i=1; i<imax; ++i)
		        if (x[ioff+i-1] < x[ioff+i]) ierr=i; 
		    if (ierr) {
		        cerr << "msort error @" << ioff+ierr
			     << " rank: " <<rank << endl;
			for (Column::row_index i=0; i<2*rank; ++i) 
			    cerr << long(x[ioff+i]) << endl;
		    }
		}
	    } else  {
	        //----------------------  Merge two blocks.
	        for (Column::row_index i=0; i<jnx; i++) {
		    Column::row_index inc=inx[ii];
		    if (jnx<imax && x[ioff+jnx] < x[ioff+inc]) {
		        swap_rows(ioff+i, ioff+jnx);
			double t = x[ioff+i];
			x[ioff+i] = x[ioff+jnx];
			x[ioff+jnx] = t;
			inx[jnx]    = inx[i];
			inx[inx[i]] = jnx;
			jnx++;
		    } else {
		        if (inc != i) {
			    swap_rows(ioff+i, ioff+inc);
			    double t = x[ioff+i];
			    x[ioff+i] = x[ioff+inc];
			    x[ioff+inc] = t;
			    inx[inc]    = inx[i];
			    inx[inx[i]] = inc;
			}
			++ii;
		    }
		}
		if (mDebug) {
		    Column::row_index ierr=0;
		    for (Column::row_index i=1; i<imax; ++i)
		        if (x[ioff+i] < x[ioff+i-1]) ierr=i; 
		    if (ierr) {
		        cerr << "msort error @" << ioff+ierr
			     << " rank: " <<rank << endl;
			for (Column::row_index i=0; i<2*rank; ++i) 
			    cerr << long(x[ioff+i]) << endl;
		    }
		}
	    }
	} // loop over segment
    }  // loop over rank
}


//======================================  Print table status
void 
Table::status(void) const {
    col_index N = size();
    for (col_index i=0; i<N; i++) mTable[i].status();
}

//======================================  Swap rows in a table
void 
Table::swap_rows(Column::row_index irow, Column::row_index jrow) {
    col_index N = size();
    for (col_index i=0; i<N; i++) mTable[i].swap(irow, jrow);
}

//======================================  Write a table
void 
Table::write(const string& f, int flags, const string& t) const {
    string ftype(t);
    if (t.empty()) {
       string::size_type      l     = f.rfind(".");
       if (l != string::npos) ftype = f.substr(l+1);
    }
    if (f == "-") {
        if      (ftype == "html") writehtml(cout);
	else if (ftype == "xml")  writexml(cout);
	else                      writetxt(cout, flags);
    }  else {
	//------------------------------  Write an hdf5 table
	if (ftype == "h5") {
#ifdef HAVE_HDF5
	    tableIO_hdf wr;
	    wr.writeTable(*this, f, mTitle);
#else
	    throw runtime_error("HDF5 table I/O not implemented");
#endif
	}

	//------------------------------  Write an html file
        else if (ftype == "html") {
	    ofstream out(f.c_str());
	    writehtml(out);
	}

	//------------------------------  Write an xml file
	else if (ftype == "xml") {
	    ofstream out(f.c_str());
	    writexml(out);
	}

	//------------------------------  Write a text table
	else {
	    ofstream out(f.c_str());
	    writetxt(out, flags);
	}
    }
}

//======================================  Write a table
ostream& 
Table::writehtml(ostream& out) const {
    html::document doc(mTitle.c_str());
    html::table& t=dynamic_cast<html::table&>(doc.addObject(html::table(mTitle)));
    col_index nCol = size();
    for (col_index i=0 ; i<nCol; i++) {
        t.addColumn(mTable[i].getTitle());
    }
    for (Column::row_index i=0; i<mRow; i++) {
        t.addRow();
	for (col_index j=0 ; j<nCol; ++j) {
	    t.insertData(i, j, mTable[j].refCell(i).getString());
	}
    }
    html::writer w(out);
    doc.write(w);
    return out;
}

//======================================  Write a text file
ostream& 
Table::writetxt(ostream& out, int flags) const {
    col_index nCol = mTable.size();

    if ((flags&1) != 0) {
        out << "# ";
	for (col_index i=0; i<nCol; i++) {
	    if (i) out << "  ";
	    out << mTable[i].getTitle();
	}
	out << endl;
    }

    if ((flags&2) != 0) {
        for (Column::row_index i=0; i<mRow; i++) {
	    for (col_index j=0; j<nCol; j++) {
	        if (j) out << "  ";
		out << mTable[j].refCell(i).getString();
	    }
	    out << endl;
	}
    }

    if ((flags&4) != 0) {
	for (col_index i=0; i<nCol; i++) {
	   if (i) out << "  ";
	   if (mTable[i].getType() == Cell::kNumeric) out << mTable[i].Sum();
	   else out << "--";
	}
	out << endl;
    }
    return out;
}

//======================================  Write a table
ostream& 
Table::writexml(ostream& out) const {
    const char* ttl=mTitle.c_str();
    xsil::ligolw doc("ligo:ldas:file");
    xsil::table& t=dynamic_cast<xsil::table&>(*doc.addObject(xsil::table(ttl)));
    t.refStream().delimit(',');
    t.refStream().setName(t.getName());
    col_index nCol = size();
    for (col_index i=0 ; i<nCol; i++) {
        t.addColumn(mTable[i].getTitle(), mTable[i].getXType());
    }
    for (Column::row_index i=0; i<mRow; i++) {
	for (col_index j=0 ; j<nCol; ++j) {
	    string typ(mTable[j].getXType());
	    if (typ == "real_4" || typ == "real_8") {
	        t.refStream().Add(mTable[j].refCell(i).getNumeric());
	    } else if (typ == "int_2" || typ == "int_4s") {
	        t.refStream().Add(int(mTable[j].refCell(i).getNumeric()));
	    } else if (typ == "ilwd:char_u") {
		string charus = mTable[j].refCell(i).getString();
		int len = charus.size();
		xsil::UCVec uc(len);
		memcpy(uc.pData, charus.c_str(), len);
	        t.refStream().Add(uc);		
	    } else {
	        t.refStream().Add(mTable[j].refCell(i).getString());
	    }
	}
	t.refStream().lineBreak();
    }
    xsil::Xwriter w(out);
    doc.Spew(w);
    return out;
}
