/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef TABLE_COLUMN_HH
#define TABLE_COLUMN_HH

#include "cell.hh"
#include "format_spec.hh"
#include <vector>

//======================================  Column class definition
namespace table {

    /**  tablepgm table Column class.
      *  \author John G. Zweizig
      *  \brief Column class.
      *  \version $Id: column.hh 6804 2013-04-27 09:19:21Z john.zweizig@LIGO.ORG $
      */
    class Column {
    public:
	/** Row index data type.
	  */
	typedef unsigned int row_index;

	/**  Column order enumeration.
	  */
	enum col_order {
	    kUnordered,   ///< Data are unordered
	    kIncreasing,  ///< Data are in increasing order
	    kDecreasing,  ///< Data are in decreasing order
	    kUndefined    ///< Data ordering is unknown
	};
    public:
	/**  Construct an empty column with the specified title.
	  *  @memo construct a column.
	  *  @param title Column title
	  */
	Column(const std::string& title);

	/**  Add a row to the end of a column and fll the new cell with the 
	  *  specified value.
	  *  @memo add a row
	  *  @param c Cell initialization value.
	  */
	void addRow(const Cell& c);

	/**  Get the column print format specifier.
	  *  @memo Get the format.
	  *  @return Format reference.
	  */
	const format_spec& format(void) const;

	/**  Get the column ordering state.
	  *  @memo Get the enumerated row ordering for this column.
	  *  @return Enumerated column ordering.
	  */
	col_order getOrdering(void) const;

	/**  Get a character pointer to the title string.
	  *  @memo Get the Title.
	  *  @return Constant pointer to the title string.
	  */
	const char* getTitle(void) const;

	/**  Get the type of the column data.
	  *  @memo Get the data type
	  *  @return Data type of cells in this column.
	  */
	Cell::cell_type getType(void) const;

	/**  Get the maximum width of the column strings.
	  *  @memo Get width.
	  *  @return Maximum string width of all cells in this column.
	  */
	unsigned int getWidth(void) const;

	/**  Get the external data type
	  *  @memo External data type.
	  *  @return External data type string.
	  */
	const char* getXType(void) const;

	/**  Insert a new row at the specified position and initialize the 
	  *  new cell with the specified cell value.
	  *  @memo Insert a row.
	  *  @param b4 Row number before which data are inserted.
	  *  @param c  New cell content.
	  */
	void insertRow(row_index b4, const Cell& c);

	/**  Insert multiple rows before the specified row. The inserted data 
	  *  are initialized with the contents of the \c n rows of column \c c
	  *  starting with row \c s.
	  *  @memo Insert multiple rows.
	  *  @param b4 Row number before which data are inserted.
	  *  @param c  Reference to column containing data.
	  *  @param s  Number of first row to be copied.
	  *  @param n  Number of rows to be copied.
	  */
	void insertRow(row_index b4, const Column& c, row_index s, row_index n);

	/**  Test the relative magnitude of the string values of two rows.
	  *  @memo Compare cells of two rows
	  *  @param i One row number.
	  *  @param j The other row number.
	  *  @return True if ith row contents is less than that of the jth row.
	  */
	bool less(row_index i, row_index j) const;

	/**  Test the relative magnitude of the numeric values of two rows.
	  *  @memo Compare cells of two rows
	  *  @param i One row number.
	  *  @param j The other row number.
	  *  @return True if ith row contents is less than that of the jth row.
	  */
	bool lesn(row_index i, row_index j) const;

	/**  Move a row from one index to another, leaving all other rows in the
	 *  same position relative to one another. The 'from' row is takes
	 *  the place of the 'to' row and all intermediate rows are shifted 
	 *  one position to fill the vacated 'from' position.
	 *  @memo Move a row.
	 *  @param from index of the row to be moved.
	 *  @param to   index of the row to be replaced by the from row.
	 */
	void move(row_index from, row_index to);

	/**  Return a constant reference to the specified cell.
	  *  @memo Reference Cell
	  *  @param i Row number of cell to be referenced.
	  *  @return Constant reference to the specified cell.
	  */
	const Cell& operator[](row_index i) const;

	/**  Return a reference to the specified cell.
	  *  @memo Reference Cell
	  *  @param i Row number of cell to be referenced.
	  *  @return Reference to the specified cell.
	  */
	Cell& operator[](row_index i);
    
	/**  Return a constant reference to the specified cell.
	  *  @memo Reference Cell
	  *  @param i Row number of cell to be referenced.
	  *  @return Constant reference to the specified cell.
	  */
	const Cell& refCell(row_index i) const;

	/**  Return a reference to the specified cell.
	  *  @memo Reference Cell
	  *  @param i Row number of cell to be referenced.
	  *  @return Reference to the specified cell.
	  */
	Cell& refCell(row_index i);

	/**  Reserve space for the specified number of rows.
	  *  \brief Reserve space.
	  *  \param N Number of rows to be reserved.
	  */
	void reserve(row_index N);

	/**  Return a constant reference to the column title string.
	  *  @memo Get title string.
	  *  @return Constant reference to column title string.
	  */
	const std::string& refTitle(void) const;

	/**  Return a constant reference to the external data type string.
	  *  \brief Reference external data type string.
	  *  \return Constant reference to column external data type string.
	  */
	const std::string& refXType(void) const;

	/**  Select the specified rows. All rows that are not selected are 
	  *  deleted.
	  *  @memo Select rows
	  *  @param mask Boolean vector indicating which rows to retain.
	  */
	void selectRows(const std::vector<bool>& mask);

	/**  Set the column format string.
	  *  @memo Set the column format.
	  *  @param fmt Column format string.
	  */
	void setFormat(const std::string& fmt);    

	/**  Set the column justification string.
	  *  @memo Set the column justification.
	  *  @param jfy Column justification string.
	  */
	void setJustify(const std::string& jfy);    

	/**  Set the column ordering state.
	  *  @memo Set the column ordering.
	  *  @param co Column ordering state.
	 */
	void setOrdering(col_order co);

	/**  Set the column title.
	  *  @memo Set the column title.
	  *  @param title Set the column title.
	 */
	void setTitle(const std::string& title);    

	/**  Set the External data type. The external data type is used to 
	  *  specify which data type to use when writing to an xml file.
	  *  @memo Set the external type.
	  *  @param xtyp External data type string.
	 */
	void setXType(const std::string& xtyp);    

	/**  Sum the numeric contents of the column. \c Sum() will throw an 
	  *  exception if one or more cells are non-numeric.
	  *  @memo sum the column data
	  *  @exception std::runtime_error is thrown if data are not numeric.
	  *  @return Numeric summ of all cell data.
	  */
	double Sum(void) const throw(std::runtime_error);

	/**  Get the number of rows in this column.
	  *  @memo Number of rows
	  *  @return Number of rows in column.
	  */
	row_index size(void) const;

	/**  Print column status.
	  *  @memo Print column status
	  */
	void status(void) const;

	/**  Swap the contents of the specified rows.
	  *  @memo Swap contents of two rows.
	  *  @param i One row
	  *  @param j The other row
	  */
	void swap(row_index i, row_index j);

	/**  Test that the indicated row is in the correct order.
	  *  @memo Test ordering
	  *  @param i Row to test.
	  */
	void testOrder(row_index i);

	/**  Compare column title to string.
	  *  @memo Compare title
	  *  @param ttl Strin to compare to title
	  *  @return true if argument string is equal to the title string.
	  */
	bool titleq(const std::string& ttl) const;

    private:
	std::string mTitle;
	std::string mXType;
	col_order   mOrdering;
	format_spec mFormat;
	std::vector<Cell> mRows;
    };

    //=====================================  Inline methods
    inline void
    Column::addRow(const Cell& c) {
	mRows.push_back(c);
    }

    inline const format_spec& 
    Column::format(void) const {
	return mFormat;
    }

    inline Column::col_order
    Column::getOrdering(void) const {
	return mOrdering;
    }

    inline const char*
    Column::getTitle(void) const {
	return mTitle.c_str();
    }

    inline const Cell& 
    Column::operator[](row_index i) const {
	return mRows[i];
    }

    inline Cell& 
    Column::operator[](row_index i) {
	return mRows[i];
    }

    inline const Cell& 
    Column::refCell(row_index i) const {
	return mRows[i];
    }
  
    inline Cell& 
    Column::refCell(row_index i) {
	return mRows[i];
    }

    inline const std::string& 
    Column::refXType(void) const {
	return mXType;
    }

    inline const char* 
    Column::getXType(void) const {
	return mXType.c_str();
    }

    inline bool
    Column::less(row_index i, row_index j) const {
	return mRows[i] < mRows[j];
    }

    inline bool
    Column::lesn(row_index i, row_index j) const {
	return mRows[i] < mRows[j];
    }

    inline const std::string&
    Column::refTitle(void) const {
	return mTitle;
    }

    inline Column::row_index
    Column::size(void) const {
	return mRows.size();
    }

    inline void
    Column::swap(row_index i, row_index j) {
	Cell t=mRows[i];
	mRows[i] = mRows[j];
	mRows[j] = t;
    }

}

#endif // !defined(TABLE_COLUMN_HH)
