/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "ParseLine.hh"
#include "SegList.hh"
#include "seg_iobase.hh"
#include "seg_iotext.hh"
#include "seg_ioxsil.hh"
#include <sys/times.h>
#include <string>
#include <fstream>
#include <sstream>
#include <iostream>
#include <stdexcept>
#include <map>
#include <cstdlib>
#include <unistd.h>

#include "calc_engine/engine.hh"
#include "seg_stack.hh"

#ifdef HAVE_HDF5
#include "seg_iohdf5.hh"
#endif

class SegEval : public Engine {
public:
    typedef std::map<segID, LockSegList> dict_type;
public:
    SegEval(dict_type& t, const std::string& cmd, int debug=0);
    ~SegEval(void);
    const LockSegList& eval(void);
    virtual void eval_funct(const std::string& sym, int N);
    virtual void push_symbol(const std::string& sym);
    virtual void push_literal(const std::string& sym);

private:
    dict_type& mDictionary;
    seg_stack  mStack;
};

class seg_calc {
public:
    seg_calc(int argc, const char* argv[]);
    void doit(void);
    void exec_cmd(const ParseLine& pl);
    void exec_stream(std::istream& file);
    void read_segfile(const std::string& file, const segID& select,
		      const std::string& version="");
    void sort(void);
    void write_segfile(const std::string& file, const segID& select,
		       const std::string& version="");
    void usertime(void) const;

public:
    typedef std::map<segID, LockSegList> seg_map;
    typedef seg_map::iterator seg_iter;
    typedef seg_map::const_iterator const_seg_iter;

private:
    bool exists(const segID&) const;
    bool exists(const std::string&) const;
    seg_iter       find(const std::string&);
    const_seg_iter find(const std::string&) const;
    seg_iter       find_nothrow(const std::string&);
    const_seg_iter find_nothrow(const std::string&) const;

private:
    std::string mCmdFile;
    std::string mInitCmd;
    std::string mOutFile;
    seg_map mSegMap;
    int mDebug;
    bool mVerbose;
    bool mSorted;
    mutable double tDelta;
};

using namespace std;

//======================================  Main function
int main(int argc, const char* argv[]) {
    try {
	seg_calc x(argc, argv);
	x.doit();
    } catch (exception& e) {
	cerr << "Caught exception: " << e.what() << endl;
    }
}

//======================================  Utility methods
inline seg_calc::seg_iter
seg_calc::find_nothrow(const string& name) {
    return mSegMap.find(segID(name));
}

inline seg_calc::const_seg_iter
seg_calc::find_nothrow(const string& name) const {
    return mSegMap.find(segID(name));
}

inline bool
seg_calc::exists(const string& name) const {
    return find_nothrow(name) != mSegMap.end();
}

inline bool
seg_calc::exists(const segID& id) const {
    return mSegMap.find(id) != mSegMap.end();
}

inline seg_calc::seg_iter
seg_calc::find(const string& name) {
    seg_iter p = find_nothrow(name);
    if (p == mSegMap.end()) 
	throw runtime_error(string("Unknown segment list: ") + name);
    return p;
}

inline seg_calc::const_seg_iter
seg_calc::find(const string& name) const {
    const_seg_iter p = find_nothrow(name);
    if (p == mSegMap.end()) 
	throw runtime_error(string("Unknown segment list: ") + name);
    return p;
}

void
seg_calc::usertime(void) const {
    struct tms t;
    times(&t);
    double tNow(t.tms_utime/double(sysconf(_SC_CLK_TCK)));
    cerr<< "user time: " << tNow-tDelta <<  " (total: " << tNow << ")" << endl;
    tDelta = tNow;
}

//======================================  Constructor
seg_calc::seg_calc(int argc, const char* argv[])
    : mDebug(0), mVerbose(false), mSorted(false), tDelta(0)
{
    string fmt, pfx;

    //----------------------------------  Parse command string.
    int nread = 0;
    for (int i=1; i<argc; ++i) {
	string argi = argv[i];
	if (argi == "-c") {
	    if (mInitCmd.empty()) {
		mInitCmd = argv[++i];
	    } else {
		mInitCmd += "; ";
		mInitCmd += argv[++i];
	    }
	} else if (argi == "-f") {
	    fmt = argv[++i];
	} else if (argi == "-i") {
	    mCmdFile = argv[++i];
	} else if (argi.substr(0,2) == "-s") {
	    string name;
	    if (argi.size() > 2) {
		if (argi[2] == ':') name = argi.substr(3);
		else throw runtime_error("Segment name syntax error");
	    }
	    segID select(name, pfx);
	    read_segfile(argv[++i], select, fmt);
	} else if (argi == "-v" || argi == "--verbose") {
	    mVerbose = true;
	} else if (argi == "--help") {
	    cout << "seg_calc command syntax: " << endl;
	    cout << "   seg_calc [-c <command-string>] [-f <text-file-format>]"
		 << endl;
	    cout << "            [-i <command-file>] [-s[:<name>] <seg-file>]"
		 << endl;
	    cout << "            <seg-file-1> [... <seg-file-N>]" << endl;
	    throw runtime_error("Syntax help");
	} else if (argi == "--debug") {
	    if (i+1 >= argc || *(argv[i+1]) == '-') mDebug = 1;
	    else                   mDebug = strtol(argv[++i], 0, 0);
	} else if (argi == "-" || argi[0] != '-') {
	    if (mVerbose) cerr << "read file: " << argi << "\r";
	    segID select("", pfx, segID::kAny);
	    read_segfile(argi, select, fmt);
	    nread++;
	} else {
	    throw runtime_error(string("Undefined keyword: ") + argv[i]);
	}
    }
    if (mVerbose && nread) cerr << endl;
    sort();
}

//======================================  Execute command stream
void
seg_calc::doit(void) {

    //----------------------------------  Execute a list of commands
    if (!mInitCmd.empty()) {
	istringstream cmdin(mInitCmd);
	exec_stream(cmdin);
    }

    //----------------------------------  Execute a list of commands
    if (!mCmdFile.empty()) {
	if (mCmdFile == "-") {
	    exec_stream(cin);
	} else {
	    ifstream in(mCmdFile.c_str());
	    if (!in.is_open()) 
		throw runtime_error(string("Can't open file: ") + mCmdFile);
	    exec_stream(in);
	}
    }

}

//======================================  Command execution
void 
seg_calc::exec_stream(std::istream& file) {
    ParseLine pl(file);
    if (!pl.isOpen()) throw runtime_error("Can't open file.");
    if (mVerbose) pl.setLog(cerr);
    while (pl.getLine() >= 0) {
	if (pl.getCount() > 0) {
	    try {
		exec_cmd(pl);
	    } catch (exception& e) {
		cerr << "Caught exception: '" << e.what() << "' on line: "
		     << pl.getLineNumber() << endl;
	    }
	    if (mVerbose) usertime();
	}
    }
}

//======================================  Command execution
void 
seg_calc::exec_cmd(const ParseLine& pl) {
    int nArg = pl.getCount();
    string cmdi = pl[0];

    //------------------------------  Define a segment list
    if (cmdi == "define") {
	string eval_cmd, name, ifo("xx");
	bool replace=false, logit=false, quiet=false;
	int i0=1;
	if (*(pl[i0]) != '-' ) name = pl[i0++];
	for (int i=i0; i<nArg; ++i) {
	    string argi=pl[i];
	    if (argi == "-eval") {
		eval_cmd = pl[++i];
	    } else if (argi == "-ifo") {
		ifo = pl[++i];
	    } else if (argi == "-log") {
		logit = true;
	    } else if (argi == "-name") {
		name = pl[++i];
	    } else if (argi == "-quiet") {
		quiet = true;
		if (i+1 < nArg && *(pl[i+1]) != '-') name = pl[++i];
	    } else if (argi == "-replace") {
		if (i+1 < nArg && *(pl[i+1]) != '-') name = pl[++i];
		replace = true;
	    } else {
		throw runtime_error(string("Undefined argument: ")+argi);
	    }
	}
	LockSegList r(ifo.c_str());
	if (name.empty()) throw runtime_error("Segment name not specified");
	segID mySeg(name);
	if (mySeg.ifo().empty()) mySeg.set_ifo(ifo);
	if (mySeg.version() < 0) mySeg.set_version(0);

	if (!replace && exists(mySeg)) {
	    if (quiet) return;
	    throw runtime_error(string("Segment exists: ")+mySeg.full_name());
	} else {
	    if (!eval_cmd.empty()) {
		SegEval ev(mSegMap, eval_cmd, mDebug);
		r = ev.eval();
	    }
	    seg_iter p = mSegMap.find(mySeg);
	    if (p != mSegMap.end()) p->second = r;
	    else          mSegMap.insert(seg_map::value_type(mySeg, r));
	}
	if (logit) cout << "# " << mySeg.full_name() << " defined as " 
			<< eval_cmd << endl;
	if (mVerbose) {
	    cerr << "SegList: " << mySeg.full_name() << " defined with "
		 << r.size() << " segments, " << r.live() << "s live." << endl;
	}

    //----------------------------------  Echo a line
    } else if (cmdi == "echo") {
	cout << pl[1] << endl;

    //----------------------------------  Execute a command file.
    } else if (cmdi == "exec") {
	if (nArg < 2) 
	    throw runtime_error("exec must have a filename argument.");
	string wd;
	for (int i=1; i<nArg-1; ++i) {
	    string argi=pl[i];
	    if (argi == "-cwd") {
		wd = pl[++i];
	    } else {
		throw runtime_error(string("Undefined argument: ") + argi);
	    }
	}

	string macro = pl[nArg-1];
	if (macro[0] != '/' && !wd.empty()) macro = wd + "/" + macro;
	ifstream in(macro.c_str());
	if (!in.is_open()) throw runtime_error(string("Can't open: ")+macro);
	if (wd.empty()) {
	    exec_stream(in);
	} else {
	    char savewd[1024];
	    if (!getcwd(savewd, sizeof(savewd))) {
		throw runtime_error("Can't retrieve current directory");
	    }
	    try {
		if (!chdir(wd.c_str())) {
		    throw runtime_error("Can't retrieve current directory");
		}
		exec_stream(in);
		if (!chdir(savewd)) {
		    cerr << "Can't return to current directory" << endl;
		    exit(1);
		}
	    } catch (...) {
		if (!chdir(savewd)) {
		    cerr << "Can't return to current directory" << endl;
		    exit(1);
		}
		throw;
	    }
	}

    //----------------------------------  Print a segment index
    } else if (cmdi == "index") {
	string name;
	for (int i=1; i<nArg; ++i) {
	    string argi=pl[i];
	    if (argi == "-name") {
		name = pl[++i];
	    } else {
		throw runtime_error(string("Undefined argument: ") + argi);
	    }
	}

	cout << "Name  size  live" << endl;
	for (const_seg_iter i=mSegMap.begin(); i != mSegMap.end(); ++i) {
	    if (name.empty() || name == i->first.full_name()) {
		LockSegList::size_type N=i->second.size();
		cout << i->first.full_name() << "  " << N << "  " 
		     << long(double(i->second.live())+0.5);
		if (!N) {
		    cout << "       --         --     ";
		} else {
		    cout << "  " << i->second[0].getStartTime().getS() 
			 << "  " << i->second[N-1].getEndTime().getS();
		}
		cout << endl;
	    }
	}
	
    //----------------------------------  Read in a segment list
    } else if (cmdi == "read") {
	string name, sfile, fmt, prefix;
	int version = segID::kUnknown;
	for (int i=1; i<nArg; ++i) {
	    string argi=pl[i];
	    if (argi == "-file") {
		sfile = pl[++i];
	    } else if (argi == "-format") {
		fmt = pl[++i];
	    } else if (argi == "-name") {
		name = pl[++i];
	    } else if (argi == "-prefix") {
		prefix = pl[++i];
	    } else if (argi == "-version") {
		version = pl.getInt(++i);
	    } else {
		throw runtime_error(string("Undefined argument: ") + argi);
	    }
	}
	if (sfile.empty()) throw runtime_error("Segment file not specified");

	segID select(name);
	if (!prefix.empty())            select.set_ifo(prefix);
	if (version != segID::kUnknown) select.set_version(version);
	read_segfile(sfile, select, fmt);

    //----------------------------------  Read in a segment list
    } else if (cmdi == "remove") {
	string name;
	for (int i=1; i<nArg; ++i) {
	    string argi=pl[i];
	    if (argi == "-name") {
		name = pl[++i];
	    } else {
		throw runtime_error(string("Undefined argument: ") + argi);
	    }
	}
	if (name.empty()) throw runtime_error("No segment list specified.");
	mSegMap.erase(find(name));
 
    //----------------------------------  Write a segment list
    } else if (cmdi == "write") {
	string name, sfile;
	string format("%i %s %d");
	for (int i=1; i<nArg; ++i) {
	    string argi=pl[i];
	    if (argi == "-name") {
		name = pl[++i];
	    } else if (argi == "-file") {
		sfile = pl[++i];
	    } else if (argi == "-format") {
		format = pl[++i];
	    } else {
		throw runtime_error(string("Undefined argument: ") + argi);
	    }
	}
	segID select(name);
	if (select.version() == segID::kUnknown) {
	    select.set_version(segID::kAny);
	}
	write_segfile(sfile, select, format);

    //------------------------------  Unknown command
    } else {
	throw runtime_error(string("Undefined command:") + cmdi);
    }
}

//======================================  Test extension
inline bool
test_ext(const std::string& file, const std::string& ext) {
    if (ext.size() > file.size()) return false;
    return file.find(ext) == file.size() - ext.size();
}

//======================================  Read a segment file
void 
seg_calc::read_segfile(const string& file, const segID& select, 
		       const string& fmt) {
    seg_iobase* rdr=0;

    //----------------------------------  Decipher the file type
    if (test_ext(file, ".xml")) {
	rdr = new seg_ioxsil();
    }
    else if (test_ext(file, ".h5")) {
#ifdef HAVE_HDF5
	rdr = new seg_iohdf5();
#else
	cerr << "*** Error: HDF5 not installed. File: " << file 
	     << " was ignored." << endl;
	return;
#endif // defined(HAVE_HDF5)
    }
    else {
	rdr = new seg_iotext();
    }
    rdr->set_debug(mDebug);

    //----------------------------------  Read in the requested segments.
    try {
	rdr->read_seg(file, select, fmt);
	rdr->merge(mSegMap, mSorted);
    }

    //----------------------------------  Failure!
    catch (std::exception& e) {
	cerr << "Exception in read from file " << file << ": " << e.what() 
	     << endl;
	delete rdr;
	throw;
    }

    //----------------------------------  Done. return
    delete rdr;
}

//======================================  Sort the segment list.
void 
seg_calc::sort(void) {
    for (seg_iter i=mSegMap.begin(); i != mSegMap.end(); ++i) {
	i->second.sort();
	i->second.coalesce();
    }
    mSorted = true;
}

//======================================  Sort the segment list.
void 
seg_calc::write_segfile(const string& file, const segID& select,
			const string& fmt) {
    seg_iobase* wrt=0;

    //----------------------------------  Decipher the file type
    if (test_ext(file, ".xml")) {
	wrt = new seg_ioxsil();
    }
    else if (test_ext(file, ".h5")) {
#ifdef HAVE_HDF5
	wrt = new seg_iohdf5();
#else
	cerr << "*** Error: HDF5 not instsalled. File: " << file 
	     << " not written." << endl;
	return;
#endif // defined(HAVE_HDF5)
    }
    else {
	wrt = new seg_iotext();
    }

    //----------------------------------  Read in the requested segments.
    try {
	wrt->write_seg(mSegMap, select, fmt, file);
    }

    //----------------------------------  Failure!
    catch (std::exception& e) {
	cerr << "Exception in write to file " << file << ": " << e.what() 
	     << endl;
	delete wrt;
	throw;
    }

    //----------------------------------  Done. return
    delete wrt;
}

//======================================  SegEval command constructor
SegEval::SegEval(dict_type& d, const std::string& cmd, int debug)
  : Engine(mStack), mDictionary(d)
{
    setDebug(debug);
    istringstream str(cmd);
    Parse(str);
}

//======================================  SegEval destructor
SegEval::~SegEval(void) {
}

//======================================  Evaluate a segment
const LockSegList& 
SegEval::eval(void) {
    crank();
    if (mStack.size() != 1) throw runtime_error("Invalid command string");
    return dynamic_cast<seg_token&>(mStack.top()).refSegment();
}

//======================================  Evaluate a function
void 
SegEval::eval_funct(const std::string& sym, int N) {
    if (sym == "not") {
        dynamic_cast<seg_token&>(mStack.top()).refSegment().invert();
    } else if (sym == "now") {
        mStack.push(double(Now().getS()));
    } else if (sym == "pad") {
	if (!mStack.numeric(1) || !mStack.numeric(2)) {
	    throw runtime_error("Pad sizes must be numeric");
	} else if (mStack.numeric(3)) {
	    throw runtime_error("Padded object must be segment list");
	} else {
	    int pEnd = int(mStack.pop_numeric());
	    int pBeg = int(mStack.pop_numeric());
	    mStack.top().refSegment().pad(pBeg, pEnd);
	}
    } else if (sym == "range") {
	if (mStack.numeric(1)) 
	    throw runtime_error("Operation not defined for numeric arguments");
	seg_token* t = dynamic_cast<seg_token*>(mStack.pop_token());
	LockSegList l("Xn");
	long nSegs=t->refSegment().size();
	if (nSegs != 0) {
	    l.insert(LockSegment(0, t->refSegment()[0].getStartTime(), 
				 t->refSegment()[nSegs-1].getEndTime()));
	}
	delete t;
	mStack.push(new seg_token(l));
    } else if (sym == "seg") {
        double end   = mStack.pop_numeric();
	double start = mStack.pop_numeric();
	LockSegList l("Xn");
	l.insert(LockSegment(0, Time(long(start)), end-start));
	mStack.push(new seg_token(l));
    } else if (sym == "xor") {
	if (mStack.numeric(1) || mStack.numeric(2)) {
	    throw runtime_error("Operation not defined for numeric arguments");
	} else {
	    seg_token* t = dynamic_cast<seg_token*>(mStack.pop_token());
	    dynamic_cast<seg_token&>(mStack.top()).refSegment() ^= 
		t->refSegment();
	    delete t;
	}
    } else {
        throw runtime_error(string("Undefined function:") + sym);
    }
}

//======================================  
void 
SegEval::push_symbol(const std::string& sym) {
    segID symID(sym);
    if (mDictionary.find(symID) == mDictionary.end()) {
        throw runtime_error(string("Undefined segment: ") + symID.full_name());
    }
    mStack.push(new seg_token(mDictionary[symID]));
}

//======================================  
void 
SegEval::push_literal(const std::string& sym) {
    push_symbol(sym);
}
