#ifndef _LIGO_EVENTITERATOR_H
#define _LIGO_EVENTITERATOR_H
/*----------------------------------------------------------------------*/
/*                                                         		*/
/* Module Name: Iterator						*/
/*                                                         		*/
/* Module Description: Iterator for traversing an event set		*/
/*                                                         		*/
/* Revision History:					   		*/
/* Rel   Date     Programmer  	Comments				*/
/* 1.0	 25Jun01  D. Sigg    	First release		   		*/
/*                                                         		*/
/* Documentation References:						*/
/*	Man Pages: Iterator.html					*/
/*	References: none						*/
/*                                                         		*/
/* Author Information:							*/
/* Name          Telephone       Fax             e-mail 		*/
/* Daniel Sigg   (509) 372-8132  (509) 372-8137  sigg_d@ligo.mit.edu	*/
/*                                                         		*/
/*                                                         		*/
/*                      -------------------                             */
/*                                                         		*/
/*                             LIGO					*/
/*                                                         		*/
/*        THE LASER INTERFEROMETER GRAVITATIONAL WAVE OBSERVATORY.	*/
/*                                                         		*/
/*                     (C) The LIGO Project, 1999.			*/
/*                                                         		*/
/*                                                         		*/
/* Caltech				MIT		   		*/
/* LIGO Project MS 51-33		LIGO Project NW-17 161		*/
/* Pasadena CA 91125			Cambridge MA 01239 		*/
/*                                                         		*/
/* LIGO Hanford Observatory		LIGO Livingston Observatory	*/
/* P.O. Box 1970 S9-02			19100 LIGO Lane Rd.		*/
/* Richland WA 99352			Livingston, LA 70754		*/
/*                                                         		*/
/*----------------------------------------------------------------------*/

#include "PConfig.h"
#include <iterator>
#include "events/IteratorImp.hh"

#ifdef __GNU_STDC_OLD
namespace std {
template <class _Category, class _Tp, class _Distance = ptrdiff_t,
          class _Pointer = _Tp*, class _Reference = _Tp&>
struct iterator {
  typedef _Category  iterator_category;
  typedef _Tp        value_type;
  typedef _Distance  difference_type;
  typedef _Pointer   pointer;
  typedef _Reference reference;
};
}
#endif

namespace events {

   class Event;
   class Set;
   class Chain;
   class List;

/** An event iterator is used to traverse an event set, an event chain,
    an event list or an evnet window. This is universal bidirectional
    iterator which hides the actual implementation and works with all 
    event containers.

    @memo Defines an event iterator
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class Iterator : public std::iterator 
   <std::bidirectional_iterator_tag, Event, int, Event*, Event&> {
      friend class ConstIterator;
   
   public:      
      /** Creates an NULL event iterator.
          @memo Constructs an NULL event iterator
          @return Default onstructor
       ******************************************************************/
      Iterator () : mIter (0) {
      }
      /** Creates iterator based on an implementation
          @memo Constructor
       ******************************************************************/
      Iterator (const IteratorImp& imp)
      : mIter (imp.Copy()) {
      }
      /** Copy constuctor
          @memo Copy constructor
       ******************************************************************/
      Iterator (const Iterator& i) 
      : mIter (i.mIter ? i.mIter->Copy() : 0) {
      }
      /** Destroy the iterator
          @memo Destructor
       ******************************************************************/
      ~Iterator() {
         delete mIter; }
      /** Assignment operator
          @memo Assignment opetator
       ******************************************************************/
      Iterator& operator= (const Iterator& i) {
         if (this != &i) { delete mIter; 
            mIter = (i.mIter ? i.mIter->Copy() : 0); }
         return *this; }
   
      /** Equality operator
          @memo Equality operator
       ******************************************************************/
      bool operator== (const Iterator& i) const {
         return (mIter == i.mIter) || 
            (mIter && i.mIter && (mIter->Get() == i.mIter->Get())); }
      /** Inequality operator
          @memo Inequality operator
       ******************************************************************/
      bool operator!= (const Iterator& i) const {
         return !(*this == i); }
   
      /** Increment operator (prefix)
          @memo Increment operator (prefix)
       ******************************************************************/
      Iterator& operator++ () {
         mIter->Inc(); 
         return *this; }
      /** Increment operator (postfix)
          @memo Increment operator (postfix)
       ******************************************************************/
      Iterator operator++ (int) {
         Iterator i = *this; mIter->Inc(); 
         return i; }
      /** Decrement operator (prefix)
          @memo Decrement operator (prefix)
       ******************************************************************/
      Iterator& operator-- () {
         mIter->Dec(); 
         return *this; }
      /** Decrement operator (postfix)
          @memo Decrement operator (postfix)
       ******************************************************************/
      Iterator operator-- (int) {
         Iterator i = *this; mIter->Dec(); 
         return i; }
      /** Plus operator. Should be made more efficient!
          @memo Plus operator
       ******************************************************************/
      Iterator operator+ (int delta) const {
         Iterator i = *this;
         for (int j = 0; j < delta; ++j) ++i;
         for (int j = 0; j < -delta; ++j) --i;
         return i; }
      /** Minus operator
          @memo Minus operator
       ******************************************************************/
      Iterator operator- (int delta) const {
         return *this + (-delta); }
      /** Dereference operator
          @memo Dereference operator
       ******************************************************************/
      reference operator*() const {
         return *mIter->Get(); }
      /** Pointer operator
          @memo Pointer operator
       ******************************************************************/
      pointer operator->() const {
         return mIter ? mIter->Get() : 0; }
      /** Get implementation
          @memo Get implementation
       ******************************************************************/
      IteratorImp* GetImplementation() {
         return mIter; }
      /** Get implementation
          @memo Get implementation
       ******************************************************************/
      const IteratorImp* GetImplementation() const {
         return mIter; }
   
   private:
      /// Iterator implementation
      IteratorImp*	mIter;   
   };


/** A constant event iterator.

    @memo Defines a const event iterator
    @author Written June 2001 by Masahiro Ito and Daniel Sigg
    @version 1.0
 ************************************************************************/
   class ConstIterator: public std::iterator 
   <std::bidirectional_iterator_tag, Event, int, 
   const Event*, const Event&> {
   
   public:
      /** Creates an NULL event iterator.
          @memo Constructs an NULL event iterator
          @return Default onstructor
       ******************************************************************/
      ConstIterator () : mIter (0) {
      }
      /** Creates iterator based on an implementation
          @memo Constructor
       ******************************************************************/
      ConstIterator (const IteratorImp& imp)
      : mIter (imp.Copy()) {
      }
      /** Copy constuctor
          @memo Copy constructor
       ******************************************************************/
      ConstIterator (const ConstIterator& i) 
      : mIter (i.mIter ? i.mIter->Copy() : 0) {
      }
      /** Convert from non-const
          @memo Convert from non-const
       ******************************************************************/
      ConstIterator (const Iterator& i) 
      : mIter (i.mIter ? i.mIter->Copy() : 0) {
      }
      /** Destroy the iterator
          @memo Destructor
       ******************************************************************/
      ~ConstIterator() {
         delete mIter; }
      /** Assignment operator
          @memo Assignment opetator
       ******************************************************************/
      ConstIterator& operator= (const ConstIterator& i) {
         if (this != &i) { delete mIter; 
            mIter = (i.mIter ? i.mIter->Copy() : 0); }
         return *this; }
      /** Assignment from non-const
          @memo Assignment opetator
       ******************************************************************/
      ConstIterator& operator= (const Iterator& i) {
         delete mIter; mIter = (i.mIter ? i.mIter->Copy() : 0);
         return *this; }
   
      /** Equality operator
          @memo Equality operator
       ******************************************************************/
      bool operator== (const ConstIterator& i) const {
         return (mIter == i.mIter) || 
            (mIter && i.mIter && (mIter->Get() == i.mIter->Get())); }
      /** Inequality operator
          @memo Inequality operator
       ******************************************************************/
      bool operator!= (const ConstIterator& i) const {
         return !(*this == i); }
   
      /** Increment operator (prefix)
          @memo Increment operator (prefix)
       ******************************************************************/
      ConstIterator& operator++ () {
         mIter->Inc(); 
         return *this; }
      /** Increment operator (postfix)
          @memo Increment operator (postfix)
       ******************************************************************/
      ConstIterator operator++ (int) {
         ConstIterator i = *this; mIter->Inc(); 
         return i; }
      /** Decrement operator (prefix)
          @memo Decrement operator (prefix)
       ******************************************************************/
      ConstIterator& operator-- () {
         mIter->Dec(); 
         return *this; }
      /** Decrement operator (postfix)
          @memo Decrement operator (postfix)
       ******************************************************************/
      ConstIterator operator-- (int) {
         ConstIterator i = *this; mIter->Dec(); 
         return i; }
      /** Plus operator. Should be made more efficient!
          @memo Plus operator
       ******************************************************************/
      ConstIterator operator+ (int delta) const {
         ConstIterator i = *this;
         for (int j = 0; j < delta; ++j) ++i;
         for (int j = 0; j < -delta; ++j) --i;
         return i; }
      /** Minus operator
          @memo Minus operator
       ******************************************************************/
      ConstIterator operator- (int delta) const {
         return *this + (-delta); }
      /** Dereference operator
          @memo Dereference operator
       ******************************************************************/
      reference operator*() const {
         return *mIter->Get(); }
      /** Pointer operator
          @memo Pointer operator
       ******************************************************************/
      pointer operator->() const {
         return mIter ? mIter->Get() : 0; }
      /** Get implementation
          @memo Get implementation
       ******************************************************************/
      const IteratorImp* GetImplementation() const {
         return mIter; }
   
   private:
      /// Iterator implementation
      IteratorImp*	mIter;   
   };


// Define our own reverse iterator since CINT and SUN C++ 
// are incompatible otherwise :-(
template <class iter>
   class events_reverse_iterator
   : public std::iterator<typename iter::iterator_category,
   typename iter::value_type, typename iter::difference_type,           
   typename iter::pointer, typename iter::reference>
   {
   public:
      typedef iter forward_iterator;
      typedef events_reverse_iterator<iter> self;
      typedef typename iter::difference_type difference_type;
      typedef typename iter::value_type value_type;
      typedef typename iter::pointer pointer;
      typedef typename iter::reference reference;
   protected:
      iter current;
   public:
      typedef iter iterator_type;
      events_reverse_iterator() {
      }
      explicit events_reverse_iterator (iter x) : current(x) {
      }
   template <class U>
      events_reverse_iterator (const events_reverse_iterator<U>& x) : current(x.base()) {
      }
      bool operator==(const self& y) {
         return current == y.current; }
      bool operator!=(const self& y) {
         return current != y.current; }
      // bool operator< (const self& y) {
         // return y.current < current; }
      // bool operator<= (const self& y) {
         // return y.current <= current; }
      // bool operator> (const self& y) {
         // return y.current > current; }
      // bool operator>= (const self& y) {
         // return y.current >= current; }
      iter base () const { 
         return current; }
      reference operator* () const 
      { forward_iterator tmp = current; 
         return *--tmp; }
      pointer operator->() const 
      { reference tmp = operator*(); 
         return (pointer)&tmp; }
      self& operator++ () { 
         --current; 
         return *this; }
      self  operator++ (int) { 
         self tmp = *this; --current; 
         return tmp; }
      self& operator-- () { 
         ++current; 
         return *this; }
      self operator-- (int) { 
         self tmp = *this; ++current; 
         return tmp; }
      // self operator+  (difference_type n) const 
      // {  self tmp(current - n); 
         // return tmp; }
      // self& operator+= (difference_type n) { 
         // current -= n; 
         // return *this; }
      // self operator-  (difference_type n) const { 
         // self tmp(current + n); 
         // return tmp; }
      // self& operator-= (difference_type n) { 
         // current += n; 
         // return *this; }
      // reference operator[] (difference_type n) const { 
         // return *(*this + n); }
      // difference_type operator- (const self& y) {
         // return y.current - current; }
   };
template <class iter>
   events_reverse_iterator<iter> 
   operator+ (typename events_reverse_iterator<iter>::difference_type n, 
             const events_reverse_iterator<iter>& x) {
      events_reverse_iterator<iter> tmp = x;
      return (x += n); }

   // Reverse iterator
   typedef events_reverse_iterator <Iterator> ReverseIterator;
   // Const reverse iterator
   typedef events_reverse_iterator <ConstIterator> ConstReverseIterator;
#ifdef __CINT__
#endif
}

namespace std {

// Specialization of the back_insert_iterator for event sets
   class back_insert_iterator_Set : public std::iterator 
   <std::output_iterator_tag, void, void, void, void> {
   public:
      // container type
      typedef events::Set container_type;
      // Create a back_insert_iterator for container
      explicit back_insert_iterator_Set (container_type& container) :
      mContainer (&container) {
      }
      // Add an element to the end of the container
      back_insert_iterator_Set& 
      operator= (const events::Event& value);
      // Dereference
      back_insert_iterator_Set& operator*() { 
         return *this; }
      // Prefix increment
      back_insert_iterator_Set& operator++() { 
         return *this; }
      // Postfix increment
      back_insert_iterator_Set operator++ (int) { 
         return *this; }
   protected:
      // Pointer to event set
      events::Set*	mContainer;
   };

// Specialization of the back_inserter for event sets
   inline back_insert_iterator_Set
   back_inserter (events::Set& container) {
      return back_insert_iterator_Set (container); }

// Specialization of the insert_iterator for event sets
   class insert_iterator_Set : public std::iterator 
   <std::output_iterator_tag, void, void, void, void> {
   public:
      // container type
      typedef events::Set container_type;
      // Create a back_insert_iterator for container
      explicit insert_iterator_Set (container_type& container) :
      mContainer (&container) {
      }
      // Insert an element into the container
      insert_iterator_Set& 
      operator= (const events::Event& value);
      // Dereference
      insert_iterator_Set& operator*() { 
         return *this; }
      // Prefix increment
      insert_iterator_Set& operator++() { 
         return *this; }
      // Postfix increment
      insert_iterator_Set operator++ (int) { 
         return *this; }
   protected:
      // Pointer to event set
      events::Set*	mContainer;
   };

// Specialization of the inserter for event sets
   inline insert_iterator_Set 
   inserter (events::Set& container, events::Iterator) {
      return insert_iterator_Set (container); }


// Specialization of the back_insert_iterator for event chain
   class back_insert_iterator_Chain : public std::iterator 
   <std::output_iterator_tag, void, void, void, void> {
   public:
      // container type
      typedef events::Chain container_type;
      // Create a back_insert_iterator for container
      explicit back_insert_iterator_Chain (container_type& container) :
      mContainer (&container) {
      }
      // Add an element to the end of the container
      back_insert_iterator_Chain& 
      operator= (const events::Event& value);
      // Dereference
      back_insert_iterator_Chain& operator*() { 
         return *this; }
      // Prefix increment
      back_insert_iterator_Chain& operator++() { 
         return *this; }
      // Postfix increment
      back_insert_iterator_Chain operator++ (int) { 
         return *this; }
   protected:
      // Pointer to event chain
      events::Chain*	mContainer;
   };

// Specialization of the back_inserter for event chains
   inline back_insert_iterator_Chain 
   back_inserter (events::Chain& container) {
      return back_insert_iterator_Chain (container); }

// Specialization of the insert_iterator for event chains
   class insert_iterator_Chain : public std::iterator 
   <std::output_iterator_tag, void, void, void, void> {
   public:
      // container type
      typedef events::Chain container_type;
      // Create a back_insert_iterator for container
      explicit insert_iterator_Chain (container_type& container) :
      mContainer (&container) {
      }
      // Insert an element into the container
      insert_iterator_Chain& 
      operator= (const events::Event& value);
      // Dereference
      insert_iterator_Chain& operator*() { 
         return *this; }
      // Prefix increment
      insert_iterator_Chain& operator++() { 
         return *this; }
      // Postfix increment
      insert_iterator_Chain operator++ (int) { 
         return *this; }
   protected:
      // Pointer to event chain
      events::Chain*	mContainer;
   };

// Specialization of the inserter for event chains
   inline insert_iterator_Chain 
   inserter (events::Chain& container, events::Iterator) {
      return insert_iterator_Chain (container); }


// Specialization of the back_insert_iterator for event lists
   class back_insert_iterator_List : public std::iterator 
   <std::output_iterator_tag, void, void, void, void> {
   public:
      // container type
      typedef events::List container_type;
      // Create a back_insert_iterator for container
      explicit back_insert_iterator_List (container_type& container) :
      mContainer (&container) {
      }
      // Add an element to the end of the container
      back_insert_iterator_List& 
      operator= (const events::Event& value);
      // Dereference
      back_insert_iterator_List& operator*() { 
         return *this; }
      // Prefix increment
      back_insert_iterator_List& operator++() { 
         return *this; }
      // Postfix increment
      back_insert_iterator_List operator++ (int) { 
         return *this; }
   protected:
      // Pointer to event list
      events::List*	mContainer;
   };

// Specialization of the back_inserter for event lists
   inline back_insert_iterator_List 
   back_inserter (events::List& container) {
      return back_insert_iterator_List (container); }

// Specialization of the insert_iterator for event lists
   class insert_iterator_List : public std::iterator 
   <std::output_iterator_tag, void, void, void, void> {
   public:
      // container type
      typedef events::List container_type;
      // Create a back_insert_iterator for container
      explicit insert_iterator_List (container_type& container) :
      mContainer (&container) {
      }
      // Insert an element into the container
      insert_iterator_List& 
      operator= (const events::Event& value);
      // Dereference
      insert_iterator_List& operator*() { 
         return *this; }
      // Prefix increment
      insert_iterator_List& operator++() { 
         return *this; }
      // Postfix increment
      insert_iterator_List operator++ (int) { 
         return *this; }
   protected:
      // Pointer to event list
      events::List*	mContainer;
   };

// Specialization of the inserter for event lists
   inline insert_iterator_List 
   inserter (events::List& container, events::Iterator) {
      return insert_iterator_List (container); }


}

#endif // _LIGO_EVENTITERATOR_H
