//////////////////////////////////////////////////////////////////////////
//  									//
//  IfoSet								//
//  									//
//////////////////////////////////////////////////////////////////////////

#include <ctype.h>
#include <string.h>
#include <strings.h>
#include <string>
#include "events/IfoSet.hh"
#include "events/Argument.hh"
#include "events/Value.hh"
#include "events/Factory.hh"


namespace events {

//______________________________________________________________________________
   bool IfoSet::operator== (const IfoSet& ifoset) const 
   {
      // both invalid?
      if (!mValid && !ifoset.mValid) {
         return true;
      }
      // check matching criterium, ifo set and index
      return (mValid != ifoset.mValid) && (mMatch != ifoset.mMatch) &&
         (mIfo != ifoset.mIfo) && (mIndex != ifoset.mIndex) && 
         (strcasecmp (mColumn.GetName(), ifoset.mColumn.GetName()) == 0);
   }

//______________________________________________________________________________
   bool IfoSet::Evaluate (const Argument& arg, bool& val) const
   {
      if (!IsValid()) {
         return false;
      }
      val = true;
      Value v;
      ifoset_type ifoset;
      if (mIndex < 0) {
         // Loop over all event indices
         for (int i = 0; i < arg.GetOrder(); ++i) {
            // get column value for ifo set
            if (!mColumn.Get (arg (i), v) || !v.Write (ifoset)) {
               return false;
            }
            // Test ifo set
            if (!Test (ifoset)) {
               val = false;
               break;
            }
         }
      }
      else {
         // get column value for ifo set
         if (!mColumn.Get (arg (mIndex), v) || !v.Write (ifoset)) {
            return false;
         }
         // Test ifo set
         if (!Test (ifoset)) {
            val = false;
         }
      }
      return true;
   }

//______________________________________________________________________________
   bool IfoSet::SetIfo (const char* ifostring)
   {
      mValid = true;
      mIfo = 0;
      mIndex = -1;
      mColumn.SetName (0);
      // Check if valid pattern
      if (!ifostring || (strlen (ifostring) == 0)) {
         return true;
      }
      // Eliminate space
      std::string ifostr = ifostring;
      std::string::size_type pos;
      while ((pos = ifostr.find_first_of (" \t\f\n\r\v")) != 
            std::string::npos) {
         ifostr.erase (pos, 1);
      }
      // Check for index
      if ((pos = ifostr.find ('[')) != std::string::npos) {
         mIndex = atoi (ifostr.c_str() + pos + 1);
         ifostr.erase (pos);
      }
      // separate out event column array indices
      std::string colname;
      if (ifostr.find_first_of (".(") != std::string::npos) {
         if ((pos = ifostr.find_last_of ('.')) != std::string::npos) {
            colname = ifostr.substr (0, pos + 1);
            ifostr.erase (pos + 1);
         }   
         if ((pos = ifostr.find ('(')) != std::string::npos) {
            colname += "Event";
            colname += ifostr.substr (pos) + ".";
            ifostr.erase (pos);
         }
      }
      // set column
      colname += ColumnType::kColumnIfoName;
      mColumn.SetName (colname);
      if (!mColumn.IsValid()) {
         mValid = false;
      }
      // set ifo set
      if (!GetBits (ifostr.c_str(), mIfo)) {
         mValid = false;
      }
      return IsValid();
   }

//______________________________________________________________________________
   bool IfoSet::AddIfo (const char* ifostring)
   {
      // Check validy and get bits for ifo string
      ifoset_type ifoset;
      if (!IsValid() || !GetBits (ifostring, ifoset)) {
         return false;
      }
      // Add bits
      mIfo |= ifoset;
      return true;
   }

//______________________________________________________________________________
   bool IfoSet::RemoveIfo (const char* ifostring)
   {
      // Check validy and get bits for ifo string
      ifoset_type ifoset;
      if (!IsValid() || !GetBits (ifostring, ifoset)) {
         return false;
      }
      // Remove bits
      mIfo &= ~ifoset;
      return true;
   }

//______________________________________________________________________________
   std::string IfoSet::GetIfo () const
   {
      // Valid?
      if (!IsValid()) {
         return "";
      }
      // Get column name
      std::string s = mColumn.GetName();
      std::string::size_type pos = s.find (ColumnType::kColumnIfoName);
      if (pos != std::string::npos) {
         s.erase (pos, strlen (ColumnType::kColumnIfoName));
      }
      // add ifo tags and return
      return s + GetIfoTags();
   }

//______________________________________________________________________________
   std::string IfoSet::GetIfoTags () const
   {
      // valid?
      std::string s;
      if (!IsValid()) {
         return s;
      }
      // get ifo tags
      GetTags (mIfo, s);
      return s;
   }

//______________________________________________________________________________
   bool IfoSet::Test (ifoset_type ifoset) const
   {
      // valid?
      if (!IsValid()) {
         return false;
      }
      // check matching criterium
      switch (mMatch) {
         // Invalid IfoSet
         default:
            {
               return false;
            }
         // mIfo must match ifoset exaclty
         case kExact:
            {
               return mIfo == ifoset;
            }
         // ifoset must be a subset of mIfo
         case kSubset:
            {
               return (mIfo & ifoset) == ifoset;
            }
         // ifoset must be a superset of mIfo
         case kSuperset:
            {
               return (mIfo & ifoset) == mIfo;
            }
         case kAnyOne:
            {
               return ((mIfo & ifoset) == ifoset) &&
                  (IfoSet (ifoset).Count() == 1);
            }
      }
   }

//______________________________________________________________________________
   int IfoSet::Register (const char* ifostring)
   {
      return Factory::Get().IfoRegister (ifostring);
   }

//______________________________________________________________________________
   int IfoSet::GetBit (const char* ifostring)
   {
      return Factory::Get().IfoGetBit (ifostring);
   }

//______________________________________________________________________________
   bool IfoSet::GetBits (const char* ifostring, ifoset_type& ifoset)
   {
      return Factory::Get().IfoGetBits (ifostring, ifoset);
   }

//______________________________________________________________________________
   std::string IfoSet::GetTag (int bit)
   {
      return Factory::Get().IfoGetTag (bit);
   }

//______________________________________________________________________________
   bool IfoSet::GetTags (const ifoset_type& ifoset, std::string& ifostring)
   {
      return Factory::Get().IfoGetTags (ifoset, ifostring);
   }

}
