/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef PARAM_HH
#define PARAM_HH
//
//    Param object contain parameters.
//
#include <string>

/** Parameter base class defines the access operators for parameter objects.
  *  @memo Parameter value base class
  *  @author John G. Zweizig
  *  @version 1.1; Last modified April 1, 1999.
  */
class Param {
  public:
    /**  Base constructor.
      *  @memo  Base constructor.
      *  @param Type character indicating object type.
      */
    Param(char Type);

    /**  Destructor.
      *  @memo Destructor.
      */
    virtual ~Param(void) {}

    /**  Cast the parameter value to a integer. If the parameter can't
      *  be converted to integer, 0 is returned.
      *  @memo Cast parameter value to integer.
      *  @return Parameter value converted to an integer.
      */
    virtual operator int(void) const = 0;

    /**  Cast the parameter value to a double precision float. If the 
      *  parameter can't be converted to a float, 0.0 is returned.
      *  @memo Cast parameter value to double precision float.
      *  @return Parameter value converted to float.
      */
    virtual operator double(void) const = 0;

    /**  Cast the parameter value to a string.
      *  @memo Cast parameter value to a string.
      *  @return Parameter value converted to a string.
      */
    virtual std::string cvtstring(void) const = 0;

    /**  Set the parameter to an integer value. The argument is converted
      *  to the parameter type if necessary. The parameter type remains 
      *  unchanged.
      *  @memo Set parameter value to integer.
      *  \param x Integer value to which parameter is set. 
      *  @return Parameter value converted to an integer.
      */
    virtual int operator=(int x) = 0;

    /**  Set the parameter to an double float value.  The argument is 
      *  converted to the parameter type if necessary. The parameter type 
      *  remains unchanged.
      *  @memo Set parameter value to integer.
      *  \param x Double value to which parameter is set. 
      *  @return Parameter value converted to an integer.
      */
    virtual double operator=(double x) = 0;

    /**  Set the parameter to a string value.  The argument is converted
      *  to the parameter type if necessary. The parameter type remains 
      *  unchanged.
      *  @memo Set parameter value to integer.
      *  \param x String value to which parameter is set. 
      *  @return Parameter value converted to an integer.
      */
    virtual std::string operator=(const std::string& x) = 0;

    /**  Write the parameter value to the specified output stream.
      *  @memo Write the parameter value.
      *  \param ostr Output stream to receive parameter definition string.
      *  @return Reference to the output stream.
      */
    virtual std::ostream& Write(std::ostream& ostr) = 0;

    /**  Get the parameter type character.
      *  @memo Get the parameter type.
      *  @return Pointer to a character string containing the character type.
      */
    const char *getType(void) const;

  private:
    /** Parameter type enumerator.
     */
    typedef enum {t_void, t_int, t_double, t_string, t_end} dtype;

    /** Parameter type .
     */
    dtype mType;

    /** Parameter type code.
     */
     static const char *mTypStr[t_end];
};

/**  Typed parameter class template.
 */
template<class T, const char tchar>
class param_type : public Param {
  public:

    /**  Default constructor.
      *  @memo Default constructor.
      */
    param_type(void);

    /**  Construct a parameter object an initialize it with an argument 
      *  of the same type.
      *  @memo Initializing Constructor.
      *  \param Value Value to which parameter will be initialized.
      */
    param_type(T Value);

    /**  Construct a parameter object and initialize it from a string.
      *  @memo Initializing Constructor.
      *  \param vStr String value for parameter
      */
    param_type(const char *vStr);

    /**  Construct a parameter object and initialize it from a string.
      *  @memo Initializing Constructor.
      *  \param x Parameter to be copied.
      */
    param_type(const Param& x);

    /**  Destroy the parameter.
      *  @memo Destructor.
      */
    ~param_type(void);

    /**  Cast the parameter value to a integer. If the parameter can't
      *  be converted to integer, 0 is returned.
      *  @memo Cast parameter value to integer.
      *  @return Parameter value converted to an integer.
      */
    operator int(void) const;

    /**  Cast the parameter value to a double precision float. If the 
      *  parameter can't be converted to a float, 0.0 is returned.
      *  @memo Cast parameter value to double precision float.
      *  @return Parameter value converted to float.
      */
    operator double(void) const;

    /**  Cast the parameter value to a string.
      *  @memo Cast parameter value to string.
      *  @return Parameter value converted to a string.
      */
    std::string cvtstring(void) const;

    /**  Set the parameter to an integer value.  The argument is converted
      *  to the parameter type if necessary. The parameter type remains 
      *  unchanged.
      *  @memo Set parameter value to integer.
      *  \param x Integer value to which parameter is assigned.
      *  @return Parameter value converted to an integer.
      */
    int operator=(int x);

    /**  Set the parameter to a double precision float value. The argument 
      *  is converted to the parameter type if necessary. The parameter 
      *  type remains unchanged.
      *  @memo Set parameter value to integer.
      *  \param x Double value to which parameter is assigned.
      *  @return Parameter value converted to an integer.
      */
    double operator=(double x);

    /**  Set the parameter to a string value. The argument is converted
      *  to the parameter type if necessary. The parameter type remains 
      *  unchanged.
      *  @memo Set parameter value to a string.
      *  \param x String value to which parameter is assigned.
      *  @return Parameter value converted to an integer.
      */
    std::string operator=(const std::string& x);

    /**  Write the parameter value to the specified output stream.
      *  @memo Write the parameter value.
      *  \param ostr Stream to which parameter definition will be written.
      *  @return Reference to the output stream.
      */
    std::ostream& Write(std::ostream& ostr) {
        return (ostr << mValue);}

  private:
    T mValue;
};

//--------------------------------------  Inline functions
inline const char*
Param::getType(void) const {
    return mTypStr[mType];
}


//
//---> Parameter type classes.
typedef param_type<int,    'i'> param_int;
typedef param_type<double, 'd'> param_double;
typedef param_type<std::string, 's'> param_string;

#endif  //  PARAM_HH
