/* -*- mode: c++; c-basic-offset: 3; -*- */
#include "PConfig.h"
#include "web/monweb.hh"
#include "web/webclient.hh"
#include "Interval.hh"
#include "MonIndex.hh"
#include "MonClient.hh"
#include "NameClient.hh"
#include "TSeries.hh"
#include "FSeries.hh"
#include "FSpectrum.hh"
#include "Histogram1.hh"
#include "xml/XsilTSeries.hh"
#include "xml/XsilFSpectrum.hh"
#include "xml/XsilHistogram.hh"
#include <cstdio>
#include <iostream>
#include <fstream>
#include <time.h>
#include <cstdlib>
#include <unistd.h>
#ifndef __GNU_STDC_OLD
#include <sstream>
#else
#include "gnusstream.h"
#endif

#ifdef GDS_GRAPHIT
const char* GraphIt = GDS_GRAPHIT;
#else
const char* GraphIt = "GraphIt";
#endif

// Creates a random set of 3*N alphanumeric characters (N numbers+N 
// capitals+Nsmalcaps) This should be unique enough ...
   char* RandomStr(int N)
   {
      if(N>60) N=60;
      static char s[180];
   
      srand(time(0) - 1023285971);
   
      for(int t=0; t<3*N; t+=3) {
         s[t]   = 48 + (int) (10.0*rand()/(RAND_MAX+1.0)); // Numeric
         s[t+1] = 65 + (int) (26.0*rand()/(RAND_MAX+1.0)); // Capital
         s[t+2] = 97 + (int) (26.0*rand()/(RAND_MAX+1.0)); // Small caps
      }
      return s;
   }

namespace web {
   using namespace std;

   const double NAME_TIMEOUT = 300;
   const int OBJECT_WAIT = 3;

   const char* const html_header =
   "<!doctype html public \"-//w3c//dtd html 4.0 transitional//en\">\n"
   "<html>\n"
   "<head>\n"
   "   <meta http-equiv=\"Content-Type\" content=\"text/html; "
   "charset=iso-8859-1\">\n"
   "   <meta name=\"Author\" content=\""
   "Daniel Sigg and Szabolcs Marka\">\n"
   "   <meta name=\"GENERATOR\" content=\"monweb (LIGO)\">\n"
   "   <title>Monitor Web</title>\n"
   "</head>\n"
   "<body>";
   const char* const html_trailer =
   "</body>\n"
   "</html>";

   // cool!
   const char* const favicon =
   "AAABAAEAEBAAAAAAAABoBQAAFgAAACgAAAAQAAAAIAAAAAEACAAAAAAAAAAAAAAAAAAAAAAA"
   "AAAAAAAAAAAAAAAA////AKiOjwB3MDAAwMDAAGkTEwCtqqoAhkREAHQhIQB9QEAAcCsrAG80"
   "NABpGhkAbCEhAIc/PgC8vLwAfUREAHQtLQBnFhYAZRkYAG8jIwBuHx8AdCMjAL+/vgBrExMA"
   "bSAgAG8iIgB1IiEAbR8fAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA"
   "AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA"
   "AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA"
   "AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA"
   "AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA"
   "AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA"
   "AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA"
   "AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA"
   "AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA"
   "AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA"
   "AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA"
   "AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA"
   "AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA"
   "AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA"
   "AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA"
   "AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA"
   "AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAPPv"
   "7wDMvb0A/v7+/v7+/v7+/v7+/v7+/v7+/v7+/v7+/v7+/v7+/v7+/v7+/v7+/v7+/v7+/v7+"
   "BAQEBP/+/v7+/v7+/v7+/v4SEhIEEgT/EhL+/v4SEv7+Ev7+/hL+EgT/Ev4S/v4SBBIE//4S"
   "/hL+BBL+Ev7+Ev4S/gQEEv8S/hIE/hL+/hL+Ev7+/hIEEv7+BP8S/v4SBBL//v4S/hL//hIE"
   "Ev7+Ev4SBAT/Ev7+EhL+BP4SEv7+/v7+BP/+/gT+/gT//v7+BAT//v4E/v4E//7+BP7+/v7+"
   "BP/+BP/+/gT+/gT+/v7+/v4E/v4E/v4E/v4E/v7+/v7+BP7+BP7+BP7+BP7+/gAAAAAAAAAA"
   "AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA"
   "AAA=";
   const int favicon_len = 1406;


//////////////////////////////////////////////////////////////////////////
//                                                                      //
// Utilities                                                            //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
template <class T>
   char getRawTS (const T& d, monweb_server::data_type dtype, 
                 int n, double x0, double dx, char*& buf, int& len, 
                 int RqstType)
   {
      char ret = 1;
   
      if ((dtype != monweb_server::binary) &&
         (dtype != monweb_server::ascii)) {
         return 0;  // Unknown type...
      }
      float* xy = new float [2 * n];
      d.getData (n, xy);
   
      bool isLogY = false;
      if (RqstType > 1) {
        // Determines whether a logarithmic representation is more 
        // suitable than a linear. Logarithic axis for X
      /*  bool isLogX = false;
        double Min = d.get??();
        double Max = d.get??();
        if ((Min >= 0.0) && ((Max/Min)>10.0)) isLogX = true;
      */
        // Logarithic axis for Y
         double Min = d.getMaximum();
         double Max = d.getMinimum();
         if ((Min >= 0.0) && ((Max/Min)>10.0)) isLogY = true;
      }
   
      for (int i = n - 1; i >= 0; --i) {
         xy[2*i+1] = xy[i];
      }
      for (int i = 0; i < n; ++i) {
         xy[2*i] = x0 + i * dx;
      }
      if (dtype == monweb_server::binary) {
         len = 2 * n * sizeof(float);
         buf = (char*) xy;
         return ret;
      }
   
      // RqstType varieties:
      // 1 - Text, binary or xml (full data should be provided)
      // 2 - Graphics formats, must be pretty... (only meaningful 
      //     data provided)
      ostringstream s;
      if ((RqstType > 1) && (isLogY))  {
         ret = 2;
         for (int i = 0; i < n; ++i) {
            if (xy[2*i+1]) {
               s << xy[2*i] << "\t" << xy[2*i+1] << endl;
            }
         }
      } 
      else {
	 if (dtype == monweb_server::ascii && x0 == 0) {
	    s << "# GPS: " << d.getStartTime().getS() << endl;
	 }
         for (int i = 0; i < n; ++i) {
            s << xy[2*i] << "\t" << xy[2*i+1] << endl;
         }
      }
   
      if (RqstType > 1) s << "End_of_buffer" << endl;
   
      delete [] xy;
      len = s.str().size();
      buf = new char [len];
      memcpy (buf, s.str().c_str(), len);
   
      // Return conditions:
      //   0 - Error, unsuccessful
      //   1 - OK, linear scaling on both axes recommended
      //   2 - OK, log scaling on Y axis recommended (data 
      //       formatted accordingly, semilog y)
      return ret;
   }

template <class T>
   char getRawData (const T& d, monweb_server::data_type dtype, 
                   int n, double x0, double dx, char*& buf, 
                   int& len, int RqstType)
   {
      char ret = 1;
   
      if ((dtype != monweb_server::binary) &&
         (dtype != monweb_server::ascii)) {
         return 0;  // Unknown type...
      }
      float* xy = new float [2 * n];
      d.getData (n, xy);
   
      for (int i = n - 1; i >= 0; --i) {
         xy[2*i+1] = xy[i];
      }

      for (int i = 0; i < n; ++i) {
         xy[2*i] = x0 + i * dx;
      }

      if (dtype == monweb_server::binary) {
         len = 2 * n * sizeof(float);
         buf = (char*) xy;
         return ret;
      }
   
      ostringstream s;      
      for (int i = 0; i < n; ++i) {
         s << xy[2*i] << "\t" << xy[2*i+1] << endl;
      }
   
      delete [] xy;
      len = s.str().size();
      buf = new char [len];
      memcpy (buf, s.str().c_str(), len);
   
      // Return conditions:
      //   0 - Error, unsuccessful
      //   1 - OK
      return ret;
   }


   char getRawHist (Histogram1& d, monweb_server::data_type dtype, 
                   int n, double x0, double dx, char*& buf, 
                   int& len, int RqstType)
   {
      char ret = 1;
      double Min =  9.9E32;
      double Max = -9.9E32;
   
      if ((dtype != monweb_server::binary) &&
         (dtype != monweb_server::ascii)) {
         return 0;
      }
      float* xy = new float [2 * n];
      for (int i = n - 1; i >= 0; --i) {
         xy[2*i+1] = d.GetBinContent(i);
         if (xy[2*i+1] > Max) Max = xy[2*i+1];
         if (xy[2*i+1] < Min) Min = xy[2*i+1];
      }
      for (int i = 0; i < n; ++i) {
         xy[2*i] = d.GetBinCenter(i);
      }
      if (dtype == monweb_server::binary) {
         len = 2 * n * sizeof(float);
         buf = (char*) xy;
         return ret;
      }
   
      bool isLogY = false;
      if (RqstType > 1) {
        // Determines whether a logarithmic representation is more 
        // suitable than a linear. Logarithic axis for Y
         if ((Min >= 0.0) && ((Max/Min)>10.0)) isLogY = true;
      }
   
      // RqstType varieties:
      // 1 - Text, binary or xml (full data should be provided)
      // 2 - Graphics formats, must be pretty... (only meaningful 
      //     data provided)
      ostringstream s;
      if ((RqstType > 1) && (isLogY))  {
         ret = 2;
         for (int i = 0; i < n; ++i) {
          // Make sure that it is suitable for semilog plotting...
            if ( xy[2*i+1] > 0 ) { 
               s << xy[2*i] << "\t" << xy[2*i+1] << endl;
            }
         }
      } 
      else if (RqstType > 1) {
         for (int i = 0; i < n; ++i) {
            if ( xy[2*i+1] > 0 ) { 
               s << xy[2*i] << "\t" << xy[2*i+1] << endl;
            }
         }
      } 
      else {
         for (int i = 0; i < n; ++i) {
            s << xy[2*i] << "\t" << xy[2*i+1] << endl;
         }
      }
   
      if (RqstType > 1) s << "End_of_buffer" << endl;
   
      delete [] xy;
      len = s.str().size();
      buf = new char [len];
      memcpy (buf, s.str().c_str(), len);
      return ret;
   }

//______________________________________________________________________________
   static string html_ref (const string& url, const string& target) 
   {
      return string ("<a href=\"") + url + "\">" + target + "</a>";
   }

//////////////////////////////////////////////////////////////////////////
//                                                                      //
// monweb_server                                                        //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   monweb_server::monweb_server (int argn, char** argv, int cachesize) 
   : basic_server (argn, argv, DEFAULT_THREADS)
   {
      if (cachesize != 0) {
         fCache.setMax (cachesize);
         fCache.enableAutoCleanup();
      }
   }

//______________________________________________________________________________
   bool monweb_server::RequestGet (const request_t& request,
                     response_t& response)
   {
      cout << "received request for \"" << request.GetDemangledUrl() <<
         "\"" << endl;
      UpdateNames();
      response.SetStatusLine ("HTTP/1.0 200 OK");

      string url = request.GetUrl();
   
      // Add date field
      
      //char buf[1024];
      //time_t tSec = getUTC (Now());
      //strftime (buf, sizeof (buf), "%a, %d %b %Y %X GMT", gmtime (&tSec));
      //response.AddHeader ("Date", buf);
      const char* tfmt = "%3W, %02d %3M %Y %02H:%02N:%02S GMT";
      response.AddHeader("Date", TimeString(Now(), tfmt));
      //tSec = getUTC (GetUpdateTime());
      //strftime (buf, sizeof (buf), "%a, %d %b %Y %X GMT", gmtime (&tSec));
      //response.AddHeader ("Last-Modified", buf);
      response.AddHeader ("Last-Modified", TimeString(GetUpdateTime(), tfmt));
      // Add no-cache field
      response.AddHeader ("Cache-Control", "no-cache");   
      // Add server field
      response.AddHeader ("Server", "MonWeb/1.0 (LIGO)");
   
      // Check URL for monitor/data object
      string left = url;
      if (left.find ('/') == 0) {
         left.erase (0, 1);
      }
      string monitor_name;
      string::size_type pos = left.find ('/');
      if (pos != string::npos) {
         monitor_name = left.substr (0, pos);
         left.erase (0, pos + 1);
         monitor_name = http_request::demangle (monitor_name);
      }
      string object_name;
      pos = left.find ('/');
      if (pos != string::npos) {
         object_name = left.substr (0, pos);
         left.erase (0, pos + 1);
         object_name = http_request::demangle (object_name);
      }
      // data type
      data_type dtype = unsupported;
      pos = left.find_last_of ('.');
      if (pos != string::npos) {
         dtype = fromExt (left.substr (pos + 1));
      }
   
      // :TODO: better top page
      if (url == "/index.html") {
         ostringstream resp;
      
         resp << html_header << endl;
         resp << "<b><font color=\"#FF6600\"><font size=+2>" <<
            "GDS DMT Monitor Web Interface</font></font></b><br>" << endl;
         resp << "<b><font color=\"#FF8800\"><font size=+1>" <<
            "LIGO Laser Interferometer Gravitational-wave Observatory" <<
            "</font></font></b><br>" << endl;
         resp << "<hr SIZE=8 WIDTH=\"100%\"><br>" << endl;
         resp << "A list of available monitors can be found " << 
            html_ref ("monitors.html", "here") << ".<BR>" << endl;
         resp << "<hr SIZE=2 WIDTH=\"100%\"><br>" << endl;
         resp << "<br><font size=-1><a href=\"http://www.adobe.com/" <<
            "products/acrobat/readstep.html\">" <<
            "AcrobatReader</a> and " << endl;
         resp << "<a href=\"http://www.adobe.com/svg/viewer/install/" <<
            "main.html\"> Scalable Vector Graphics (SVG) Viewer</a> " <<
            "are necessary to access PDF files! </font>" << endl;
         resp << "<hr SIZE=2 WIDTH=\"100%\"><br>" << endl;
         resp << "<br><font size=-2>Questions or comments can be mailed to "
	      << "<a href=\"mailto:john.zweizig@ligo.org\">.</font>" << endl;
         resp << html_trailer << endl;
      
	 response.SetData(resp.str());
         response.AddHeader ("Content-Type", "text/html");
         response.TryCompress();
      }
      
      // No robots!
      else if (url == "/robots.txt") {
         ostringstream resp;
         resp << "# go away" << endl;
         resp << "User-agent: *" << endl;
         resp << "Disallow: /" << endl;
	 response.SetData(resp.str());
         response.AddHeader ("Content-Type", "text/plain");
      }
      
      // favicon
      else if (url == "/favicon.ico") {
         char* p = new char[favicon_len + 10];
         if (xml::base64decode (favicon, strlen (favicon), p, favicon_len, 0)) {
            response.SetData(p, favicon_len);
            response.RemoveHeader ("Cache-Control");   
            response.AddHeader ("Content-Type", "image/x-icon");
         }
         else {
            response.SetStatusLine ("HTTP/1.0 404 Not found");
         }
	 delete[] p;
      }
      
      // List of monitors (text file)
      else if (url == "/monitors.txt") {
         thread::semlock lockit (fMux);
         ostringstream resp;
         for (monitor_list::iterator i=fNames.begin(); i != fNames.end(); ++i){
            resp << i->first << "\r\n";
         }
         if (fNames.empty()) {
            resp << "Unavailable" << "\r\n";
         }
	 response.SetData(resp.str());
         response.AddHeader("Content-Type", "text/plain");
         response.TryCompress();
      }
      
      // List of monitors (html file)
      else if (url == "/monitors.html") {
         ostringstream resp;
         resp << html_header << endl;
         thread::semlock lockit (fMux);
         for (monitor_list::iterator i=fNames.begin(); i != fNames.end(); ++i) {
            string url = http_request::mangle(i->first) + "/objects.html";
            resp << html_ref(url, i->first) << "<BR>" << endl;
         }
         if (fNames.empty()) {
            resp << "No monitors available<BR>" << endl;
         }
         resp << html_trailer << endl;
	 response.SetData(resp.str());
         response.AddHeader ("Content-Type", "text/html");
         response.TryCompress();
      }
      
      // list of objects (text file)
      else if ((left == "objects.txt") && object_name.empty() && 
              IsMonitor (monitor_name)) {
         thread::semlock lockit (fMux);
         ostringstream resp;
         monitor_list::iterator i = fNames.find (monitor_name);
         if ((i != fNames.end()) && i->second.first.fLoaded &&
            !i->second.second.empty()) {
            for (object_list::iterator j = i->second.second.begin();
                j != i->second.second.end(); ++j) {
               resp << "\"" << j->first << "\"\t";
               switch (j->second.fType) {
                  case TimeSeries:
                     resp << "TimeSeries\t";
                     break;
                  case FrequencySeries:
                     resp << "FrequencySeries\t";
                     break;
                  case Spectrum:
                     resp << "Spectrum\t";
                     break;
                  case Histogram1D:
                     resp << "Histogram1D\t";
                     break;
               }
               if (j->second.fOpt.empty()) {
                  resp << "-\r\n";
               }
               else {
                  resp << "\"" << j->second.fOpt << "\"\r\n";
               }
            }
         }
         else {
            resp << "Unavailable" << "\r\n";
         }
         response.SetData(resp.str());
         response.AddHeader ("Content-Type", "text/plain");
         response.TryCompress();
      }
      
      // list of objects (html file)
      else if ((left == "objects.html") && object_name.empty() && 
              IsMonitor (monitor_name)) {
         ostringstream resp;
         resp << html_header << endl;
         resp << "<h1>" << monitor_name << "</h1><BR>" << endl;
         thread::semlock lockit (fMux);
         monitor_list::iterator i = fNames.find (monitor_name);
         if ((i != fNames.end()) && i->second.first.fLoaded &&
            !i->second.second.empty()) {
            resp << "<table BORDER WIDTH=\"100%\" >" << endl;
            resp << "<tr><td>Name</td><td>Type</td><td>Data" << 
               "</td></tr>" << endl;
            for (object_list::iterator j = i->second.second.begin();
                j != i->second.second.end(); ++j) {
               string urlbase = http_request::mangle (j->first) + "/data.";
               resp << "<tr><td>" << j->first << "</td>" << endl;
               switch (j->second.fType) {
                  case TimeSeries:
                     resp << "<td>time series</td>";
                     break;
                  case FrequencySeries:
                     resp << "<td>frequency series</td>";
                     break;
                  case Spectrum:
                     resp << "<td>power spectrum</td>";
                     break;
                  case Histogram1D:
                     resp << "<td>1D histogram</td>";
                     break;
                  default:
                     resp << "<td></td>" << endl;
                     break;
               }
               resp << "<td>"
		    << html_ref (urlbase + "xml", "xml") << ", "
		    << html_ref (urlbase + "txt", "text") << ", "
		    << html_ref (urlbase + "dat", "binary") << " , "
		    << html_ref (urlbase + "png", "png") << " , "
		    << html_ref (urlbase + "svg", "svg") << " , "
		    << html_ref (urlbase + "ps",  "ps") << " , "
		    << html_ref (urlbase + "pdf", "pdf") << " , "
		    << html_ref (urlbase + "gif", "gif") 
		    << "</td></tr>" << endl;
            }
            resp << "</table>" << endl;
         }
         else {
            resp << "No data objects available<BR>" << endl;
         }
         resp << html_trailer << endl;
         response.SetData(resp.str());
         response.AddHeader ("Content-Type", "text/html");
         response.TryCompress();
      }
      
      // data object
      else if ((dtype != unsupported) && IsObject (monitor_name, object_name)) {
         char* buf = 0;
         int len = 0;
         if (GetData (monitor_name, object_name, buf, len, dtype)) {
            response.SetData(buf, len);
            delete[] buf;
            response.AddHeader ("Content-Type", mimeType (dtype));
            // cache it
            if ((fCache.getMax() != 0) && 
               (strcasecmp (response.GetStatusLine(), "HTTP/1.0 200 OK") == 0)) {
               cout << "Cache \"" << request.GetDemangledUrl() << "\"" << endl;
               webcache::cachepage page (Now() + Interval(CACHING_TIME), 
                                    response.GetData(), response.GetLength(),
                                    response.GetHeader());
               if ((dtype != jpeg) && (dtype != gif) && (dtype != pdf)) {
                  page.setAllowCompress();
               }
               fCache.add (request.GetUrl(), page);
            }
            if ((dtype != jpeg) && (dtype != gif) && (dtype != pdf)) {
               response.TryCompress();
            }
         }
         else {
            cout << "?????????????????????????????????????????? *" << endl;
            response.SetStatusLine ("HTTP/1.0 404 Not found");
         }
      }
      
      
      // page not found
      else {
         cout << "?????????????????????????????????????????? $1" << endl;
         response.SetStatusLine ("HTTP/1.0 404 Not found");
      }
   
      // Set content length
      if (response.GetLength() > 0) {
         char buf[1024];
         sprintf (buf, "%i", response.GetLength());
         response.AddHeader ("Content-Length", buf);
      }
   
      return true; 
   }

//______________________________________________________________________________
   bool monweb_server::UpdateNames()
   {
      // Check timestamp
      thread::semlock lockit (fMux);
      Time now = Now();
      if (now - fNamesTime < Interval (NAME_TIMEOUT)) {
         return true;
      }
      // Get monitor list
      fNames.clear();
      lmsg::NameClient nc;
      lmsg::MsgAddr addr;
      lmsg::NameProcs type = lmsg::p_Monitor;
      //nc.setDebug(1);
      std::string index;
      int rc = nc.getIndex(index, addr.getIPAddr(), type);
      if (rc) {
         cerr << "Error in nameserver lookup " << rc << endl;
	 // wait at least one second before next request
         fNamesTime = Now() - Interval (NAME_TIMEOUT - 1);
         return false;
      }
      // go through index
      while (!index.empty()) {
         string line;
         string::size_type pos = index.find_first_of (';');
         if (pos == string::npos) {
            line = index;
            index = "";
         }
         else {
            line = index.substr (0, pos);
            index.erase (0, pos + 1);
         }
         if (!line.empty()) {
            fNames[line] = object_listing (object_access(), object_list());
         }
      }
      fNamesTime = Now();
      return true;
   }

//______________________________________________________________________________
   bool monweb_server::UpdateObjects (monitor_list::iterator i)
   {
      if (i->second.first.fLoaded) {
         return true;
      }
      if (i->second.first.fUpdateTime + Interval (OBJECT_WAIT) > Now()) {
         return false;
      }
      // create a monitor client
      string mon = i->first;
      i->second.second.clear();
      MonClient mc;
      //fMC.setDebug((lmsg::index_t)1);
      mc.setServer (mon.c_str());
      // get index
      MonIndex mi;
      if (!mi.setIndex (mc)) {
         i->second.first.fUpdateTime = Now();
         return false;
      }
      for (MonIndex::const_index_iterator j = mi.begin(); 
          j != mi.end(); j++) {
         data_object dobj;
         dobj.fName = mi.getName (j);
         string typen = mi.getType(j);
         dobj.fOpt = mi.getComment (j);
         if (typen == kSeriesTypenames[0]) {
            dobj.fType = TimeSeries;
         }
         else if (typen == kSeriesTypenames[1]) {
            dobj.fType = FrequencySeries;
         }
         else if (typen == kSeriesTypenames[2]) {
            dobj.fType = Spectrum;
         }
         else if (typen == kSeriesTypenames[3]) {
            dobj.fType = Histogram1D;
         }
         else {
            continue;
         }
         i->second.second.insert (object_list::value_type 
                              (dobj.fName, dobj));
      }
      i->second.first.fLoaded = true;
      i->second.first.fUpdateTime = Now();
      return true;
   }

//______________________________________________________________________________
   Time monweb_server::GetUpdateTime() const 
   {
      thread::semlock lockit (fMux);
      return fNamesTime;
   }
//______________________________________________________________________________
   bool monweb_server::IsMonitor (const string& name)
   {
      if (name.empty()) {
         return false;
      }
      thread::semlock lockit (fMux);
      monitor_list::iterator i = fNames.find (name);
      // check if object list needs lookup
      if ((i != fNames.end()) && !i->second.first.fLoaded) {
         UpdateObjects (i);
      }
      return (i != fNames.end());
   }

//______________________________________________________________________________
   bool monweb_server::IsObject (const string& mon, 
                     const string& obj) 
   {
      if (mon.empty() || obj.empty()) {
         return false;
      }
      thread::semlock lockit (fMux);
      monitor_list::iterator i = fNames.find (mon);
      // check if object list needs lookup
      if (i == fNames.end()) {
         cerr << "ERROR: MONITOR NOT FOUND " << mon << endl;
         return false;
      }
      if (!i->second.first.fLoaded) {
         if (!UpdateObjects (i)) {
            cerr << "ERROR: OBJECT LIST NOT FOUND FOR " << mon << endl;
            return false;
         }
      }
      // Look for object
      object_list::iterator j = i->second.second.find (obj);
      if (j == i->second.second.end()) {
         cerr << "ERROR: OBJECT NOT FOUND " << obj << "(" << mon << ")" << endl;
      }
      return (j != i->second.second.end());
   }

//______________________________________________________________________________
   bool monweb_server::GetData (const std::string& mon, const std::string& obj,
				char*& buf, int& len, data_type dtype)
   {
      buf = 0; len = 0;
      if (mon.empty() || obj.empty()) {
         cout << "?????????????????????????????????????????? 1" << endl;
         return false;
      }
      // Determine type
      SeriesType type;
      {
         thread::semlock lockit (fMux);
         monitor_list::iterator i = fNames.find (mon);
         // check if object is valid
         if ((i == fNames.end()) || !i->second.first.fLoaded) {
            cout << "?????????????????????????????????????????? 2" << endl;
            return false;
         }
         // Look for object
         object_list::iterator j = i->second.second.find (obj);
         if (j == i->second.second.end()) {
            cout << "?????????????????????????????????????????? 3" << endl;
            return false;
         }
         type = j->second.fType;
      }

      //--------------------------------  Get data from monitor
      MonClient mc;
      //fMC.setDebug((lmsg::index_t)1);
      mc.setServer (mon.c_str());
   
      time_t CurrentTime;
      time (&CurrentTime);
      // --- Dumps the request details into the access log file
      // if (fLog) {
         // *fLog << "UTC :: " << asctime(gmtime(&CurrentTime));
         // *fLog << fileType(dtype) << " : " << mimeType(dtype) << 
            // " | " << mon << " - " << obj << endl ;
      // }
   
      //--------------------------------  Create scratch file name jic.
      ostringstream TempDataCore;
      TempDataCore << "/tmp/MonWeb_" << pthread_self() << RandomStr(10);
      string DataOutFile = TempDataCore.str() + ".dat";

      //--------------------------------  Handle type-dependent request.
      switch (type) {

      //--------------------------------  Time Series
      case TimeSeries:
	 {
	    TSeries ts;
	    int err = mc.getData (obj.c_str(), &ts);
	    if (err) {
	       cout << "?????????????????????????????????????????? 4" << endl;
	       return false;
	    }
	    switch (dtype) {
	    case xml:
	       {
		  ostringstream s;
		  s << xml::xsilHeader() << endl;
		  s << xml::xsilTSeries (&ts) << endl;
		  s << xml::xsilTrailer() << endl;
		  len = s.str().size();
		  buf = new char [len];
		  memcpy (buf, s.str().c_str(), len);
		  break;
	       }
	    case ascii:
	    case binary:
	       {
		  return getRawTS 
		     (ts, dtype, ts.getNSample(), 0,
		      (double)ts.getTStep(), buf, len, 1);
	       }
	    case png:
	    case gif:
	    case ps:
	    case pdf:
	    case svg:
	       {
		  // --- Create a temporary file
		  ofstream Odat(DataOutFile.c_str(), ios::out | ios::app);
		  if (!Odat) {
		     cerr << "WebView error: Cannot open temporary data "
			  << "buffer (" << DataOutFile <<  ")! Please "
			  << "check the disk usage and the permissions "
			  << "on /tmp (should be drwxrwxrwt)." << endl;
		     return false;
		  }

		  //-------------------  Fiddle the units for long (>10h) plots
		  size_t N = ts.getNSample();
		  double x0(0);
		  double dt = ts.getTStep();
		  string xunit="s";
		  if (N*dt > 36000) {
		     xunit="h";
		     dt *= 1.0/3600.0;
		  }    
		      
		  char isLog = getRawTS (ts, ascii, ts.getNSample(), x0,
					 dt, buf, len, 2);
		  
		  Odat << buf << endl ;
		  Odat.close();
		  delete[] buf;
		  buf = 0;
		  
		  // --- Creates graphs in the desired format 
		  ostringstream GraphCommand;
		  GraphCommand << GraphIt << " " << DataOutFile 
			       << " " << fileType(dtype);
		  if (ts.getTStep().GetSecs()  < 1) {
		     GraphCommand << " \"Time (" << xunit << ", "
				  << (1/ts.getTStep())  << "Hz sampling)\" ";
		  } 
		  else {
		     GraphCommand << " \"Time (" << xunit << ", " 
				  << (ts.getTStep()) << "s/sample)\" ";
		  }

		  if (strlen(ts.getName()) > 0) {
		     GraphCommand << " \"Magnitude (" << ts.getName()
				  << ")\" \"";
		  } 
		  else {
		     GraphCommand << " \"Magnitude\" \"";
		  }
		  GraphCommand << obj << " - " << mon << "\" " 
			       << (unsigned int) ts.getStartTime().getS();
		  if (isLog == 2) {
		     GraphCommand << " \"-l y -S 0 -m 2 \""
				  << " \"0 0 1.0 -S 2 0.01 -m 0 \"";
		  } 
		  else {
		     GraphCommand << " \"-S 0 -m 2 \""
				  << " \"0 0 1.0 -S 2 0.01 -m 0 \"";
		  }

		  // --- Dump the request details into the access 
		  //     log file
		  if (fLog) *fLog << ts.getStartTime().getS()<< " | Command : "
				  << GraphCommand.str() << endl ;

		  // --- Executes shell script
		  if ( system(GraphCommand.str().c_str()) != 0 ) {
		     if (fLog) *fLog << "ERROR encountered while executing "
				     << "system(...) command!" << endl ;
		     cerr << "WebView error: ERROR encountered while " 
			  << "executing system(...) command!" << endl ;
		     return false;
		  }

		  // --- Retrive the result...
		  string DataInFile = TempDataCore.str()+"."+fileType(dtype);
		  // --- Dums the details into the access log file
		  if (fLog) *fLog << mimeType(dtype) <<" - "<< fileType(dtype)
				  << " :: " << DataInFile << endl;

		  time_t StartTime, NowTime;
		  time(&StartTime);
		  bool TimeOut=false;
		  ifstream INF(DataInFile.c_str(), ios::in | ios::binary);
		  while ((!INF) && (!TimeOut)) {
		     INF.open(DataInFile.c_str(), ios::in | ios::binary);
		     time(&NowTime);
		     if ( (NowTime - StartTime) > 180) TimeOut = true;
		     usleep(500000); // should never have to wait 1/2 second
		  }
		  if (TimeOut) {
		     cerr << "WebView error: Cannot open the temporary "
			  << "graphics file (" << DataInFile
			  << ")! Please check the disk usage and the "
			  << "permissions; start with /tmp (It should"
			  << " be drwxrwxrwt)." << endl;
		     if (fLog) *fLog << "WebView error: Cannot open the "
				     << "temporary graphics file ("
				     << DataInFile << ")" << endl;
		     return false;
		  }

		  // --- Send data...
		  if (fLog) *fLog << "Sending data ... ";
		  char ch;
		  ostringstream s;
		  while(INF) {
		     INF.get(ch);
		     s << ch ;
		  }

		  len = s.str().size();
		  buf = new char [len];
		  memcpy (buf, s.str().c_str(), len);
		  if (fLog) *fLog << "Data transmitted." << endl;

		  // --- Cleaning up
		  remove(DataInFile.c_str());
		  if (fLog) *fLog << "Removed: " << DataInFile << endl;
		  break;
	       }
	       // :TODO: all other formats
	    default:
	       return false;
	    }
	    break;
	 }
         
      //--------------------------------  Frequency series
      case FrequencySeries:
	 {
	    FSeries fs;
	    int err = mc.getData (obj.c_str(), &fs);
	    if (err) return false;
	    
	    switch (dtype) {
	    case xml:
	       {
		  ostringstream s;
		  s << xml::xsilHeader() << endl;
		  s << xml::xsilFSeries (&fs) << endl;
		  s << xml::xsilTrailer() << endl;
		  len = s.str().size();
		  buf = new char [len];
		  memcpy (buf, s.str().c_str(), len);
		  break;
	       }
	    case ascii:
	    case binary:
	       {
		  return getRawData(fs, dtype, fs.getNStep(), fs.getLowFreq(),
				    fs.getFStep(), buf, len, 1);
	       }
	       // :TODO: all other formats
	    default:
	       return false;
	    }
	    break;
	 }
      case Spectrum:
	 {
	    FSpectrum fs;
	    int err = mc.getData (obj.c_str(), &fs);
	    if (err) return false;

	    switch (dtype) {
	    case xml:
	       {
		  ostringstream s;
		  s << xml::xsilHeader() << endl;
		  s << xml::xsilFSpectrum (&fs) << endl;
		  s << xml::xsilTrailer() << endl;
		  len = s.str().size();
		  buf = new char [len];
		  memcpy (buf, s.str().c_str(), len);
		  break;
	       }
	    case ascii:
	    case binary:
	       {
		  return getRawData(fs, dtype, fs.getNStep(), fs.getLowFreq(),
				    fs.getFStep(), buf, len, 1);
	       }
	    case png:
	    case gif:
	    case ps:
	    case pdf:
	    case svg:
	       {
		  //----------------------  Try to open the scratch file
		  ofstream Odat(DataOutFile.c_str(), ios::out | ios::app);
		  if (!Odat) {
		     cerr << "WebView error: Cannot open temporary data "
			  << "buffer (" << DataOutFile <<  ")! Please "
			  << "check the disk usage and the permissions "
			  << "on /tmp (should be drwxrwxrwt)." << endl;
		     return false;
		  }

		  // --- Obtain data and write it into a temporary file
		  getRawData (fs, ascii, fs.getNStep(), fs.getLowFreq(), 
			      fs.getFStep(), buf, len, 1);	       
		  Odat << buf << endl ;
		  Odat.close();
		  delete[] buf;
		  buf = 0;

		  // --- Creates graphs in the desired format 
		  ostringstream GraphCommand;
	  	  GraphCommand << GraphIt << " " << DataOutFile 
			       << " " << fileType(dtype);
		  GraphCommand << " \"Frequency (" << fs.getFStep()
			       << "Hz bins)\" ";
		  if (strlen(fs.getName()) > 0) {
		     GraphCommand << " \"PSD (" << fs.getName()
				  << ")\" \"";
		  } 
		  else {
		     GraphCommand << " \"PSD\" \"";
		  }
		  GraphCommand << obj << " - " << mon << "\" " 
			       << fs.getStartTime().getS();
		  GraphCommand << " \"-l y -S 0 -m 2 \""
			       << " \"0 0 1.0 -S 2 0.01 -m 0 \"";
		  // --- Dumps the request details into the access 
		  //     log file
		  if (fLog) {
		     *fLog << fs.getStartTime().getS()<< " | Command : "
			   << GraphCommand.str() << endl ;
		  }
		  // --- Executes shell script
		  if ( system(GraphCommand.str().c_str()) != 0 ) {
		     if (fLog) {
			*fLog << "ERROR encountered while executing "
			      << "system(...) command!" << endl ;
		     }
		     cerr << "WebView error: ERROR encountered while " 
			  << "executing system(...) command!" << endl ;
		     return false;
		  }
		  
		  // --- Retrives the result...
		  string DataInFile = TempDataCore.str() + "." + fileType(dtype);
		  // --- Dumps the details into the access log file
		  if (fLog) *fLog << mimeType (dtype) <<" - "<< fileType(dtype)
				  << " :: " << DataInFile << endl;

		  time_t StartTime, NowTime;
		  time(&StartTime);
		  bool TimeOut=false;
		  ifstream INF(DataInFile.c_str(), ios::in | ios::binary);
		  while ((!INF) && (!TimeOut)) {
		     INF.open(DataInFile.c_str(), ios::in | ios::binary);
		     time(&NowTime);
		     if ( (NowTime - StartTime) > 180) TimeOut = true;
		  }
		  if (TimeOut) {
		     cerr << "WebView error: Cannot open the temporary "
			  << "graphics file (" << DataInFile
			  << ")! Please check the disk usage and the "
			  << "permissions; start with /tmp (It should"
			  << " be drwxrwxrwt)." << endl;
		     if (fLog) *fLog << "WebView error: Cannot open the "
				     << "temporary graphics file ("
				     << DataInFile << ")" << endl;
		     return false;
		  }
		  // --- Sending data...
		  if (fLog) *fLog << "Sending data ... ";
		  char ch;
		  ostringstream s;
		  while(INF) {
		     INF.get(ch);
		     s << ch ;
		  }
	       
		  len = s.str().size();
		  buf = new char [len];
		  memcpy (buf, s.str().c_str(), len);
		  if (fLog) *fLog << "Data transmitted." << endl;

		  // --- Cleaning up
		  remove(DataInFile.c_str());
		  if (fLog) *fLog << "Removed: " << DataInFile << endl;
		  break;
	       }
	       // :TODO: all other formats
	    default:
	       return false;
	    }
	    break;
	 }

      case Histogram1D:
	 {
	    Histogram1 hist;
	    int err = mc.getData (obj.c_str(), &hist);
	    if (err) {
	       return false;
	    }
	    switch (dtype) {
	    case xml:
	       {
		  ostringstream s;
		  s << xml::xsilHeader() << endl;
		  s << xml::xsilHistogram (&hist) << endl;
		  s << xml::xsilTrailer() << endl;
		  len = s.str().size();
		  buf = new char [len];
		  memcpy (buf, s.str().c_str(), len);
		  break;
	       }
	    case ascii:
	    case binary:
	       {
		  return getRawHist (hist, dtype, int(hist.GetNBins()), 
				     1, 0, buf, len, 1);
	       }
	    case png:
	    case gif:
	    case ps:
	    case pdf:
	    case svg:
	       {
		  //-----------------------  Try to open temp file.
		  ofstream Odat(DataOutFile.c_str(), ios::out | ios::app);
		  if (!Odat) {
		     cerr << "WebView error: Cannot open temporary "
			  << "data buffer (" << DataOutFile
			  << ")! Please check the disk usage and the "
			  << "permissions on /tmp (It should be "
			  << "drwxrwxrwt)." << endl;
		     return false;
		  }

		  //-----------------------  Obtain data
		  char isLog = getRawHist(hist, ascii, int(hist.GetNBins()), 
					  1, 0, buf, len, 2);

		  //-----------------------  write it into a temporary file
		  Odat << buf << endl;
		  Odat.close();
		  delete[] buf;
		  buf = 0;

		  // --- Create graphs in the desired format 
		  ostringstream GraphCommand;
		  GraphCommand << GraphIt << " " << DataOutFile 
			       << " " << fileType(dtype);
		  if (strlen(hist.GetXLabel()) > 0) {
		     GraphCommand << " \"" << hist.GetXLabel() << "\" ";
		  } 
		  else {
		     GraphCommand << " \"Bins (axis has no proper"
				  << " label...)\"";
		  }
		  if (strlen(hist.GetNLabel()) > 0) {
		     GraphCommand << " \"" << hist.GetNLabel() << "\" \"";
		  } 
		  else {
		     GraphCommand << " \"Number of events (axis"
				  << " has no proper label...)\" \"";
		  }
		  GraphCommand << obj << " - " << mon << "\" " 
			       << hist.GetTime().getS();
		  if (isLog == 2) {
		     GraphCommand << " \"-l y -S 0 -m 2 \""
				  << " \"0 0 1.0 -S 2 0.01 -m 0 \"";
		  } 
		  else {
		     GraphCommand << " \"-S 0 -m 2 \""
				  << " \"0 0 1.0 -S 2 0.01 -m 0 \"";
		  }

		  // --- Dump the request details into the access log file
		  if (fLog) *fLog << "Command : " << GraphCommand.str()
				  << endl ;
		  // --- Executes shell script
		  if ( system(GraphCommand.str().c_str()) != 0 ) {
		     if (fLog) *fLog << "ERROR encountered while "
				     << "executing system(...) command!"
				     << endl;
		     cerr << "WebView error: ERROR encountered while "
			  << "executing system(...) command!" << endl ;
		     return false;
		  }

		  // --- Retrives the result...
		  string DataInFile = TempDataCore.str() +"."+ fileType(dtype);

		  // --- Dumps the details into the access log file
		  if (fLog) *fLog << mimeType (dtype) << " - " 
				  << fileType(dtype) << " :: "
				  << DataInFile << endl;

		  time_t StartTime, NowTime;
		  time(&StartTime);
		  bool TimeOut=false;
		  ifstream INF(DataInFile.c_str(), ios::in | ios::binary);
		  while ((!INF) && (!TimeOut)) {
		     INF.open(DataInFile.c_str(), ios::in | ios::binary);
		     time(&NowTime);
		     if ( (NowTime - StartTime) > 180) TimeOut = true;
		  }
		  if (TimeOut) {
		     cerr << "WebView error: Cannot open the temporary "
			  << "graphics file (" << DataInFile
			  << ")! Please check the disk usage and the "
			  << "permissions; start with /tmp (It should"
			  << " be drwxrwxrwt)." << endl;
		     if (fLog) *fLog << "WebView error: Cannot open the"
				     << " temporary graphics file ("
				     << DataInFile << ")" << endl;
		     return false;
		  }
		  // --- Sending data...
		  if (fLog) *fLog << "Sending data ... ";
		  char ch;
		  ostringstream s;
		  while(INF) {
		     INF.get(ch);
		     s << ch ;
		  }

		  len = s.str().size();
		  buf = new char [len];
		  memcpy (buf, s.str().c_str(), len);
		  if (fLog) *fLog << "Data transmitted." << endl;
		  // --- Cleaning up
		  remove(DataInFile.c_str());
		  if (fLog) *fLog << "Removed: " << DataInFile << endl;
		  break;
	    }
	    // :TODO: all other formats
	    default:
	       return false;
	    }
	    break;
	 }
      }
      // time_t FinishTime;
      // time (&FinishTime);
      // double DeltaTime = difftime(FinishTime, CurrentTime);
      // if (fLog) *fLog << "UTC :: " << asctime(gmtime(&FinishTime)) << 
      // "Time required to satisfy request ( " << fileType(dtype) << 
      // " ) = " << DeltaTime << " secs" << endl;
      // if (fLog) *fLog << " -----------  -----------  -----------  " <<
      // "----------- " << endl;
      return true;
   }

//______________________________________________________________________________
   monweb_server::data_type monweb_server::fromExt (const string& ext)
   {
      if (ext == "xml") {
         return xml;
      }
      else if (ext == "txt") {
         return ascii;
      }
      else if (ext == "dat") {
         return binary;
      }
      else if (ext == "svg") {
         return svg;
      }
      else if (ext == "jpg") {
         return jpeg;
      }
      else if (ext == "png") {
         return png;
      }
      else if (ext == "gif") {
         return gif;
      }
      else if (ext == "pdf") {
         return pdf;
      }
      else if (ext == "ps") {
         return ps;
      }
      else {
         return unsupported;
      }
   }

//______________________________________________________________________________
   string monweb_server::mimeType (data_type dt) 
   {
      switch (dt) {
         case xml:
            return "text/xml";
         case ascii:
            return "text/plain";
         case binary:
            return "application/octet-stream";
         case svg:
            return "image/svg+xml";
         case jpeg:
            return "image/jpeg";
         case gif:
            return "image/gif";
         case png:
            return "image/x-png";
         case pdf:
            return "application/pdf";
         case ps:
            return "application/postscript";
         default:
            return "";
      }
   }

//______________________________________________________________________________
   string monweb_server::fileType (data_type dt) 
   {
      switch (dt) {
         case xml:
            return "xml";
         case ascii:
            return "txt";
         case binary:
            return "dat";
         case svg:
            return "svg";
         case jpeg:
            return "jpg";
         case gif:
            return "gif";
         case png:
            return "png";
         case pdf:
            return "pdf";
         case ps:
            return "ps";
         default:
            return "unknown";
      }
   }


}
