#ifndef CHANNAME_HH
#define CHANNAME_HH

#include <string>
#include <stdexcept>

/**  %ChanName parses and constructs standard channel names of the format
  *  \c Xn:\<ssid\>-\<locale\>_\<name>, where \c X is the GW detector site, 
  *  \c n is the detector or configuration number, \c \<ssid\> is the 
  *  subsystem ID, \c \<locale\>_ is an optional location specifier, and 
  *  \c \<name\> is the channel specifier.
  *  @author John Zweizig
  *  @version 1.0; Last modified: January 11, 2005
  */
class ChanName {
public:
  /**  Construct an empty channel name.
    *  @memo Default constructor.
    */
  ChanName(void);

  /**  Construct a %ChanName by parsing a channel name string. If the
    *  \a locale flag is true, the first word after the "-" is assumed 
    *  to be the locale sub-field. Otherwise the locale field is left empty.
    *  \brief Construct %ChanName from a string.
    *  \param name %Channel name string.
    *  \param locale True if a locale is specified in the string.
    */
  ChanName(const std::string& name, bool locale=true);

  /**  Construct a %ChanName from a channel name string using an existing
    *  channel name to define default fields.
    *  \brief Construct from a string and defaults.
    *  \param name %Channel name string.
    *  \param deflt Parsed channel name to supply default values for 
    *               unspecified fields.
    */
  ChanName(const std::string& name, const ChanName& deflt);

  /**  Get the IFO name in the format "Xn".
    *  \brief Get the ifo name.
    *  \return IFO name string.
    */
  const std::string& getIFO(void) const;

  /**  Get the subsystem name string.
    *  \brief Get the sub-system name.
    *  \return Sub-system name string.
    */
  const std::string& getSubSystem(void) const;

  /**  Get the channel Locale name.
    *  \brief Locale sub-field.
    *  \return Local sub-field string.
    */
  const std::string& getLocale(void) const;

  /**  Get the channel specific name field.
    *  \brief Name sub-field
    *  \return Name sub-field string.
    */
  const std::string& getName(void) const;

  /**  Get the full channel name
    *  \brief Full channel name
    *  \return channel name string.
    */
  std::string getChannel() const;

  /**  Set the IFO field of the channel name.
    *  \brief Set the ifo field.
    *  \param ifo Ifo field string.
    */
  void setIFO(const std::string& ifo) throw(std::runtime_error);

  /**  Set the sub-system field of the channel name.
    *  \brief Set the subsystem field.
    *  \param det New subsystem name string.
    */
  void setSubSystem(const std::string& det) throw(std::runtime_error);

  /**  Set the channel name locale field.
    *  \brief Set locale field
    *  \param loc Locale field string.
    */
  void setLocale(const std::string& loc) throw(std::runtime_error);

  /**  Set the channel name field.
    *  \brief Set channel name field.
    *  \param name New name string.
    */
  void setName(const std::string& name) throw(std::runtime_error);

  /**  Set the channel name. The first word of the name field will be assumed
    *  to be a locale if \a locale is true. Otherwise the locale will be 
    *  assumed to be null and the entire text after the "-" will be assigned 
    *  to the name field.
    *  \brief Set channel name
    *  \param name Locale or channel name.
    *  \param locale If true, argument string is a locale.
    */
  void setChannel(const std::string& name, bool locale=true) 
       throw(std::runtime_error);

  /**  Set the channel name, filling in any missing fields from the 
    *  Second argument %ChanName instance.
    *  \brief Set channel name with defaults.
    *  \param name Channel name string.
    *  \param deflt Default field values.
    */
  void setChannel(const std::string& name, const ChanName& deflt)
       throw(std::runtime_error);

private:
  std::string mIFO;
  std::string mSubSystem;
  std::string mLocale;
  std::string mName;
};

//======================================  Inline methods
inline const std::string& 
ChanName::getIFO(void) const {
    return mIFO;
}

inline const std::string& 
ChanName::getSubSystem(void) const {
    return mSubSystem;
}

inline const std::string& 
ChanName::getLocale(void) const {
    return mLocale;
}

inline const std::string& 
ChanName::getName(void) const {
    return mName;
}

#endif // !defined(CHANNAME_HH)
