/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "frame_name.hh"
#include <iostream>
#include <sstream>
#include <cmath>
#include <cstdio>
#include <unistd.h>
#include <sys/stat.h>

using namespace std;

//=============================================================================
//
//           Static Utility functions
//
//=============================================================================

//======================================  Print first N digits of an integer
static int
put_int(string& s, unsigned long d, int count) {
    unsigned long  dig = d%10;
    unsigned long rest = d/10;
    int     N = 0;
    if (rest) N = put_int(s, rest, count);
    if (!count || N < count) s += char('0' + dig);
    return N+1;
}

//======================================  Path function.
static string 
getPath(const std::string& outdir, const Time& start, const Time& end) {
    Time::ulong_t tStart = start.getS();
    Time::ulong_t tEnd   = end.getS();
    if (end.getN()) tEnd++;
    string s;
    string::size_type N = outdir.size();
    for (string::size_type i=0; i<N; ++i) {
	if (outdir[i] != '%') {
	    s += outdir[i];
	} else {
	    int count = 0;
	    char fmt  = outdir[++i];
	    while (fmt >= '0' && fmt < '9') {
		count *= 10;
		count += fmt - '0';
		fmt =  outdir[++i];
	    }
	    switch(fmt) {
	    case 'g':
		put_int(s, tStart, count);
		break;
	    case 'r':
		put_int(s, tStart/int(pow(10.0,count)+0.5), 0);
		break;
	    case 'd':
		put_int(s, tEnd-tStart, count);
		break;
	    default:
		s += fmt;
		break;
	    }
	}
    }
    return s;
}

//======================================  Recursive makedir
static bool
recursive_make_dir(const std::string& dir, int depth=1, int mode=0776) {
    if (dir.empty()) return true;
    if (!access(dir.c_str(), X_OK | W_OK)) return true;
    if (depth < 1) {
	cerr << "make_dir: request to create more than <depth> directories." 
	     << endl;
	return false;
    } 
    string::size_type inx=dir.find_last_of("/");
    if (inx != 0 && inx != string::npos && 
	!recursive_make_dir(dir.substr(0,inx), depth-1, mode)) return false;
    if (!mkdir(dir.c_str(), mode)) return true;
    perror("frame_name::make_dir: mkdir failed");
    return false;
}

//=============================================================================
//
//           frame_name class implementation
//
//=============================================================================

//======================================  Default constructor.
frame_name::frame_name(void)
    : _prefx("frame"), _ext("gwf"), _dt(1), _deep(2)
{}

//======================================  Default constructor.
frame_name::frame_name(const std::string& dir_pat, const std::string& pfx_pat, 
		       int dt, const std::string& ext)
  :  _dir(dir_pat), _prefx(pfx_pat), _ext(ext), _dt(dt), _deep(2)
{}

//======================================  Get the directory name
std::string
frame_name::dir_name(const Time& t) const {
    return getPath(_dir, t, t);
}

//======================================  Construct a frame file path
std::string
frame_name::file_path(const Time& t, int dt) const {
    string dirt = dir_name(t);
    ostringstream path;
    if (!dirt.empty()) path << dirt << "/";
    if (!t) {
	path << _prefx;
    } else {
	path << _prefx << "-" << t.getS() << "-";
	if (!dt) path << _dt;
	else     path << dt;
    }
    if (!_ext.empty()) path << "." << _ext;
    return path.str();
}

//======================================  make the directory.
bool
frame_name::make_dir(const std::string& dir, int deep, int mode) {
    if (!deep) deep = _deep;
    return recursive_make_dir(dir, deep, mode);
}

//======================================  Set directory pattern 
void 
frame_name::set_directory(const std::string& dir_pat) {
    _dir = dir_pat;
    if (_dir.empty()) return;
    string::size_type inx = dir_pat.size();
    while (--inx && _dir[inx] == '/') _dir.erase(inx,1);
}

//======================================  Set directory pattern
void
frame_name::set_depth(int deep) {
    _deep = deep;
}

//======================================  Set default frame length
void 
frame_name::set_dt(int dt) {
    _dt = dt;
}

//======================================  Set frame file extension
void 
frame_name::set_extension(const std::string& ext) {
    _ext = ext;
}

//======================================  Set frame file prefix
void 
frame_name::set_prefix(const std::string& pfx) {
    _prefx = pfx;
}

//======================================  Construct a temporary file path
std::string
frame_name::temp_path(const Time& t, int dt) const {
    ostringstream path;
    path << dir_name(t) << "/." << _prefx << "-" << t.getS() << "-";
    if (!dt) path << _dt;
    else     path << dt;
    path << ".tmp";
    return path.str();
}
