/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef MULTIDACC_HH
#define MULTIDACC_HH

#include "Dacc.hh"
#include "Time.hh"
#include "Interval.hh"
#include "ChannelIndex.hh"
#include "MultiList.hh"
#include <string>
#include <iosfwd>

using namespace std;

class ChanList;   // Hide the channel list from root.
class TSeries;

/**  MultiDacc methods are used to access multiple data channels from multiple
 *  frame file lists. MultiDacc has the same interfaces with Dacc. MultiDacc
 *  is basically a vector of Dacc's. 
 *  @memo LIGO Multiple stream data access class.
 *  @author  Junwei Cao
 *  @version 0.2; Last modified December 18, 2008
 */
class MultiDacc
{
public:
    ///  Size type definition
    typedef unsigned int size_type;

    ///  Define the FSeries container.
    typedef Dacc::fseries_type fseries_type;

public:
    //----------------------------------  Constructors and destructors.
    /**  Default (Null) constructor.
      *  \brief Default constructor.
      */
    MultiDacc(void);

    /**  Construct and connect to source (a list of frame file lists).
      *  \brief Construct and open.
      *  \param Source Input frame path.
      *  \param start  Start of processing time.
      */
    explicit MultiDacc(const char* Source, const Time& start=Time(0));

    /**  Destructor.
      *  \brief Destructor.
      */
    ~MultiDacc(void);

    /**  Close all the currently open files. 
      *  Close all open files.
      */
    void close(void);

    /**  Open a file for each initialized Dacc stream.
      *  \brief Open a frame file for each accessor.
      *  \return Zero on success.
      */
    int open(void);
  
    //-----------------------------------  Status manipulation
    /**  Set the DaccIn buffer count.
      *  @memo Set the DaccIn Buffer count.
      *  \param N Buffer count for shared memory input.
      */
    void setBuffer(int N);
  
    /**  If the debug flag is set, explanatory messages will be printed 
      *  when an error is encountered.
      *  @memo Set the debug flag.
      *  \param N Debug printout level.
      */
    void setDebug(int N);

    /**  Set or clear the IgnoreMissingChannel flag. If this flag is set, a 
      *  call to fillData will succede (\e i.e the return code will be 0)
      *  even if one of more channels are not found.
      *  \brief Set the "Ignore missing channel" flag.
      *  \param yn  New setting for the "Ignore missing channel" flag.
      */
    void setIgnoreMissingChannel(bool yn);

    /**  Set the noWait flag as specified. If noWait is set, the Dacc 
      *  methods that would otherwise wait for online data (e.g. synch, 
      *  fillData) will instead return with error code -8.
      *  @memo Set the noWait flag.
      *  \param now New setting for the nowait flag.
      */
    void setNoWait(bool now=true);

    /**  The default stride is used if a time interval is not specified
      *  when fillData() is invoked.
      *  @memo Define the default Stride.
      *  \param Dt Default data stride length.
      */
    void setStride(Interval Dt);

    //----------------------------------  Channel list manipulation
    /**  The specified channel is added to the list of channels of
      *  the corresponding Dacc.
      *  @memo Add a channel to the request list.
      *  @param Name  Channel name
      *  @param id    Input stream identifier.
      *  @param decim Channel decimation factor.
      *  @param TSptr Optional message pointer address 
      */
    void addChannel(const char* Name, int id=-1, int decim=0, TSeries **TSptr=0);
  
    /**  Add an process frequency series to the list. The purpose of the 
      *  method is similar to that of #addChannel# except that the 
      *  channel being searched for is a processed data frequency series.
      *  @memo Add an FSeries data channel to the request list.
      *  @param Name  Name of channel to collect data from.
      *  \param id    Input stream identifier.
      *  @param FSptr Address of an FSeries pointer.
      */
    void addFSeries(const char* Name, int id=-1, fseries_type **FSptr=0);

    /**  Add a Processed data channel to the list. The purpose of the method 
      *  and its arguments are identical to those for addChannel() except 
      *  that any search for the channel is limited to Processed data.
      *  @memo Add a processed data channel to the request list.
      *  @param Name   Name of channel to collect data from.
      *  \param id    Input stream identifier.
      *  @param decim  Decimation factor.
      *  @param TSptr  Address of TSeries pointer.
      */
    void addProcessed(const char* Name, int id=-1, int decim=0, TSeries **TSptr=0);

    /**  Add an Adc channel to the list. The purpose of the method and its
      *  arguments are identical to those for addChannel() except that any 
      *  search for the channel is limited to the raw ADC data.
      *  @memo Add a raw data channel to the request list.
      *  @param Name   Name of channel to collect data from.
      *  \param id    Input stream identifier.
      *  @param decim  Decimation factor.
      *  @param TSptr  Address of TSeries pointer.
      */
    void addRaw(const char* Name, int id=-1, int decim=0, TSeries **TSptr=0);
  
    /**  Add a Simulated data channel to the list. The purpose of the method 
      *  and its arguments are identical to those for addChannel() except 
      *  that any search for the channel is limited to Simulated data.
      *  @memo Add a simulated data channel to the request list.
      *  @param Name   Name of channel to collect data from.
      *  \param id    Input stream identifier.
      *  @param decim  Decimation factor.
      *  @param TSptr  Address of TSeries pointer.
      */
    void addSimulated(const char* Name, int id=-1, int decim=0, TSeries **TSptr=0);

    /**  The named channel is removed from the request list.
      *  @memo Remove a channel from the request list.
      *  \param Name Name of channel to be removed.
      */
    void rmChannel(const char* Name);
  
    /**  The data pointer for the frame currently in memory is incremented
      *  past the specified time. If the interval is not specified, 
      *  negative or zero, the rest of the data in the current frame are 
      *  skipped. If a discontinuity in the frame times is found, #flush#
      *  stops skipping at the start of the new data segment.
      *  @return Non-zero if an end of file is found.
      *  @memo Skip data.
      *  @param Stride Time interval to be skipped.
      */
    int flush(Interval Stride=Interval(0.0));

    /**  Get the current debug printout level.
      *  @memo Get the debug flag.
      *  @return Debug print level.
      */
    int getDebug(void);

    /**  Get the total number of frames read by the first input stream.
      *  \brief Total number of frames
      *  \return Total number of frames read.
      */
    long getTotalFrames(void) const;

    /**  Get the time of the next data to be read from the streams.
      *  \brief  Get the Current time.
      *  \return Start time for next data fill.
      */
    Time getCurrentTime(void) const;

    /**  Returns the time at which data collection started with the
      *  last invocation of fillData.
      *  @memo Get the start time of the last fill.
      *  @return Start time of last fillData() stride.
      */
    Time getFillTime(void) const;

    /**  Returns the time interval over which data are collected.
      *  \brief Get the stride interval.
      *  \return Default stride interval.
      */
     Interval getStride(void) const;

    /**  Data from channels specified by addChannel() are copied to TSeries
      *  objects.
      *  \brief  Fill the requested TSeries.
      *  \param Stride Time interval to be read in.
      *  \param start  Start a stride.
      *  \return 
      *  <table>
      *    <tr><td>0</td><td>Successful completion.</td></tr>
      *    <tr><td>-1</td><td>Frame start not contiguous to previous data.</td>
      *        </tr>
      *    <tr><td>-2</td><td>Sample rate incompatible with previous data.</td>
      *        </tr>
      *    <tr><td>-3</td><td>Requested data not found in current frame.</td>
      *        </tr>
      *    <tr><td>-4</td><td>Error reading frame.</td></tr>
      *    <tr><td>-5</td><td>Frame data are not self-consistent.</td></tr>
      *    <tr><td>-6</td><td>TSeries is not allocated.</td></tr>
      *    <tr><td>-7</td><td>Unsupported data type.</td></tr>
      *    <tr><td>-8</td><td>Signal received while reading.</td></tr>
      *    <tr><td>-9</td><td>Invalid data in structure.</td></tr>
      *  </table>
      */
    int fillData(Interval Stride=Interval(0.0), bool start=true);

    /**  Zero all TSeries in the channel list. Allocate sufficient
      *  memory to hold the given sample time.
      *  @memo   Reset channel TSeries.
      *  @param  Dt     Time interval to be used in allocating memory.
      */
    void zeroChans(Interval Dt);

    /**  Copy data to channel TSeries from a single frame.
      *  @memo   Fill the requested TSeries from a single frame.
      *  @param  Offset Offset of start of data from start of fram.
      *  @param  Dt     Time interval to be read in.
      *  \return 
      *  <table>
      *    <tr><td>0</td><td>Successful completion.</td></tr>
      *    <tr><td>-1</td><td>Frame start not contiguous to previous data.</td>
      *        </tr>
      *    <tr><td>-2</td><td>Sample rate incompatible with previous data.</td>
      *        </tr>
      *    <tr><td>-3</td><td>Requested data not found in current frame.</td>
      *        </tr>
      *    <tr><td>-4</td><td>Error reading frame.</td></tr>
      *    <tr><td>-5</td><td>Frame data are not self-consistent.</td></tr>
      *    <tr><td>-6</td><td>TSeries is not allocated.</td></tr>
      *    <tr><td>-7</td><td>Unsupported data type.</td></tr>
      *  </table>
      */
    int fillChans(Interval Offset, Interval Dt);
  
    /**  Seek the specified time. Frames are read from the listed input file(s)
      *  until one containing the specified time is found.
      *  @memo Skip to the specified time.
      *  \param STime Seek time.
      *  @return Zero if seek was successful.
      */
    int seek(Time STime=Time(0, 0));

    /**  The frame(s) that will provide the data for the next read operation 
      *  are read into memory and the current time is set. If more than one
      *  stream is being read, all streams are synchronized to the latest 
      *  time read by any of the streams.
      *  @memo Synchronize the current frame buffers.
      *  @return Zero if successful or -1 if unable to read in a frame.
      */
    int synch(void);

    /**  Test to see is the names channel has been requested..
      *  @memo Test if channel was requested.
      *  @return True if the specified channel has been requested.
      *  @param Name Channel name to be tested.
      */
    bool isChannelRead(const char* Name) const;

    /**  A formatted list of requested channels, their decimation factors, 
      *  and latest time copied is printed to the specified output stream. 
      *  \brief List requested channels.
      *  \param out STL output stream to which channels are listed.
      *  \return Output stream reference.
      */
    std::ostream& list(std::ostream& out) const;

    /**  Add multiple streams to the MultiDacc. Each line in the file 
      *  specified by \a name is the path of a file to be used to define 
      *  the input frames for a stream.
      *  \brief Add multiple input streams
      *  \param name File path for list of input streams.
      */
    void addMulti(const std::string& name);

    /**  Add a single stream to the %MultiDacc. Each line in the file 
      *  specified by \a name is the path of one or more input frames
      *  to be read by the new stream. A stream containing a single path 
      *  may be added by prefixing the frame path with an '='.
      *  \brief Add a single input stream.
      *  \param name Input frame path.
      */
    void addSingle(const std::string& name);

    /**  Get the number of Dacc input streams that will read data in
      *  parallel.
      *  \brief Number of input steams.
      *  \return Number of defined Dacc streams.
      */
    unsigned int getNDacc(void) const;

    /**  Get a pointer to the ith input stream dDacc object.
      *  \brief Get a pointer to a Dacc stream.
      *  \param i Input stream number.
      *  \return Pointer to the specified input stream.
      */
    Dacc* getDacc(unsigned int i);

    /**  Returns a pointer to the last %TSeries filled for the named channel.
      *  \brief Get a pointer to the data.
      *  \param name Channel name string pointer.
      *  \return Pointer to current data %TSeries.
      */
    TSeries* refData(const char* name);

    /**  Returns a pointer to the last frequency series filled for the named 
      *  channel.
      *  \brief Get a pointer to frequency series data.
      *  \param name Channel name string pointer.
      *  \return Pointer to current data frequency series.
      */
    fseries_type* refFData(const char* name);
  
private:
#ifndef __CINT__
    typedef std::list<ChannelIndex> ChannelList;
    typedef ChannelList::iterator chan_iter;
    typedef ChannelList::const_iterator const_chan_iter;

    const_chan_iter findChannel(const char* Name) const;
    chan_iter       findChannel(const char* Name);

    /**  List of requested channels.
      *  @memo List of requested channels.
      */
    ChannelList mChanList;

    int getDaccIndex(const char* Name);
#endif
    MultiList mList;

    ///  Frame Input object
    typedef std::vector<Dacc*> DaccVector;
    DaccVector mIn;
};

//======================================  Inline methods
#ifndef __CINT__
inline unsigned int
MultiDacc::getNDacc() const {
    return mIn.size();
}

inline Dacc*
MultiDacc::getDacc(unsigned int i) {
    return mIn[i];
}
#endif // !defined(__CINT__)

#endif // MULTIDACC_HH
