/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "FrVectRef.hh"
#include "DVecType.hh"
#include "framecpp/Dimension.hh"
#include "framecpp/FrVect.hh"

using namespace std;
using FrameCPP::Dimension;
using FrameCPP::FrVect;

//======================================  Construct an empty FrVectRef 
FrVectRef::FrVectRef(void) 
  : mOwner(false)
{
#ifndef FCPP_SHARED_PTRS
    mVect = 0;
#endif
}

//======================================  Construct an FrVectRef 
FrVectRef::FrVectRef(vect_ptr_type v) 
  : mOwner(false), mVect(v)
{}

//======================================  Construct an FrVectRef from data
FrVectRef::FrVectRef(const DVector& v, double x0, double dx, 
		     const string& unit) 
  : mOwner(false)
{
#ifndef FCPP_SHARED_PTRS
    mVect = 0;
#endif
    setVect(v, x0, dx, unit);
}

//======================================  Destroy an FrVectRef
FrVectRef::~FrVectRef(void) {
    clear();
}

//======================================  Construct a DVector from an FrVect 
DVector* 
FrVectRef::getDVector(void) const {
    if (empty()) return 0;
    return getDVector(0, mVect->GetNData());
}

//======================================  Construct a DVector from an FrVect 
DVector* 
FrVectRef::getDVector(size_type first, size_type last) const {
    DVector* r(0);
    if (empty()) return r;

    //----------------------------------  Uncompress the data
#ifndef FCPP_SHARED_PTRS
    const_cast<FrVect*>(mVect)->Uncompress();
#else
    mVect->Uncompress();
#endif

    //----------------------------------  Construct a data vector
    long len   = last - first;
    int  vtype = mVect->GetType();

#ifndef FCPP_SHARED_PTRS
    const void* dinp =mVect->GetData();
#else
    const void* dinp =mVect->GetData().get();
#endif
    switch (vtype) {
    case FrVect::FR_VECT_2S:
    case FrVect::FR_VECT_2U:
        r = new DVecType<short>(len, (const short*)dinp + first);
	break;
    case FrVect::FR_VECT_4S:
    case FrVect::FR_VECT_4U:
        r = new DVecType<int>(len, (const int*)dinp + first);
	break;
    case FrVect::FR_VECT_4R:
        r = new DVecType<float>(len, (const float*)dinp + first);
	break;
    case FrVect::FR_VECT_8R:
        r = new DVecType<double>(len, (const double*)dinp + first);
	break;
    case FrVect::FR_VECT_8C:
        r = new DVecType<fComplex>(len, (const fComplex*)dinp + first);
	break;
    default:
        r = 0;
    }
    return r;
}

//======================================  Get the number of bits in a word
FrVectRef::size_type
FrVectRef::getNBits(void) const {
    if (empty()) return 0;
    return 8 * FrVect::GetTypeSize(mVect->GetType());
}

//======================================  Get the y units
const std::string&
FrVectRef::getUnits(void) const {
    static std::string null_str;
    if (empty()) return null_str;
    return mVect->GetUnitY();
}

//======================================  Compress an FrVect
void
FrVectRef::compress(compression_mode cmode) {
    switch (cmode) {
    case kGZip:
        mVect->Compress(FrVect::GZIP, 1);
        break;
    case kGZipDifferential:
        mVect->Compress(FrVect::DIFF_GZIP, 1);
        break;
    case kZeroSuppress2:
        mVect->Compress(FrVect::ZERO_SUPPRESS_SHORT, 1);
        break;
    case kZeroSuppress4:
        mVect->Compress(FrVect::ZERO_SUPPRESS_INT_FLOAT, 1);
        break;
    case kZeroSuppOrGZip:
        mVect->Compress(FrVect::ZERO_SUPPRESS_OTHERWISE_GZIP, 1);
        break;
    default:
        break;
    }
}

//======================================  Replicate an FrVect
FrVectRef::vect_ptr_type
FrVectRef::replicate(void) const {
    if (null()) return vect_ptr_type();
    return vect_ptr_type(new vect_type(*mVect));
}

//======================================  Set reference to a specified FrVect
void
FrVectRef::setName(const string& name) {
    if (!mVect || !mOwner) return;
#if LDAS_VERSION_NUMBER < 112001
    const_cast<std::string&>(mVect->GetName()) = name;
#else
    mVect->SetName(name);
#endif
}

//======================================  Set reference to a specified FrVect
void
FrVectRef::setUnits(const string& unit) {
    if (!mVect || !mOwner) return;
    mVect->SetUnitY(unit);
}

//======================================  Set reference to a specified FrVect
void
FrVectRef::setVect(vect_ptr_type v, bool owner) {
#ifndef FCPP_SHARED_PTRS
    if (mVect && mOwner) delete mVect;
#endif
    mVect = v;
    mOwner = owner;
}

//======================================  Set an FrVectRef from data
void
FrVectRef::setVect(const DVector& v, double x0, double dx, 
		   const string& unit) {

    union multiptr {
	const void*       v;
	const short*      si;
	const int*        i;
	const long*       li;
	const float*      f;
	const double*     d;
	const COMPLEX_8*  c8;
	const COMPLEX_16* c16;
    } pData;

    //----------------------------------  Get the length and pointer.
    size_type len = v.getLength();
    if (!len)   return;
    pData.v = v.refData();

    //----------------------------------  Define the dimensionality
    Dimension dim(len, dx, unit, x0);

    //----------------------------------  Copy the data
    vect_type* pVect(0);
    switch (v.getType()) {
    case DVector::t_short:
        pVect = new FrVect("data", 1, &dim, pData.si);
        break;
	//------------------------------  Constructor not implemented in 
	//                                framecpp-1.7.0
	//case DVector::t_long:
        //mVect = new FrVect("data", 1, &dim, pData.li);
        //break;
    case DVector::t_int:
        pVect = new FrVect("data", 1, &dim, pData.i);
        break;
    case DVector::t_float:
        pVect = new FrVect("data", 1, &dim, pData.f);
        break;
    case DVector::t_double:
        pVect = new FrVect("data", 1, &dim, pData.d);
        break;
    case DVector::t_complex:
        pVect = new FrVect("data", 1, &dim, pData.c8);
        break;
    case DVector::t_dcomplex:
        pVect = new FrVect("data", 1, &dim, pData.c16);
        break;
    default:
        break;
    }
    setVect(vect_ptr_type(pVect) ,true);
}

//======================================  Get an index into the data.
FrVectRef::size_type 
FrVectRef::getIndex(size_type dim, double x0) const {
    if (empty() || dim >= mVect->GetNDim()) return 0;
    Dimension d=mVect->GetDim(dim);
    double vx0 = d.GetStartX();
    if (x0 < vx0) return 0;
    return size_type((x0 - vx0)/d.GetDx() + 0.5);
}

//======================================  Copy vector to a DVector
DVector* 
FrVectRef::getDVector(size_type dim, double x0, double dx) const {
    return getDVector(getIndex(dim, x0), getIndex(dim, x0+dx));
}

//======================================  Get the ith dimension start value
double   
FrVectRef::getDimX0(size_type i) const {
    if (empty() || i > mVect->GetNDim()) return 0;
    return mVect->GetDim(i).GetStartX();
}

//======================================  Get the ith dimension end value
double   
FrVectRef::getDimXmax(size_type i) const {
    if (empty() || i > mVect->GetNDim()) return 0;
    FrameCPP::Dimension d=mVect->GetDim(i);
    return d.GetStartX() + d.GetDx() * d.GetNx();
}

//======================================  Get the ith dimension increment
double   
FrVectRef::getDimDx(size_type i) const {
    if (empty() || i > mVect->GetNDim()) return 0;
    return mVect->GetDim(i).GetDx();
}

//======================================  Get the ith dimension length
FrVectRef::size_type 
FrVectRef::getDimNx(size_type  i) const {
    if (empty() || i > mVect->GetNDim()) return 0;
    return mVect->GetDim(i).GetNx();
}

//======================================  Get the FrVect length
FrVectRef::size_type 
FrVectRef::size(void) const {
    if (null()) return 0;
    return mVect->GetNData();
}
