/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef CHANNEL_HH
#define CHANNEL_HH

#include "Time.hh"
#include "Interval.hh"
#include <string>

#include "FramePointerTypes.hh"

class TSeries;
class FSeries;
class FrVectRef;

/**  Channel holds the information about the time series being accumulated 
  *  by Dacc.
  *  \brief Channel list entry for DMT data accessor.
  *  \author John Zweizig
  *  \version 1.2; Last modified: September 18, 2009
  */
class Channel {
public:
    ///  Unsigned integer type definition
    typedef unsigned int uint_type;

    ///  Size type definition
    typedef unsigned long size_type;

    /**  The channel type enumerator specifies where the channel class will
      *  look for the requested channel name. If the channel type is not 
      *  specified explicitly, it is set to \c kUnknown. In this case, the
      *  read methods (\e i.e. FillSeries()) will search for the named channel 
      *  in the raw data followed by the processed data.
      *  @memo Channel type enumerator.
      */
    enum ChanType {
	kUnknown,   ///< Unknown channel type
	kRaw,       ///< Raw data (FrAdcData) channel.
	kProcessed, ///< Processed data channel
	kSimulated, ///< Simulated data channel
	kFSeries    ///< Frequency series channel.
    };

    /** Enumerate invalid data test flag numbers.
      */ 
    enum ChanFlags {
	kAllowNonNormal, ///< Allow non-normal data flag.
	kAllowNotValid   ///< Allow invalid (inf or NaN) data flag.
    };

//--------------------------------------  Constructors/Destructors
public:
    /**  Construct an empty Channel instance.
      *  \brief Default Constructor.
      */
    Channel(void);

    /**  Construct an object given a channel name, a decimation factor, 
      *  an output pointer address and a channel type.
      *  \brief Construct a channel object.
      *  \param name   Channel name string
      *  \param ctlvar Address of pointer to receive TSeries address.
      *  \param decim  Decimation factor.
      *  \param t      Channel type.
      */
    Channel(const std::string& name, TSeries** ctlvar, uint_type decim=0, 
	    ChanType t=kUnknown);

    /**  Construct an object given a channel name, a decimation factor, 
      *  an output pointer address and a channel type.
      *  \brief Construct a channel object.
      *  \param name   Channel name string
      *  \param ctlvar Address of pointer to receive FSeries address.
      */
    Channel(const std::string& name, FSeries** ctlvar);

    /**  Construct a channel object identical to an already existing one.
      *  \brief Copy constructor.
      *  \param data Channel instance to be copied
      */
    Channel(const Channel& data);

    /**  Copy a channel object to an existing object.
      *  \brief Assignment operator.
      *  \param data Channel instance to be copied
      *  \return Reference to modified instance.
      */
    Channel& operator=(const Channel& data);

    /**  Destroy a channel entry.
      *  \brief Destructor
      */
    ~Channel(void);

    //------------------------------------  Accessors
    /**  EqName() returns true if the channel name is the same as the argument.
      *  \brief Compare the name.
      *  \param name Channel name string
      *  \return True if the channel name matches the string.
      */
    bool EqName(const std::string& name) const;

    /**  Equal returns true if the channel name is the same as the argument 
      *  and the channel type is the same, or the requested type is unknown.
      *  \brief Compare the name and type.
      *  \param name Channel name string to be tested
      *  \param t    Channel type to be tested.
      *  \return True if the channel matches the specifications.
      */
    bool Equal(const std::string& name, ChanType t=kUnknown) const;

    /**  getDebug() returns the debug flag. When the debug flag is set, 
      *  Channel methods print messages to cerr any time an error occurs.
      *  \brief Get the debug flag.
      *  \return Debugging flag.
      */
    bool getDebug(void) const;

    /**  getDecim() returns the decimation factor.
      *  \brief Get the decimation factor.
      *  \return Channel decimation factor.
      */
    uint_type getDecim(void) const;
    
    /**  getLast() returns the time of the last sample added to the channel 
      *  time series.
      *  \brief Get the end time.
      *  \return Time of last sample processed.
      */
    Time getLast(void) const;

    /**  getName() returns a pointer to the channel name string.
      *  \brief Get the channel name
      *  \return Pointer to channel name string.
      */
    const char* getName(void) const;

    /**  The channel type is returned. The type can be \c kUnknown if the 
      *  type has not been specified and no data from this channel has been 
      *  read, \c kRaw if the channel is a raw (Adc) data channel or 
      *  \c kProcessed if the data are in a process data channel.
      *  \brief Get the channel type.
      *  \return Channel type.
      */
    ChanType getType(void) const;

    //------------------------------------  Mutators
    /**  setAccumulate() sets/clears the accumulate mode for the channel.
      *  If accumulate mode is set, the data will be appended to the TSeries. 
      *  By default the TSeries data are reset with each call to FillData().
      *  \brief Set channel accumulation mode.
      *  \param Acc New value for accumulation mode flag.
      */
    void setAccumulate(bool Acc=true) {mAccumulate = Acc;}

    /**  The debug flag is set as specified.
      *  \brief Set or clear the debug flag.
      *  \param yorn New value for debug flag.
      */
    void setDebug(bool yorn) {mDebug = yorn;}

    /**  The specified flags is set or cleared.
      *  \brief Set/clear a processing flag.
      *  \param flag Enumerated flag number
      *  \param state New value for specified flag.
      */
    void setFlag(ChanFlags flag, bool state=true);

    //------------------------------------  Crunchers
    /**  The user TSeries data vector is expanded to accommodate data over
      *  a specified time interval.
      *  \brief Allocate an output TSeries with a specified length.
      *  \param Stride Length of data to allocate.
      */
    void allocate(Interval Stride);

#ifndef __CINT__
    /**  Data from the specified frame are (optionally) decimated and copied 
      *  into the data vector(s).
      *  \brief Fill time series from a frame in memory. 
      *  \param frame Pointer to frame containing data to be read.
      *  \param off   Time offset to start of data.
      *  \param dT    Time interval to be read in.
      *  \return
      *  <table>
      *    <tr><td>0</td><td>Successful completion.</td></tr>
      *    <tr><td>-1</td><td>Start not contiguous to previous data.</td></tr>
      *    <tr><td>-2</td><td>Sample rate incompatible with previous data.</td>
      *        </tr>
      *    <tr><td>-3</td><td>Requested data not found in frame.</td></tr>
      *    <tr><td>-5</td><td>Frame data are not self-consistent.</td></tr>
      *    <tr><td>-6</td><td>TSeries is not allocated.</td></tr>
      *    <tr><td>-7</td><td>Unsupported data type.</td></tr>
      *  </table>
      */
    int  FillSeries(frameh_pointer frame, Interval off, Interval dT);

    /**  Data from the specified adc block are (optionally) decimated and 
      *  copied into the data vector(s).
      *  \brief Fill time series from an adcData object.
      *  \param adc   Pointer to FrAdcData containing data.
      *  \param t0    Start time for data unpacking.
      *  \param off   Time offset to start of data.
      *  \param dT    Time interval to be read in.
      *  \return
      *  <table>
      *    <tr><td>0</td><td>Successful completion.</td></tr>
      *    <tr><td>-1</td><td>Start not contiguous to previous data.</td></tr>
      *    <tr><td>-2</td><td>Sample rate incompatible with previous data.</td>
      *        </tr>
      *    <tr><td>-3</td><td>Requested data not found in frame.</td></tr>
      *    <tr><td>-5</td><td>Frame data are not self-consistent.</td></tr>
      *    <tr><td>-6</td><td>TSeries is not allocated.</td></tr>
      *    <tr><td>-7</td><td>Unsupported data type.</td></tr>
      *  </table>
      */
    int  FillSeries(fradcdata_pointer adc, const Time& t0, 
		    Interval off, Interval dT);

    /**  Data from the specified FrProcData block are (optionally) decimated 
      *  and copied into the data vector(s).
      *  \brief Fill time series from an FrProcData object.
      *  \param proc  Pointer to FrProcData containing data.
      *  \param t0    Start time for data unpacking.
      *  \param off   Time offset to start of data.
      *  \param dT    Time interval to be read in.
      *  \return
      *  <table>
      *    <tr><td>0</td><td>Successful completion.</td></tr>
      *    <tr><td>-1</td><td>Start not contiguous to previous data.</td></tr>
      *    <tr><td>-2</td><td>Sample rate incompatible with previous data.</td>
      *        </tr>
      *    <tr><td>-3</td><td>Requested data not found in frame.</td></tr>
      *    <tr><td>-5</td><td>Frame data are not self-consistent.</td></tr>
      *    <tr><td>-6</td><td>TSeries is not allocated.</td></tr>
      *    <tr><td>-7</td><td>Unsupported data type.</td></tr>
      *  </table>
      */
    int  FillSeries(frprocdata_pointer proc, const Time& t0, 
		    Interval off, Interval dT);

    /**  Data from the specified FrSimData block are (optionally) decimated 
      *  and copied into the data vector(s).
      *  \brief Fill time series from an FrSimData object.
      *  \param proc  Pointer to FrSimData containing data.
      *  \param t0    Start time for data unpacking.
      *  \param off   Time offset to start of data.
      *  \param dT    Time interval to be read in.
      *  \return
      *  <table>
      *    <tr><td>0</td><td>Successful completion.</td></tr>
      *    <tr><td>-1</td><td>Start not contiguous to previous data.</td></tr>
      *    <tr><td>-2</td><td>Sample rate incompatible with previous data.</td>
      *        </tr>
      *    <tr><td>-3</td><td>Requested data not found in frame.</td></tr>
      *    <tr><td>-5</td><td>Frame data are not self-consistent.</td></tr>
      *    <tr><td>-6</td><td>TSeries is not allocated.</td></tr>
      *    <tr><td>-7</td><td>Unsupported data type.</td></tr>
      *  </table>
      */
    int  FillSeries(frsimdata_pointer proc, const Time& t0, 
		    Interval off, Interval dT);
#endif

    /**  Return a constant pointer to the time series with the current data 
      *  for this channel.
      *  \brief Reference the channel data.
      *  \return Constant pointer to time series.
      */
    const TSeries* refSeries(void) const {return (mUserPtr)? *mUserPtr : 0;}

    /**   Return a constant pointer to the time series with the current data 
      *  for this channel.
      *  \brief Reference the TSeries.
      *  \return Pointer to time series.
      */
    TSeries* refSeries(void) {return (mUserPtr)? *mUserPtr : 0;}

    /**  Return a pointer to the frequency series filled by the last 
      *  call to FillSeries.
      *  \brief Reference the FSeries.
      *  \return Constant pointer to frequency series containing current data.
      */
    const FSeries* refFSeries(void) const {return (mFUserPtr)? *mFUserPtr : 0;}

    /**  Return a pointer to the frequency series filled by the last 
      *  call to FillSeries.
      *  \brief Reference the FSeries.
      *  \return Pointer to frequency series containing current data.
      */
    FSeries* refFSeries(void) {return (mFUserPtr)? *mFUserPtr : 0;}

    /**  Sufficient space is reserved in the target time series to accommodate 
      *  the specified interval. No action is taken if the sample rate is 
      *  unknown (zero) or if the target series is a frequency series.
      *  \brief resize the target series to accommodate the specified interval.
      *  \param dT Time interval for which space is to be reserved.
      */
    void reserve(Interval dT);

private:                                // Private functions.

    /**  Data from the specified FrVect block are (optionally) decimated 
      *  and copied into the data vector(s).
      *  \brief Fill time series from an FrVect object.
      *  \param vec Pointer to FrVect containing data.
      *  \param t0  Start time for data unpacking.
      *  \param off Time offset to start of data.
      *  \param dT  Time interval to be read in.
      *  \param fix Fix the data for readout errors.
      *  <table>
      *     <tr><td>0</td><td>No transformation is necessary</td></tr>
      *     <tr><td>1</td><td>Swap words to fix S2 data error.</td></tr>
      *  </table>
      *  \return
      *  <table>
      *    <tr><td>0</td><td>Successful completion.</td></tr>
      *    <tr><td>-1</td><td>Start not contiguous to previous data.</td></tr>
      *    <tr><td>-2</td><td>Sample rate incompatible with previous data.</td>
      *        </tr>
      *    <tr><td>-3</td><td>Requested data not found in frame.</td></tr>
      *    <tr><td>-5</td><td>Frame data are not self-consistent.</td></tr>
      *    <tr><td>-6</td><td>TSeries is not allocated.</td></tr>
      *    <tr><td>-7</td><td>Unsupported data type.</td></tr>
      *  </table>
      */
    int  FillSeries(const FrVectRef& vec, const Time& t0, 
		    Interval off, Interval dT, int fix=0);

    /**  Data from the specified FrVect block are copied into the data vector.
      *  \brief Fill frequency series from an FrVect object.
      *  \param vec Pointer to FrVect containing data.
      *  \param t0  Start time for data unpacking.
      *  \param dT  Time interval to be read in.
      *  \return
      *  <table>
      *    <tr><td>0</td><td>Successful completion.</td></tr>
      *    <tr><td>-1</td><td>Start not contiguous to previous data.</td></tr>
      *    <tr><td>-2</td><td>Sample rate incompatible with previous data.</td>
      *        </tr>
      *    <tr><td>-3</td><td>Requested data not found in frame.</td></tr>
      *    <tr><td>-5</td><td>Frame data are not self-consistent.</td></tr>
      *    <tr><td>-6</td><td>TSeries is not allocated.</td></tr>
      *    <tr><td>-7</td><td>Unsupported data type.</td></tr>
      *  </table>
      */
    int  FillFSeries(const FrVectRef& vec, const Time& t0, Interval dT);

    /**  Delete a locally managed series.
      *  \brief Delete a locally managed series.
      */
    void deleteSeries(void);

    /**  Compare the specified interval to the current sample interval.
      *  set mSample if it isn't set and return an error indicator.
      *  \brief Set the sample interval.
      */
    int setSample(const FrVectRef& vec);

    /**  Compare the specified interval to the current sample interval.
      *  set mSample if it isn't set and return an error indicator.
      *  \brief Set the sample interval.
      */
    int setSample(Interval dT);

    /**  Test a channel processing flag.
      */
    bool testFlag(ChanFlags flg) const;

private:                                // Private data members.
    /**  The name of the channel to be processed.
      *  \brief Channel Name.
      */
    std::string mName;

    /**  Decim8 contains the number of raw samples that must be averaged 
      *  together for a single TSeries entry.
      *  \brief Decimation Factor.
      */
    uint_type mDecim8;

    /**  If mAccumulate is true, each call to FillSeries results in the 
      *  new data being appended to the those data that are already in the
      *  target TSeries. If it is false, the new data read by fillSeries 
      *  overwrite the previous contents of the TSeries.
      *  \brief Accumulation mode flag.
      */
    bool mAccumulate;

    /**  If set, copious error and status messages are printed to cerr.
      *  \brief Debug flag.
      */
    bool mDebug;

    /**  mSample contains sample time (1/<sample-frequency>) of the channel
      *  in the input frame data.
      *  \brief Raw Sample time.
      */
    Interval mSample;

    /**  The absolute (GPS) time of the end of the time bin containing the 
      *  last sample processed.
      *  \brief Last sample time.
      */
    Time mLast;

    /**  The number of samples that have been added to mAccVal (in the range 
      *  0 to mDecim8-1)
      *  \brief Number of samples added to mAccVal.
      */
    uint_type mNSample;

    /**  The sum of the last mNSample digitizations.
      *  \brief Current accumulator.
      */
    double mAccVal;

    /**  Address of the users pointer to the Time series to be filled.
      *  \brief User TSeries pointer address.
      */
    TSeries** mUserPtr;

    /**  Address of the users pointer to the Time series to be filled.
      *  \brief User FSeries pointer address.
      */
    FSeries** mFUserPtr;

    /**  Local pointer used if the user doesn't specify one.
      *  \brief Local pointer.
      */
    union {
	FSeries* fsp;
	TSeries* tsp;
    } myPtr;

    /**  Flag indicating type of channel (Raw or processed).
      *  \brief Channel type
      */
    ChanType  mType;

    /**  Flags to indicate optional processing for channel.
      *  \brief Processing flags
      */
    uint_type mChanFlags;
};

//======================================  Inline methods.
inline bool 
Channel::EqName(const std::string& name) const {
    return (mName == name);
}

inline bool 
Channel::Equal(const std::string& name, ChanType t) const {
    return (mName == name && (t == kUnknown || t == mType));
}

inline bool 
Channel::getDebug(void) const {
    return mDebug;
}

inline Channel::uint_type 
Channel::getDecim(void) const {
    return mDecim8;
}

inline Time 
Channel::getLast(void) const {
    return mLast;
}

inline const char* 
Channel::getName(void) const {
    return mName.c_str();
}

inline Channel::ChanType
Channel::getType(void) const {
    return mType;
}

inline bool 
Channel::testFlag(ChanFlags flg) const {
    return (mChanFlags & (1 << int(flg))) != 0;
}

#endif // CHANNEL_HH
