/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "PConfig.h"
#include <unistd.h>
#include "TrigMgr.hh"
#include "TrigMsg.hh"
#include "Time.hh"
#include "ParseLine.hh"
#include "html/color.hh"
#include "html/document.hh"
#include "html/size.hh"
#include "html/text.hh"
#include "html/writer.hh"
#include "html/Table2C.hh"
#include "lmsg/ErrorList.hh"
#include <cmath>
#include <float.h>
#ifdef sun
#include <ieeefp.h>
inline int isfinite(double x) {return finite(x);}
#elif defined (P__WIN32)
inline int isfinite(double x) {return finite(x);}
#endif
#include <fstream>
#include "PConfig.h"
#ifdef __GNU_STDC_OLD
#include <gnusstream.h>
#else
#include <sstream>
#endif
#include <sys/time.h>
#include <cstdlib>

using namespace std;
using namespace trig;
using namespace lmsg;

static const char* TrigMgrVsn = "$Header: https://redoubt.ligo-wa.caltech.edu/svn/gds/trunk/Services/TrigMgr/TrigMgr.cc 7362 2015-05-01 00:34:20Z john.zweizig@LIGO.ORG $";

static const unsigned int idLength = 26;

namespace lmsg {
//
//    Trigger Manager message handlers.
//
//    All trigger manager messages are passed on to the appropriate handler
//    method from the TrigMgr class. Note that handlers assume that the 
//    server application is the trigger manager.
//
template<>
error_type
Handler<TMM_Register>::handleMsg(AppServer& app, const MsgHeader& hdr) {
    return ((TrigMgr&) app).HandleRegister(hdr, mMessage);
}

template<>
error_type
Handler<TMM_Segment>::handleMsg(AppServer& app, const MsgHeader& hdr) {
    return ((TrigMgr&) app).HandleSegment(hdr, mMessage);
}

template<>
error_type
Handler<TMM_Trigger>::handleMsg(AppServer& app, const MsgHeader& hdr) {
    return ((TrigMgr&) app).HandleTrigger(hdr, mMessage);
}

template<>
error_type
Handler<TMM_Close>::handleMsg(AppServer& app, const MsgHeader& hdr) {
    return ((TrigMgr&) app).HandleClose(hdr, mMessage);
}

template<>
error_type
Handler<TMM_RemNode>::handleMsg(AppServer& app, const MsgHeader& hdr) {
    return ((TrigMgr&) app).HandleRmNode(hdr, mMessage);
}

template<>
error_type
Handler<TMM_Reconfigure>::handleMsg(AppServer& app, const MsgHeader& hdr) {
    return ((TrigMgr&) app).HandleReconfig(hdr, mMessage);
}

template class Handler< TMM_Register >;
template class Handler< TMM_Trigger >;
template class Handler< TMM_Close >;
template class Handler< TMM_Segment >;
template class Handler< TMM_RemNode >;
template class Handler< TMM_Reconfigure >;

}

//
//    Main function of the trigger manager. Construct it and run it.
//
int
main(int argc, const char* argv[]) {
    TrigMgr x(argc, argv);
    return x.Application();
}

bool
test_float(const double& x) {
    if (! isfinite(x)) return false;
    if (x == 0) return true;
    double ax = fabs(x);
    if (ax >= FLT_MAX) return false;
    if (ax <= FLT_MIN) return false;
    return true;
}

//
//    Trigger and segment validity checks
//
bool 
TrigValid(const TrigBase& t) {
    const char* pID = t.getID();
    bool rc = pID && *pID;
    if (!rc) return false;

    rc = t.getTime() != Time(0) && (double(t.getDt()) >= 0.0);
    if (!rc) return rc;

    rc = test_float(double(t.getDt())) &&
         test_float(t.getIntensity()) &&
         test_float(t.getSignificance()) && 
         test_float(t.getFrequency()) && 
         test_float(t.getBandwidth()) && 
         test_float(t.getFreqPeak()) && 
         test_float(t.getFreqAvg()) && 
         test_float(t.getFreqSigma()) && 
         test_float(t.getNoisePower()) && 
         test_float(t.getSignalPower()) && 
         test_float(t.getConfidence());
    return rc;
}

//======================================  Check that the segment is valid
bool 
SegValid(const Segment& s) {
    //----------------------------------  Check for non-null group name
    if (!s.getGroup() || !*s.getGroup()) return false; 

    //----------------------------------  Check for non-zero start time
    if (!s.getStartTime()) return false;

    //----------------------------------  Check for positive, non-zero duration
    if (s.getStartTime() >= s.getEndTime()) return false;

    //----------------------------------  Check for positive version 
    if (s.getVersion() < 0) return false; 

    //----------------------------------  Comment must not contain quotes or
    //                                    newline characters.
    string comstr = s.getComment();
    if (comstr.find_first_of("\"\n\r") != string::npos) return false;
    return true;
}

//======================================  Check that the process is valid
bool
ProcValid(const TrigProc& p) {
    return true;
}

//======================================  Trigger manager constructor.
//
//    The only constructor passes the command line argument list as 
//    arguments in the usual form. The constructor first parses the 
//    command line and then sets up message and signal handling.
//
TrigMgr::TrigMgr(int argc, const char* argv[]) 
  : AppServer(MgrName, o_register, p_Server), mEnableDB(true), 
    mEnableEpX(true), mTimer(300.0), mNotify("{}"), 
    mTerm(SIGTERM), mReconfigure(SIGHUP), mTrigLog("Trigger Log")
{
    cout << "Starting trigger manager version: " << TrigMgrVsn << endl;

    //----------------------------------  Set default values
    if (getenv("DMTPARS")) {
	mConfigFile = getenv("DMTPARS");
    } else {
	mConfigFile   = getenv("HOME");
	mConfigFile  += "/pars";
    }
    mConfigFile  += "/TrigMgr.conf";
    mDefParam._TableCount = 1000;
    mDefParam.setMode(TrigStream::param::kCount, 120, 300);

    //----------------------------------  Parse arguments
    for (int i=1 ; i<argc ; i++) {
        string argi(argv[i]);
        if (argi == "-ofile") {
	    mOutDir = argv[++i];
	} else if (argi == "-noDB") {
	    cout << "Warning: -noDB option selected!" << endl;
	    cout << "Triggers will not be written to the meta-data base"
		 << endl;
	    mEnableDB = false;
	} else if (argi == "-noEpics") {
	    cout << "Warning: -noEpics option selected!" << endl;
	    cout << "Alarms will not be sent to Epics"	 << endl;
	    mEnableEpX = false;
	} else if (argi == "-access") {
	    mConfigFile = argv[++i];
	} else if (argi == "-notify") {
	    mNotify = argv[++i];
	} else if (argi == "-debug") {
	    if (i >= argc-1)            setDebug(1);
	    else if (*argv[i+1] == '-') setDebug(1);
	    else                        setDebug(strtol(argv[++i],0,0));
	} else if (argi == "-ntrig") {
	    mDefParam._TableCount = strtol(argv[++i],0,0);
	} else if (argi == "-streams") {
	    mStreamConf = argv[++i];
	} else if (argi == "-timeout") {
	    mTimer = strtod(argv[++i],0);
	} else if (argi == "-idletime") {
	    mDefParam._Inactive = strtod(argv[++i],0);
	} else if (argi == "-maxtime") {
	    mDefParam._MaxDelay = strtod(argv[++i],0);
	} else {
	    cerr << "Unrecognized argument: " << argv[i] << endl;
	    cerr << "Command syntax is as follows:" << endl;
	    cerr << argv[0] << " [-access <list>] [-ofile <trigger-dir>] " 
		 << "[-ntrig <in-file>]" << endl;
	    cerr << "        [-notify <email-addr>] [-debug [<level>]] "
		 << "[-noDB] [-noEpics]" << endl;
	    cerr << "        [-timeout <max-time>] [-streams <str-file>]" 
		 << endl;
	    mTerm.setSig0();
	    return;
	}
    }

    mCTable.readFile(mConfigFile);
    mCTable.reset();

    if (mOutDir.empty()) {
        mOutDir = getenv("HOME");
	mOutDir += "/triggers";
    }
    //chdir(mOutDir.c_str());

    //----------------------------------  Trigger log file name
    const char* htmlDir = getenv("DMTHTMLOUT");
    if (!htmlDir) htmlDir = ".";
    mTrigLog.setFile(string(htmlDir) + "/TriggerLog.html");

    //----------------------------------  Define message handlers
    addHandler(tmm_Register, new Handler<TMM_Register>);
    addHandler(tmm_Segment,  new Handler<TMM_Segment>);
    addHandler(tmm_Trigger,  new Handler<TMM_Trigger>);
    addHandler(tmm_Close,    new Handler<TMM_Close>);
    addHandler(tmm_rmNode,   new Handler<TMM_RemNode>);

    //----------------------------------  Set the output streams.
    if (configStreams(mStreamConf)) {
	mTerm.setSig0();
	return;
    }

    //----------------------------------  Define Signal handlers
    mTerm.add(SIGINT);
    mForce.add(SIGALRM);

    //----------------------------------  Start the timer
    setTimer(mTimer);
}

//======================================  Trigger manager destructor
TrigMgr::~TrigMgr(void) {
    close();
}

//======================================  Set the timer
void 
TrigMgr::setTimer(Interval dT) {
    itimerval time;
    long nsec = long(dT);
    time.it_interval.tv_sec  = 0;
    time.it_interval.tv_usec = 0;
    time.it_value.tv_sec  = nsec;
    time.it_value.tv_usec = long((double(dT) - nsec)*1000000);
    setitimer(ITIMER_REAL, &time, 0);
}

//======================================  Main trigger manager function
int 
TrigMgr::Application(void) {
    while (!mTerm) {

        //------------------------------  Wait for/handle a message
        try {
	    if (getDebug() > 1) cout << "Wait for message..." << endl;
	    int rc = handleMessage();
	    if (rc != OK && rc != Continue) {
	        cerr << "Error " << rc << " in handleMessage()" << endl;
	    } else if (getDebug() > 1) {
	        cout << "Message received" << endl;
	    }
	} catch (exception& e) {
	    cerr << "Exception handling message: " << e.what() << endl;
	}

	//------------------------------  Handle watchdog if triggered
	if (mReconfigure) {
	    mReconfigure.clear();
	    mCTable.readFile(mConfigFile, false);
	    closeStreams();
	    configStreams(mStreamConf);
	}

	//------------------------------  Handle watchdog if triggered
	if (mForce) {
	    bool timer = mForce.test(SIGALRM);
	    mForce.clear();
	    if (timer) setTimer(mTimer);
	    writeStats();
	    writeSegStats();
	    if (timer) mCTable.epoch();
	    else       mCTable.reset();	    
	}

	//------------------------------  Reset output timers if empty
	Time now = Now();
        int N=mStreamList.size();
	for (int i=0; i<N; i++) mStreamList[i]->tick(now);

	WriteDB(false);
	mTrigLog.writeLog();
    }
    writeStats();
    writeSegStats();
    WriteDB(true);
    return 0;
}

//======================================  Close output streams.
void
TrigMgr::closeStreams(void) {
    for (stream_iter i=mStreamList.begin(); i != mStreamList.end(); ++i) {
	TrigStream* pStream = *i;
	*i = 0;                   // j.i.c
	pStream->WriteDB(true);
	delete pStream;
    }
    mStreamList.clear();
}

//======================================  Configure output streams.
bool
TrigMgr::configStreams(const string& file) {
    //----------------------------------  Set up default streams.
    if (file.empty()) {
	cout << "Configuring default streams" << endl;
	mStreamList.push_back(new LdasStream("prod", 0x0001));
	mStreamList.push_back(new LdasStream("test", 0x0002));
	mStreamList.back()->setExcludeMsk(0x0001);
	if (mEnableEpX) mStreamList.push_back(new EpxStream("epics", 0x0004));
	mStreamList.push_back(new SegDBStream("seg", 0x0008));

	//------------------------------  Set default stream params.
	int N = mStreamList.size();
	for (int i=0; i < N; ++i) {
	    mStreamList[i]->setParam(mDefParam);
	}

    //----------------------------------  Configure streams from file.
    } else {
	ParseLine pl(file.c_str());
	if (!pl.isOpen()) {
	    cerr << "Unable to open file: " << file << endl;
	    return true;
	}
	cout << "Configuring streams defined in: " << file << endl;

	unsigned int mask = 1;
	while (pl.getLine() >= 0) {
	    long nArgs = pl.getCount();
	    if (! nArgs) continue;
	    string name = pl[0];
	    string type = "LDAS";
	    string out  = "TrigMgr%u.txt";
	    string script;
	    TrigStream::param pars = mDefParam;
	    for (int i=1; i<nArgs; ++i) {
		string argi = pl[i];
		if (argi == "-idletime") {
		    pars._Inactive = pl.getDouble(++i);
		} else if (argi == "-insert") {
		    pars._Insert = pl[++i];
		} else if (argi == "-mask") {
		    mask = strtoul(pl[++i], 0, 0);
		} else if (argi == "-maxtime") {
		    pars._MaxDelay = pl.getDouble(++i);
		} else if (argi == "-mode") {
		    argi = pl[++i];
		    if (argi == "Count") {
			pars._Mode = TrigStream::param::kCount;
		    } else if (argi == "Timer") {
			pars._Mode = TrigStream::param::kTimer;
		    } else {
			cout << "Undefined parameter: " << argi << endl;
			return true;
		    }
		} else if (argi == "-out") {
		    out = pl[++i];
		} else if (argi == "-type") {
		    type = pl[++i];
		} else {
		    cout << "Unrecognized parameter: " << argi << endl;
		    return true;
		}
	    }

	    //--------------------------  Add the defined stream
	    TrigStream* p = 0;
	    if (type == "LDAS") {
		p = new LdasStream(name, mask);
	    } else if (type == "Segment") {
		p = new SegDBStream(name, mask);
	    } else if (type == "Epics") {
		p = new EpxStream(name, mask);
	    } else {
		cout << "Undefined stream type: " << type << endl;
		return true;
	    }

	    //--------------------------  select the next mask bit
	    pars._OutDir = out;
	    if (pars._Insert.empty()) pars._Insert = p->refParams()._Insert;
	    p->setParam(pars);
	    mStreamList.push_back(p);
	    cout << "Add stream: " << name << " type: " << type << " mask: "
		 << mask << " out: " << pars._OutDir << endl;
	    mask += mask;
	}
    }
    return false;
}

//======================================  Handle a registration request
error_type
TrigMgr::HandleRegister(const MsgHeader& hdr, const TMM_Register& msg) {

    //----------------------------------  See if process is already registered
    for (const_proc_iter i=mProcDB.begin() ; i != mProcDB.end() ; i++) {
        if (i->second.refData() == msg.refValue()) {
	    cout << "Process Name: " << msg.refValue().getName() 
		 << " ID: " << i->first << " already registered." << endl;
	    return reply(hdr, TMM_ProcID(i->first));
	}
    }

    //----------------------------------  See if the ID is already assigned
    if (*(msg.refValue().getProcessID())) {
        string ident = msg.refValue().getProcessID();
	if (ident.size() == idLength && 
	    ident.find_first_not_of("0123456789") == ident.npos) {
	    mProcDB[ident] = msg.refValue();
	    cout << "Process " << msg.refValue().getName() << " ID: " 
		 << ident << " registered with existing ID." << endl;
	    return reply(hdr, TMM_ProcID(ident));
	}
    }

    //----------------------------------  Add new process to database.
    string Unique = getUnique(Now());
    mProcDB[Unique] = msg.refValue();
    mProcDB[Unique].refData().setProcessID(Unique.c_str());


    //-----------------------------------  Send process to Stream DBs
    int N=mStreamList.size();
    for (int i=0; i<N; ++i) {
        mStreamList[i]->addProcess(mProcDB[Unique].refData());
    }

    cout << "Process " << msg.refValue().getName() << " ID: " 
	 << Unique << " registered." << endl;
    return reply(hdr, TMM_ProcID(Unique));
}

//======================================  Add a trigger to the database.
error_type
TrigMgr::HandleTrigger(const MsgHeader& hdr, const TMM_Trigger& msg) {
    //Time now=Now();

    //-----------------------------------  Find process entry. Bail if none.
    proc_iter iProc = mProcDB.find(msg.refValue().getProcess());
    if (getDebug()) cout << "TrigMgr: Trigger received from process: "
			 << msg.refValue().getProcess() << endl;
    if (iProc == mProcDB.end()) {
        cout << "TrigMgr: Process: " << msg.refValue().getProcess() 
	     << " not registered" << endl;
	return reply(hdr, TMM_NAck()); 
    }

    //-----------------------------------  Eliminate repeated messages
    const trig::TrigProc* pProc = &(iProc->second.refData());
    if (!iProc->second.testMsgID(hdr.getTransID())) {
        cout << "Message: " << hdr.getTransID() << " from process: " 
	     << pProc->getName() << " was repeated." << endl;
	return reply(hdr, TMM_Ack());
    }

    //-----------------------------------  See if trigger is valid.
    iProc->second.incrementRecv();
    if ( !TrigValid(msg.refValue()) ) {
        iProc->second.incrementError();
        cout << "Trigger: " << msg.refValue().getID() << "/" 
	     << msg.refValue().getSubID() << " from " << pProc->getName() 
	     << " v" << pProc->getVersion() << " is not valid." << endl;
        return reply(hdr, TMM_NAck());
    }

    //-----------------------------------  See if trigger is allowed.
    const TrigCtlEntry* pCtl = mCTable.find(*pProc, msg.refValue());
    if (!pCtl) {
        cout << "Trigger: " << msg.refValue().getID() << "/" 
	     << msg.refValue().getSubID() << " from " << pProc->getName() 
	     << " v" << pProc->getVersion() << " rejected." << endl;
        return reply(hdr, TMM_NAck());
    }

    //-----------------------------------  Add trigger to the html log
    mTrigLog.addTrigger(msg.refValue(), *pProc);
    pCtl->count();

    //-----------------------------------  Send triggers to Stream DBs
    int disp = pCtl->getDestMask();
    bool wok(false);
    int N=mStreamList.size();
    for (int i=0; i<N; ++i) {
        wok |= mStreamList[i]->addTrigger(msg.refValue(), disp);
    }
    if (!wok) return reply(hdr, TMM_NAck());
    iProc->second.incrementWrite();
    return reply(hdr, TMM_Ack());
}

//======================================  Add a trigger to the database.
error_type
TrigMgr::HandleSegment(const MsgHeader& hdr, const TMM_Segment& msg) {
    //----------------------------------  Find process entry. Bail if none.
    string procid(msg.refValue().getProcess());
    proc_iter iProc = mProcDB.find(procid);
    if (iProc == mProcDB.end()) {
        cout << "TrigMgr: Process: " << msg.refValue().getProcess() 
	     << " not registered" << endl;
	return reply(hdr, TMM_NAck());
    }
    const trig::TrigProc* pProc = &(iProc->second.refData());

    //----------------------------------  Eliminate repeated messages
    if (!iProc->second.testMsgID(hdr.getTransID())) {
        cout << "Message: " << hdr.getTransID() << " from process: " 
	     << pProc->getName() << " was repeated." << endl;
	return reply(hdr, TMM_Ack());
    }
    iProc->second.incrementRecv();

    //----------------------------------  See if segment is valid.
    if ( !SegValid(msg.refValue()) ) {
        iProc->second.incrementError();
        cout << "Segment: " << msg.refValue().getGroup() << "/" 
	     << msg.refValue().getVersion() << " from " << pProc->getName() 
	     << " v" << pProc->getVersion() << " is not valid." << endl;
        return reply(hdr, TMM_NAck());
    }

    //----------------------------------  See if segment is allowed
    const TrigCtlEntry* pCtl = mCTable.find(*pProc, msg.refValue());
    if (!pCtl) {
        cout << "Segment: " << msg.refValue().getGroup() << "/"
	     << msg.refValue().getVersion() << " from " << pProc->getName()
	     << " v" << pProc->getVersion() << " rejected." << endl;
        return reply(hdr, TMM_NAck());
    }

    //----------------------------------  Send segment to test-DB
    int disp = pCtl->getDestMask();
    bool wok(false);
    int N=mStreamList.size();
    for (int i=0; i<N; ++i) {
        wok |= mStreamList[i]->addSegment(msg.refValue(), disp);
    }
    if (!wok) return reply(hdr, TMM_NAck());
    pCtl->count();
    iProc->second.incrementWrite();
    return reply(hdr, TMM_Ack());
}

//======================================  Handle a close request
error_type
TrigMgr::HandleClose(const MsgHeader& hdr, const TMM_Close& msg) {
    string ident = msg.refValue();
    cout << "Process: " << mProcDB[ident].refData().getName() 
	 << " ID: " << ident << " terminated." << endl;
    Process& pr = mProcDB[ident];
    pr.setClosed();
    if (!pr.isReleasable()) mProcDB.erase(ident);
    return reply(hdr, TMM_Ack());
}

//======================================  Handle a remove-node request
error_type
TrigMgr::HandleRmNode(const MsgHeader& hdr, const TMM_RemNode& msg) {
    string ident = msg.refValue();
    cout << "Removing node: " << ident << endl;
    for (proc_iter i=mProcDB.begin() ; i != mProcDB.end() ; i++) {
        if (ident == i->second.refData().getNode()) {
	    cout << "Process: " << i->second.refData().getName() 
		 << " ID: " << i->first << " terminated." << endl;
	    i->second.setClosed();
	}
    }
    PurgeProcDB();
    return reply(hdr, TMM_Ack());
}


//======================================  Handle a remove-node request
error_type
TrigMgr::HandleReconfig(const MsgHeader& hdr, const TMM_Reconfigure& msg) {
    mCTable.readFile(mConfigFile, false);
    closeStreams();
    configStreams(mStreamConf);

    //-----------------------------------  Register processes with all streams
    int N=mStreamList.size();
    for (int i=0; i<N; ++i) {
	for (proc_iter j=mProcDB.begin(); j != mProcDB.end(); ++j) {
	    mStreamList[i]->addProcess(j->second.refData());
	}
    }
    return OK;
}

//======================================  Write data to the database
void 
TrigMgr::WriteDB(bool force) {
    for (stream_iter i=mStreamList.begin(); i != mStreamList.end(); ++i) {
	(*i)->WriteDB(force);
    }
    PurgeProcDB();
}

//======================================  Delete unneeded process entries
void
TrigMgr::PurgeProcDB(void) {
    for (proc_iter i=mProcDB.begin() ; i != mProcDB.end() ; ) {
	if (i->second.isReleasable()) {
	    mProcDB.erase(i);
	    i = mProcDB.begin();
	} else {
	    i++;
	}
    }
}

//======================================  Write out process statistics
void
TrigMgr::writeStats(void) const {

    //----------------------------------  Build the file name
    const char* dir = getenv("DMTHTMLOUT");
    if (!dir)   dir = ".";
    string     ofile = string(dir) + "/TrigMgr.html";

    //----------------------------------  Build an html document
    html::document doc("Trigger statistics");
    char time[32];
    doc.setBackgroundColor(html::color("white"));
    LocalStr(Now(), time, "%M %d, %Y %02H:%02N");
    html::text hdr(string("Trigger statistics at: ") + time);
    hdr.setSize(html::size(2));
    html::block cb("center");
    cb.addObject(hdr);
    doc.addObject(cb);

    //----------------------------------  Add the TrigMgr parameter table
    html::Table2C parTbl("TrigMgr Parameters");
    parTbl.addRow("Timer",             double(mTimer));
    parTbl.addRow("Triggers per file", double(mDefParam._TableCount));
    parTbl.addRow("Output Directory",  mOutDir);
    doc.addObject(parTbl);


    //----------------------------------  Add the Trigger statistics table
    html::table streamTbl("Stream Statistics");
    streamTbl.addColumn("Stream");
    streamTbl.addColumn("Type");
    streamTbl.addColumn("Unwritten");
    streamTbl.addColumn("Total Trigs");
    streamTbl.addColumn("# Files");
    streamTbl.addColumn("Last Write (UTC)");

    int N=mStreamList.size();
    for (int i=0; i<N; ++i) {
        int row = streamTbl.addRow();
	const TrigStream* pi = mStreamList[i];
	streamTbl.insertData(row, 0, pi->getName());
	streamTbl.insertData(row, 1, pi->getType());
	int nTot = pi->getNTrigs(Time(0)) + pi->getNSegs(Time(0));
	streamTbl.insertData(row, 2, double(nTot));
	const TrigStream::stream_stats& s = mStreamList[i]->refStats();
	streamTbl.insertData(row, 3, double(s.trigTot + s.segTot));
	streamTbl.insertData(row, 4, double(s.fileTot));
	html::text tWrite(s.lastWrite, "%2m/%02d/%Y %2H:%02N:%02S");
	streamTbl.insertData(row, 5, tWrite);
    }
    doc.addObject(streamTbl);

    //----------------------------------  Add the Trigger statistics table
    doc.addObject(mCTable.makeHtmlTable());

    //----------------------------------  Add process table.
    html::table procTbl("Current Processes");
    procTbl.addColumn("Process");
    procTbl.addColumn("Node");
    procTbl.addColumn("PID");
    procTbl.addColumn("Received");
    procTbl.addColumn("Written");
    procTbl.addColumn("Errors");
    long totRecv(0), totWrite(0), totError(0);
    for (const_proc_iter i=mProcDB.begin() ; i != mProcDB.end() ; i++) {
        int row = procTbl.addRow();
	procTbl.insertData(row, 0, i->second.refData().getName());
	procTbl.insertData(row, 1, i->second.refData().getNode());
	procTbl.insertData(row, 2, double(i->second.refData().getProcID()));
	procTbl.insertData(row, 3, double(i->second.getRecvCount()));
	totRecv += i->second.getRecvCount();
	procTbl.insertData(row, 4, double(i->second.getWriteCount()));
	totWrite += i->second.getWriteCount();
	procTbl.insertData(row, 5, double(i->second.getErrorCount()));
	totError += i->second.getErrorCount();
    }
    int row = procTbl.addRow();
    procTbl.insertData(row, 0, "Totals");
    procTbl.insertData(row, 3, double(totRecv)  );
    procTbl.insertData(row, 4, double(totWrite) );
    procTbl.insertData(row, 5, double(totError) );
    doc.addObject(procTbl);

    //----------------------------------  Write the document
    ofstream f(ofile.c_str());
    html::writer w(f);
    doc.write(w);
}

//======================================  Write out segment statistics
void
TrigMgr::writeSegStats(void) const {

    //----------------------------------  Build the file name
    const char* dir = getenv("DMTHTMLOUT");
    if (!dir)   dir = ".";
    string     ofile = string(dir) + "/SegmentStats.html";

    //----------------------------------  Build an html document
    html::document doc("Segment statistics");
    char time[32];
    doc.setBackgroundColor(html::color("white"));
    LocalStr(Now(), time, "%M %d, %Y %02H:%02N");
    html::text hdr(string("Segment statistics at: ") + time);
    hdr.setSize(html::size(2));
    html::block cb("center");
    cb.addObject(hdr);
    doc.addObject(cb);

    //----------------------------------  Write the document
    int N=mStreamList.size();
    for (int i=0; i<N; ++i) {
	const TrigStream* pi = mStreamList[i];
	pi->segStats(doc);
    }

    //----------------------------------  Write the document
    ofstream f(ofile.c_str());
    html::writer w(f);
    doc.write(w);
}
