/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "DQSegWriter.hh"
#include "DQSegTable.hh"
#include "xsil/Xwriter.hh"
#include "xsil/ligolw.hh"
#include "lmsg/ErrorList.hh"
#include <fstream>
#include <sstream>
#include <algorithm>
#include <stdexcept>

using namespace std;

using namespace trig;

typedef std::list<DQSegDef>   segdef_list;
typedef segdef_list::iterator segdef_iter;

//======================================  Segment writer constructor
DQSegWriter::DQSegWriter(void) {
}

//======================================  Segment writer destructor
DQSegWriter::~DQSegWriter(void) {
}

//======================================  Accept a trigger for writing.
lmsg::error_type
DQSegWriter::addTrigger(const TrigBase& t, const TrigProc& p) {
    throw std::runtime_error("mySQl database doesn't handle triggers");
    return lmsg::OK;
}

//======================================  Accept a segment for writing.
lmsg::error_type
DQSegWriter::addSegment(const Segment& s,  const TrigProc& p) {
    proc_iter pit = insert_proc(p);
    if (!mSegList.empty() && mSegList.back() == s) {
	cerr << "Segment repeated. Group: " << s.getGroup() << endl;
	return lmsg::Invalid;
    }
    mSegList.push_back(s);
    mSegList.back().setProcess(pit->getProcessID());
    TrigProc::gps_t tEnd = s.getEndTime().getS();
    if (tEnd > pit->getEndTime()) pit->setEndTime(tEnd);
    return lmsg::OK;
}

//======================================  Clear the tables
void 
DQSegWriter::clear(const Time& start, const Time& end) {      
    mProcList.clear();

    //----------------------------------  Clear the used segments
    for (seg_iter i=mSegList.begin(); i!=mSegList.end(); ) {
	if (!end) {
	    i = mSegList.erase(i);
	} else if (i->getStartTime() < end) {
	    if (i->getEndTime() > end) (i++)->setStartTime(end);
	    else                       i = mSegList.erase(i);
	} else {
	    i++;
	}
    }
}

//======================================  Get the earliest segment start time.
Time
DQSegWriter::getEarly(void) const {
    Time t(0);
    for (const_seg_iter i=mSegList.begin(); i!=mSegList.end(); ++i) {
	if (!t || i->getStartTime() < t) t = i->getStartTime();
    }
    return t;
}

//======================================  Data size methods
int
DQSegWriter::getNSegs(const Time& t) const {
    if (!t) return mSegList.size();
    int N = 0;
    for (const_seg_iter i=mSegList.begin(); i!=mSegList.end(); ++i) {
	if (i->getStartTime() < t) N++;
    }
    return N;
}

//====================================  Accept a segment for writing.
lmsg::error_type
DQSegWriter::setProcess(const TrigProc& p) {
    refProcess()  = p;
    //proc_iter pit = 
    insert_proc(refProcess());
    return lmsg::OK;
}

//====================================  Write out the tables
lmsg::error_type
DQSegWriter::write(const string& file, const Time& start, 
		   const Time& end) const {
    if (mSegList.empty()) return lmsg::OK;

    //--------------------------------  Build the process table
    ProcTable* pTab = new ProcTable(true);
    for (const_proc_iter i=mProcList.begin(); i != mProcList.end(); ++i) {
	pTab->addRow(*i);
    }
  
    //--------------------------------  Build the segment tables
    DQSegDefTable* dTab = new DQSegDefTable;
    DQSegTable*    sTab = new DQSegTable;
    DQSegMapTable* mTab = new DQSegMapTable;
    segdef_list    sdl;
    int segN(0);
    for (const_seg_iter i=mSegList.begin(); i != mSegList.end(); ++i) {
	if (!end || i->getStartTime() < end) {
	    Segment seg(*i);
	    if (end != Time(0) && seg.getEndTime() > end) seg.setEndTime(end);

	    //------------------------  Find the segment definition
	    segdef_iter df = find(sdl.begin(), sdl.end(), *i);
	    if (df == sdl.end()) {
		int nRow = sdl.size();
		sdl.push_back(seg);
		df = sdl.end(); --df;
		df->setDefinerID(dTab->citeTable("segment_def_id", nRow));
		dTab->addRow(*df);
	    }
	  
	    //------------------------  Add the segment
	    seg.setSegID(sTab->citeTable("segment_id", segN++));
	    sTab->addRow(seg);

	    //------------------------  Add the segment
	    mTab->addRow(seg, *df);
	}
    }

    //--------------------------------  Build up a document
    xsil::ligolw lwfile;
    lwfile.addObject(pTab);
    lwfile.addObject(dTab);
    lwfile.addObject(sTab);
    lwfile.addObject(mTab);

    //--------------------------------  Write it to the outptu file
    std::ofstream str(file.c_str());
    xsil::Xwriter xw(str);
    lwfile.Spew(xw);
    return lmsg::OK;
}

//====================================  Search for process
DQSegWriter::proc_iter 
DQSegWriter::insert_proc(const TrigProc& p) {
    int inx(0);
    for (proc_iter i=mProcList.begin(); i != mProcList.end(); ++i) {
	if (*i == p) return i;
	++inx;
    }
    mProcList.push_back(p);
    proc_iter l = mProcList.end(); --l;
    //if (!*(l->getProcessID())) {
    ostringstream id;
    id << "process:process_id:" << inx;
    l->setProcessID(id.str());
    //}
    string ifo = l->getIFOs();
    for (string::size_type i=0; i<ifo.size(); ) {
	if (ifo[i] == ' ') ifo.erase(i, 1);
	else               i++;
    }
    l->setIFOs(ifo);
    return l;
}

