#ifndef TUNNEL_HH
#define TUNNEL_HH

#include "lmsg/MsgAddr.hh"
#include "NameClient.hh"
#ifdef TCP_TRANSPORT
#include "lmsg/TransportTCP.hh"
#else
#include "lmsg/TransportMsg.hh"
#endif
#include "SigFlag.hh"
#include "Time.hh"
#include "Interval.hh"
#include <map>
#include <string>
#include <vector>

/**  @name Tunnel
  *  The DMT Message Tunnel provides a path for messages to travel between 
  *  processes connected to two disjoint networks. Tunnel processes usually 
  *  reside on gateway nodes with connections to both the source and 
  *  destination networks. The Tunnel is closely associated with the name 
  *  server. covering the gateway node. WHen a request is received by the 
  *  name server for a target address that is not reachable by the requesting
  *  process, the name server searches its list of tunnels for one that 
  *  connects the requester's network to the target network. If an appropriate
  *  tunnel is found, the name server returns the tunnel's server port address
  *  and a gives the index of the target process entry as a process ID. When
  *  the tunnel receives a message to be routed, it requests the target 
  *  address from the name server (with an IDget request). It then records
  *  the reply address and forwards the message to the target port. When a
  *  reply is received, the message is resent to the original requestor.
  *  @memo LMSG network message bridge.
  *  @author J. Zweizig
  *  @version 1.0; Modified July 22, 2002
  */
//@{
//@}

class Tunnel {
public:
    typedef lmsg::MsgAddr::ipport_t ServerID;

public:
    Tunnel(int argc, const char* argv[]);
    ~Tunnel(void);
    int MainLoop(void);
    ServerID addClient(const lmsg::MsgAddr& addr);
    bool isActive(void) const {return mActive;}

private:
    class ClientRec {
    public:
      ClientRec(void) : mValid(false) {}
      explicit ClientRec(const lmsg::MsgAddr& addr);
      bool isValid(void) const;
      void Annul(void);
      const lmsg::MsgAddr& getReturn(void) const;
    private:
      bool          mValid;
      lmsg::MsgAddr mReturn;
    };

private:
    typedef std::map<ServerID, lmsg::MsgAddr> ServerMap;
    typedef ServerMap::const_iterator server_iter;

    typedef std::vector<ClientRec> ClientMap;

private:
    ServerMap  mServerMap;
    ClientMap  mClientMap;
    lmsg::NameClient   mNameServer;
    std::string mTunnelName;
#ifdef TCP_TRANSPORT
    lmsg::TransportTCP mTunnelServer;
    lmsg::TransportTCP mTunnelClient;
#else
    lmsg::TransportMsg mTunnelServer;
    lmsg::TransportMsg mTunnelClient;
#endif
    SigFlag       mTerm;
    unsigned long mNetAddr;
    unsigned long mNetMask;
    int           mDebug;
    bool          mActive;
    Interval      mStatTime;
    Time          mLastTime;
    std::string   mStatFile;
    long          mRequests;
    long          mReplies;
    long          mAddrErrors;
    long          mLookupErrors;
    long          mSendErrors;
    long          mReplyErrors;
};

inline
Tunnel::ClientRec::ClientRec(const lmsg::MsgAddr& addr) 
  : mValid(true), mReturn(addr)
{}

inline bool
Tunnel::ClientRec::isValid(void) const {
    return mValid;
}

inline void
Tunnel::ClientRec::Annul(void) {
    mValid = false;
}

inline const lmsg::MsgAddr&
Tunnel::ClientRec::getReturn(void) const {
    return mReturn;
}




#endif  // TUNNEL_HH
