#ifndef MONINDEX_HH
#define MONINDEX_HH

#include <string>
#include <map>

class MonClient;

/**  The MonIndex class provides access to the monitor data server index 
  *  information.
  *  @memo Access monitor index information.
  *  @author John Zweizig
  *  @version 1.0 ; Modified June 5, 2000
  */
class MonIndex {
public:

  class Entry {
  public:
    Entry(void) {}
    Entry(const std::string& Type, const std::string& Comment);
    Entry(const Entry& x);
    ~Entry(void);
    const char* getType(void) const;
    const char* getComment(void) const;
  private:
    std::string mType;
    std::string mComment;
  };

  typedef std::map<std::string, Entry> IndexMap;

  /**  The index iterator may be used to iterate over all entries in the 
    *  index. The first and last legal values for the iterator are given
    *  by begin() and end(), respectively, and the iterator corresponding 
    *  to a given entry may be fetched with find(). const_index_iterator
    *  has all te properties of a standard template library iterator.
    *  @memo Iterator.
    */
  typedef IndexMap::const_iterator const_index_iterator;

public:
  /**  Construct an empty index.
    *  @memo Default constructor.
    */
  MonIndex(void);

  /**  Construct an index and fill it from the specified client.
   */
  MonIndex(MonClient& mc);

  /**  Construct an index and initialize it from the specified server.
   */
  MonIndex(const char* Server);

  /**  Destroy an index.
   */
  ~MonIndex(void);

  /**  Return a pointer to first entry in the index.
    *  @memo Index start iterator.
    *  @return iterator pointing to the first entry in the Index.
    */
  const_index_iterator begin(void) const;

  /**  Return a pointer to last entry in the index.
    *  @memo Index end iterator.
    *  @return iterator pointing to the last entry in the Index.
    */
  const_index_iterator end(void) const;

  /**  Return a pointer to named entry in the index.
    *  @memo Get pointer to specified entry.
    *  @return iterator pointing to the name entry in the Index.
    */
  const_index_iterator find(const std::string& Name) const;

  /**  Test whether the specified entry exists.
    */
  bool exists(const std::string& name) const;

  /**  Get the comment field from the named entry.
    *  @memo Get the specified comment.
    *  @return Pointer to the named comment
    */
  const char* getComment(const std::string& name) const;
 
  /**  Get the comment field from the named entry.
    *  @memo Get the specified comment.
    *  @return Pointer to the named comment
    */
  const char* getComment(const_index_iterator& iter) const;

  /**  Get the name from the entry pointed to by an iterator.
   */
  const char* getName(const_index_iterator& iter) const;

  /**  Get the type field from the named entry.
   */
  const char* getType(const std::string& name) const;

  /**  Get the type field from the named entry.
   */
  const char* getType(const_index_iterator& iter) const;

  /**  Add entries to an index. The specified client interface is
    *  used to access the index information from the connected server.
    *  @memo Get index information.
    *  @param mc A connected Client interface.
    *  @return true if successful
    */
  bool setIndex(MonClient& mc);

  /**  Add entries to an index. A temporary client interface is created
    *  and connected to the specified server. This is used to access the
    *  index information from the server.
    *  @memo Get index information.
    *  @param Server Pointer to a character string holding the server name.
    *  @return true if successful
    */
  bool setIndex(const char* Server);

  /**  Remove all entries from the index.
   */
  void clear(void);

private:
  IndexMap mIndex;
};

//--------------------------------------  Index accessors
inline MonIndex::const_index_iterator 
MonIndex::begin(void) const {
    return mIndex.begin();
}

inline MonIndex::const_index_iterator 
MonIndex::end(void) const {
    return mIndex.end();
}

inline MonIndex::const_index_iterator 
MonIndex::find(const std::string& Name) const {
    return mIndex.find(Name);
}


#endif // MONINDEX_HH
