/* -*- mode: c++; c-basic-offset: 3 -*- */
#define PIDCVSHDR "$Header: https://redoubt.ligo-wa.caltech.edu/svn/gds/trunk/Monitors/kleineWelle/kleineWelleM.cc 7347 2015-04-23 18:11:19Z john.zweizig@LIGO.ORG $"
#define PIDTITLE  "kleineWelle"
#include "ProcIdent.hh"
#include "mywv2lib.hh"
#include "gamma.hh"
#include "kleineWelleM.hh"
#include "LPEFilter.hh"
#include "DVector.hh"
#include "SBTrigger.hh"
#include "Segment.hh"

#include <iostream>
#include <fstream>
#include <iomanip>
#include <cstdlib>
#include <unistd.h>
#include <sstream>
#include <cmath>
#include <iterator>
#include <sys/stat.h>
#include <time.h>

#define KWVERSION "August 4, 2009"

EXECDAT(kleineWelle)

using namespace std;

//======================================  Channel data constructor
kwChan::kwChan(const std::string& name)
   : channel(name),              // name of the channel
     lowFrequency(64),           // low frequency bound of analysis
     highFrequency(1024),        // high frequency bound of analysis
     significance(10.0),         // significance threshold for trigger output
     threshold(3.0),             // threshold on sigmas for black pixels
     maxDist(2),                 // distance on time-scale plane for clustering
     decimateFactor(-1),         // times to decimate timeseries (-1 for auto)
     transientDuration(4.0),     // minimum time to cut for filter transients
     lpefilter(NULL),
     skipNext(1),                // flag to skip trigger generation for one stride after filter reset
     simdata(false),             // don't read from simdata
     fs(0),                      // sample frequency of channel (0 for auto)
     wavePtr(NULL)
{
}

kwChan::~kwChan(void) {
   if(wavePtr) {
      delete wavePtr;
      wavePtr   = NULL;
   }
   if(lpefilter) {
      delete lpefilter;
      lpefilter = NULL;
   }
}

// Channel operator
kwChan& kwChan::operator=(const kwChan& x) {
   fs             = x.fs;
   channel        = x.channel;
   significance   = x.significance;
   threshold      = x.threshold;
   size           = x.size;
   lpefLength     = x.lpefLength;
   maxDist        = x.maxDist;
   lowFrequency   = x.lowFrequency;
   highFrequency  = x.highFrequency;
   decimateFactor = x.decimateFactor;
   transientDuration = x.transientDuration; // not used
   if (x.wavePtr) wavePtr = new Haar_Wavelet(size);
   else           wavePtr = NULL;
   prefilter      = x.prefilter; // auto-pipe clones filter.
   if (x.lpefilter) lpefilter  = x.lpefilter->clone();
   else             lpefilter  = NULL;
   simdata        = x.simdata;
   return *this;
}

// set up sample-frequency derived data
void kwChan::setup(const TSeries* ts) {
   size = ts->getNSample();
   fs = int(size / ts->getInterval() + 1e-6);
   FilterDesign fd(fs);
   if(highFrequency > fs / 2)  // cap to nyquist
      highFrequency = fs / 2;
   lowScale  = (int)ceil(log2(fs / highFrequency));
   highScale = (int)ceil(log2(fs / lowFrequency));
   std::stringstream out;
   out << channel << "_" << lowFrequency << "_" << highFrequency;
   uniqueID = out.str();
   uniqueID.replace(2, 1, "_");
   /// decimator is part of prefilter
   if(decimateFactor == -1)
      decimateFactor = lowScale - 1;  // leave headroom for linear filter
   fs = (fs >> decimateFactor);
   // need to adjust wavelet parameters
   size = (size >> decimateFactor);
   lowScale -= decimateFactor;
   highScale -= decimateFactor;
   // prefilter
   fd.decimateBy2(decimateFactor, 3); // type 3 has most ringing, but least timeshift
   fd.butter(kHighPass, 6, lowFrequency, 0);
   prefilter.set(fd.release());
   // construct LEPF
   lpefLength = min(int(double(transientDuration) * fs), 
		    int(16*highFrequency/lowFrequency));
   if (lpefilter) delete lpefilter;
   lpefilter = new LPEFilter(lpefLength, 0, size-1);
   if (wavePtr) delete wavePtr;
   wavePtr   = new Haar_Wavelet(size);          // initialize wavelet class
}

// grab timeseries and train/apply filters, put events in events vector
event_vect 
kwChan::process(MultiDacc& dacc) {
   TSeries* tsIn = dacc.refData(channel.c_str());
   if(tsIn->getNSample() > 0 && tsIn->refDVect()->finite()) {
      tsIn->Convert(DVector::t_float);
      Time t0 = tsIn->getStartTime();
      if(fs == 0) setup(tsIn);         // first time data is looked at, do setup
      TSeries tsHpf = prefilter(*tsIn); //autopipe::operator()
      lpefilter->train(tsHpf);
      if(!skipNext) {
	 lpefilter->apply(tsHpf, tsLpef);
	 tsLpef.getData(size, wavePtr->getRawArray()); // put data into wavelet class
	 wavePtr->transform(1);                     // wavelet transform
	 const double* wavearray = wavePtr->getWaveArray();
	 element_vect elements;
	 // build vector of black pixels
	 for(int scale=lowScale; scale <= highScale; scale++) {
	    double dt = double(1 << scale)/double(fs); // duration of one pixel
	    int lsize = size >> scale;  // index where array for scale begins
	    double lsum = 0;
	    for(int i = lsize; i < 2*lsize; i++) // calculate sigma
	       lsum += wavearray[i]*wavearray[i];
	    double sigma = sqrt(lsum / lsize);
	    for(int i = lsize; i < 2*lsize; i++) {
	       double amplitude = wavearray[i];
	       double normalized = amplitude / sigma;
	       if(abs(normalized) > threshold) {
		  double time = dt * double(i-lsize);
		  elements.push_back(Element(amplitude*amplitude, 
					     normalized*normalized,
					     i, scale, time, 
					     time + dt, dt, 0));
	       }
	    }
	 }

	 // assign cluster ID to black pixels
	 int freeID = 0;
	 int N = elements.size();
	 if (N > 0) {
	    vector<int> pvect(N, 0);
	    for (int i=0; i<N; ++i)
	       pvect[i] = i;
	    int ifree = 0;
	    for (int i=0; i<N; ++i) {
	       if (i == ifree) {
		  ifree++;
		  elements[pvect[i]].clusterID = freeID++;
	       }
	       const Element& testEl(elements[pvect[i]]);
	       for (int j=ifree; j<N; ++j) {
		  int pj = pvect[j]; 
		  if (testEl.dist(elements[pj]) <= maxDist) {
		     elements[pj].clusterID = testEl.clusterID;
		     if (j != ifree) {
			pvect[j]     = pvect[ifree];
			pvect[ifree] = pj;
		     }
		     ifree++;
		  }
	       }
	    }
	 }

	 // sum clusters by ID
	 event_vect events(freeID, Event(uniqueID));
	 for(element_iter i=elements.begin(); i < elements.end(); i++) {
	    events[i->clusterID] += *i;
	 }

	 // estimate cluster parameters
	 int j=0;
	 for(int i = 0; i < freeID; i++) {
	    events[i].average(t0, fs);
	    if(events[i].significance >= significance) {
	       if (j != i) events[j] = events[i];
	       j++;
	    }
	 }

	 // return sorted events
	 events.erase(events.begin()+j, events.end());
	 sort(events.begin(), events.end());
	 return(events);
      } else { // flagged to skip this stride, do not output events
	 skipNext = 0;
	 event_vect events(0, Event(""));
	 return(events);
      }
   } else {
      // rate flag is not handled here, maybe should be eventually
      skipNext = 1; // flag to skip next stride
      if(fs != 0)   // channel has been initialized
	 prefilter->reset();
      if(lpefilter)
	 lpefilter->reset();
      if(tsIn->getNSample() == 0)
	 cerr << "channel missing: " << tsIn->getName() << endl;
      else if(!tsIn->refDVect()->finite())
	 cerr << "non-finite (NaN or inf) data: " << tsIn->getName() << endl;
      event_vect events(0, Event(""));
      return(events);
   }
}

//======================================  Channel data constructor
kwRateFlag::kwRateFlag(const std::string& name)
   : channelName(name),          // name of the channel with frequency bounds
     sigThreshold(0.0),          // significance threshold for trigger output
     nStride(0),                 // number of strides for averaging
     counts(NULL)
{
}

kwRateFlag::~kwRateFlag(void) {
   if(counts) {
      delete counts;
      counts = NULL;
   }
}

kleineWelle::kleineWelle(int argc, const char *argv[])
   : MultiStream(argc, argv),
     mTrigClient(trig::TrigWriter::kSBTrig),
     mSegClient(trig::TrigWriter::kS6Seg),
     stride(16.0),
     basename("KW"),
     segname("KW_DQ"),
     lasttime(0)
{
   bool syntax = false;        // flag for syntax errror

   // state vector handling
   // const char* OscFile = "LockLoss.conf";
   // OperStateCondList* mOsc = new OperStateCondList(*getDacc(0));
   // mOsc->readConfig(OscFile);

   // clear process XML output
   mTrigClient.setTableFile("/dev/null");
   mTrigClient.flush(Time(0, 0), Time(0, 0));
   mSegClient.setTableFile("/dev/null");
   mSegClient.flush(Time(0, 0), Time(0, 0));

   // parse command syntax
   if(argc == 1) {
      syntax = true;
   } else {
      optionsFile.open(argv[1], ios::in);
      if(!(optionsFile.is_open())) {
	 cerr << "cannot open options file: " << argv[1] << endl;
	 finish();
	 return;
      } else if (Debug()) {
	 cerr << "reading options file: " << argv[1] << endl;
      }
   }
   if(syntax) {
      cerr << "kleineWelle version " << KWVERSION << endl;
      cerr << "Command syntax:" << endl;
      cerr << argv[0] << " [<options file>] [<MultiStream-args>]" << endl;
      cerr << "please see sample options file 'optM' for example" << endl;
      finish();
      return;
   }

   //-----------------------------------  Read in options from file
   string segment_file_name;
   kwChan default_chan = kwChan("default");
   kwChan temp = kwChan("default");
   while(!optionsFile.eof()) {
      string option, word;
      getline(optionsFile, option);
      if (Debug()) cerr << "Option: " << option << endl;
      istringstream optionsStream(option);
      optionsStream >> word;
      if(word.length() == 0 || word[0] == '#') {
	 if (Debug()) cerr << "Option line skipped" << endl;
	 continue;
      }
      else if(word == "stride") {         // DMT analysis stride
	 double epoch;
	 optionsStream >> epoch;
	 stride = Interval(epoch);
      }
      else if(word == "basename")         // prefix for trigger files
	 optionsStream >> basename;
      else if(word == "segname")          // prefix for segment files
	 optionsStream >> segname;
      else if(word == "threshold")        // default threshold on coefficient amplitude
	 optionsStream >> default_chan.threshold;
      else if(word == "significance")     // default threshold on cluster significance
	 optionsStream >> default_chan.significance;
      else if(word == "maxDist")          // default distance for clustering
	 optionsStream >> default_chan.maxDist;
      else if(word == "decimateFactor")   // times to decimate data (-1 = auto, 0 = none)
	 optionsStream >> default_chan.decimateFactor;
      else if(word == "transientDuration") { // length of filter transient to remove
	 double epoch;
	 optionsStream >> epoch;
	 default_chan.transientDuration = Interval(epoch);
      }
      else if(word == "vetoseg") {
	 std::string vetoName;
	 double sigThresh;
	 optionsStream >> vetoName;
	 optionsStream >> sigThresh;
	 mVetoList.insert(pair<std::string,double>(vetoName,sigThresh));
      }
      else if(word == "rateflag") {
	 kwRateFlag* temp2 = new kwRateFlag("default");
	 optionsStream >> temp2->channelName;
	 optionsStream >> temp2->sigThreshold;
	 optionsStream >> temp2->rateThreshold;
	 optionsStream >> temp2->nStride;
	 temp2->counts = new int(2*temp2->nStride + 1);
	 for(int i = 0; i < 2*temp2->nStride + 1; i++) {
	    temp2->counts[i] = -1; // -1 for no counts data
	 }
	 mRateFlagList.push_back(*temp2);
	 temp2 = NULL; // push_back only made a copy of the pointer, don't let destructor get called
      }
      else if (word == "simdata") {
	 optionsStream >> default_chan.simdata;
      }
      else if(word == "channel") {    // KW analysis channel
	 temp = default_chan;         // initialize channel with default values 
	 optionsStream >> temp.channel;
	 optionsStream >> temp.lowFrequency;
	 optionsStream >> temp.highFrequency;
	 optionsStream >> temp.significance;
	 mChanList.push_back(temp);
	 if (Debug()) cerr << "Configure channel: " << temp.channel << endl;
      }
      else
	 cerr << "bad option: " << word << endl;
   }
   optionsFile.close();

   // Check at least one channel
   if (mChanList.empty()) {
      cerr << "No channels were configured!" << endl;
      finish(); 
      return;
   } else {
      for (channel_iter i=mChanList.begin(); i != mChanList.end(); ++i) {
	 if (i->simdata) getDacc().addSimulated(i->channel.c_str(), 0);
	 else            getDacc().addChannel(i->channel.c_str(), 0);
	 // do not abort on invalid data
	 getDacc(0)->setChannelFlag(i->channel.c_str(), Channel::kAllowNonNormal);
	 getDacc(0)->setChannelFlag(i->channel.c_str(), Channel::kAllowNotValid);
      }
   }

   // Set up data stream.
   getDacc().setStride(stride);
   getDacc().setIgnoreMissingChannel(true);
   setStrideAlignment(long(stride), 0.0);
   if (Debug()) cout << "finished configuration setup" << endl;
}

kleineWelle::~kleineWelle()
{
   cout << "# finished." << endl;
}

void kleineWelle::sendSeg(vetoseg_iter this_veto, Time start_sec, Time end_sec, int active)
{
   int version=1; // New convention: Online segments are version 1
   std::stringstream segname;
   std::stringstream comment;
   string name = this_veto->first;
   std::string::size_type chan_length = name.size();
   std::string::size_type chan_position = name.find('_');
   std::string::size_type upper_position = name.find_last_of("_");
   std::string::size_type lower_position = name.find_last_of("_", upper_position-1);
   std::string ifo_name = name.substr(0,2);
   comment << "KleineWelle veto derived from " << ifo_name << ":" 
	   << name.substr(chan_position+1,lower_position-3) << "; " 
	   << name.substr(lower_position+1,(upper_position-lower_position)-1) 
	   << " < f < " 
	   << name.substr(upper_position+1, chan_length - upper_position) 
	   << "; threshold " << this_veto->second;
   segname << "DMT-KW_" << name.substr(chan_position+1,chan_length) 
	   << "_" << this_veto->second;
   std::string segmentname = segname.str();
   segmentname.replace(10,1,"_");
   trig::Segment s_kw(segmentname,version,start_sec,end_sec);
   s_kw.setIfos(ifo_name.c_str());
   s_kw.setComment(comment.str().c_str());
   s_kw.setActivity(active);
   lmsg::error_type rc = mSegClient.sendSegment(s_kw);
   if (rc) cout << "Error sending segment: error code " << rc << endl;
}  /* END: sendSeg() */

void kleineWelle::sendRateSeg(rateflag_iter this_rateflag, Time start_sec, 
			      Time end_sec, int active)
{
   int version=1; // New convention: Online segments are version 1
   std::stringstream segname;
   std::stringstream comment;
   std::string::size_type chan_length = this_rateflag->channelName.size();
   std::string::size_type chan_position = this_rateflag->channelName.find('_');
   std::string::size_type upper_position = this_rateflag->channelName.find_last_of("_");
   std::string::size_type lower_position = this_rateflag->channelName.find_last_of("_", upper_position-1);
   std::string ifo_name = this_rateflag->channelName.substr(0,2);
   comment << "KleineWelle rate flag derived from " << ifo_name << ":" << 
      this_rateflag->channelName.substr(chan_position+1,lower_position-3) << "; " <<
      this_rateflag->channelName.substr(lower_position+1,(upper_position-lower_position)-1) << " < f < " <<
      this_rateflag->channelName.substr(upper_position+1, chan_length - upper_position) << 
      "; threshold " << this_rateflag->sigThreshold << " with rate > " << this_rateflag->rateThreshold << " Hz using "
	   << stride * (double)(2*this_rateflag->nStride + 1) << " second stride";
   segname << "DMT-KW_" << this_rateflag->channelName.substr(chan_position+1) << "_RATE" << this_rateflag->sigThreshold << "_"
	   << this_rateflag->rateThreshold << "HZ_" << stride * (double)(2*this_rateflag->nStride + 1) << "S";
   std::string segmentname = segname.str();
   segmentname.replace(10,1,"_");
   trig::Segment s_kw(segmentname,version,start_sec,end_sec);
   s_kw.setIfos(ifo_name.c_str());
   s_kw.setComment(comment.str().c_str());
   s_kw.setActivity(active);
   if(Debug()) cout << "sendseg " << segmentname << " " << start_sec << " " << end_sec << " " << active << endl << comment.str() << endl;;
   lmsg::error_type rc = mSegClient.sendSegment(s_kw);
   if (rc) cout << "Error sending segment: error code " << rc << endl;
}  /* END: sendRateSeg() */

void kleineWelle::ProcessData(void)
{
   Time t0 = getDacc().getFillTime();
   double rem = fmod(t0.totalS(), stride.GetSecs());
   std::map<std::string,Time> lastVetoSeg;
   time ( &rawtime );
   timeinfo = gmtime ( &rawtime );
   cout << "# processing at " << t0 << " offset " << t0.totalS() 
	<< " " << stride.GetSecs() << " " << rem << " - " << asctime(timeinfo);

   // do we need tolerance here? (not as long as the stride is an int).
   if(rem == 0) {

      // check state vector (not done yet)
      if(t0 == lasttime) {
	 std::stringstream path;
	 std::stringstream segpath;
	 path << basename << "-" << t0.getS()/100000;
	 std::string dir = path.str();
	 path << "/" << basename << "-" << t0.getS() << "-" 
	      << stride.GetSecs() << ".trg";
	 std::string file = path.str();
	 segpath << dir << "/" << segname << "-" << t0.getS() << "-" 
                 << stride.GetSecs() << ".xml";
	 std::string segfile = segpath.str();

	 ofstream outfile;
	 outfile.open(file.c_str());
	 if(!outfile) {
	    mkdir(dir.c_str(), 0755);
	    outfile.open(file.c_str());
	    if(!outfile) {
	       cerr << "cannot create file: " << file;
	       finish();
	    }
	 }

	 std::stringstream trendpath;
	 std::string trendFilename;
	 ofstream outTrendFile;
	 trendpath << dir << "/" << basename << "_TRENDS-"
		   << (int)t0.getS()/100000 << "00000-100000.txt";
	 outTrendFile.open(trendpath.str().c_str(), ios::app);

	 if(!outTrendFile) {
	    cerr << "cannot append to file: " << trendpath.str() << endl;
	    finish();
	 }

	 if((int)outTrendFile.tellp() == 0) {
	    int col = 1;
	    outTrendFile << "#" << setw(2) << col++ << ": Time";
	    for(channel_iter k=mChanList.begin(); k != mChanList.end(); k++) {
	       outTrendFile << "\n#" << setw(2) << col++ << ": "
			    << k->uniqueID << "-Count(Significance>0)";
	       outTrendFile << "\n#" << setw(2) << col++ << ": " 
			    << k->uniqueID << "-Count(Significance>50)";
	       outTrendFile << "\n#" << setw(2) << col++ << ": " 
			    << k->uniqueID << "-Ave-Frequency";
	       outTrendFile << "\n#" << setw(2) << col++ << ": " 
			    << k->uniqueID << "-Ave-Duration";
	       outTrendFile << "\n#" << setw(2) << col++ << ": " 
			    << k->uniqueID << "-Ave-Significance";
	       outTrendFile << "\n#" << setw(2) << col++ << ": " 
			    << k->uniqueID << "-Max-Significance";
	       outTrendFile << "\n#" << setw(2) << col++ << ": " 
			    << k->uniqueID << "-Ave-Npts";
	    }
	    outTrendFile << flush;
	 }

	 // initialize the veto segments
	 for(vetoseg_iter i=mVetoList.begin(); i != mVetoList.end(); ++i)
	    lastVetoSeg.insert(pair<std::string,Time>(i->first,Time(t0.getS())));

	 outTrendFile << "\n" << t0.getS();

	 // process the channels and write the kw veto files
	 for(channel_iter i=mChanList.begin(); i != mChanList.end(); ++i) {
	    event_vect events = i->process(getDacc());

	    // generate and output trends
	    double C_Trigger_0(0.0), C_Trigger_50(0.0);
	    double Ave_Fre(0.0), Ave_Duration(0.0), Ave_Sig(0.0);
	    double Max_Sig(0.0), Ave_Npts(0.0);
	    for(event_iter j=events.begin(); j != events.end(); ++j) {
	       ++C_Trigger_0;
	       if (j->significance>50)
		  ++C_Trigger_50;
	       Ave_Fre=Ave_Fre+j->frequency;
	       Ave_Duration=Ave_Duration+j->endTime-j->startTime;
	       Ave_Sig=Ave_Sig+j->significance;
	       Max_Sig=(Max_Sig>j->significance?Max_Sig:j->significance);
	       Ave_Npts=Ave_Npts+j->npts;
	    }

	    //calculate averages
	    if (C_Trigger_0 > 1) {
	       Ave_Fre      /= C_Trigger_0;
	       Ave_Duration /= C_Trigger_0;
	       Ave_Sig      /= C_Trigger_0;
	       Ave_Npts     /= C_Trigger_0;
	    }

	    //Write statistics into output file
	    outTrendFile << ' ' << C_Trigger_0 << ' ' << C_Trigger_50
			 << ' ' << Ave_Fre << ' ' << Ave_Duration
			 << ' ' << Ave_Sig << ' ' << Max_Sig << ' ' << Ave_Npts;

	    // generate rateflag if requested
	    for(rateflag_iter this_rateflag = mRateFlagList.begin(); 
		this_rateflag != mRateFlagList.end(); this_rateflag++) {
	       // should we put a check here in case a channel is accidentally
	       // defined twice with the same parameters?
	       if(this_rateflag->channelName == i->uniqueID) {
		  int count = 0;
		  for(event_iter j=events.begin(); j != events.end(); ++j) {
		     if(j->significance > this_rateflag->sigThreshold) {
			count++;
		     }
		  }
		  int k = 0;
		  // shift over historical counts eliminating first one
		  for(k = 0; k < 2 * this_rateflag->nStride; k++) {
		     this_rateflag->counts[k] = this_rateflag->counts[k+1];
		  }
		  // fill in most recent value
		  this_rateflag->counts[k] = count;
		  int total = 0;
		  bool skip = 0;
		  for(k = 0; k < 2 * this_rateflag->nStride + 1; k++) {
		     if(this_rateflag->counts[k] == -1) {
			skip = 1; // rate flag is skipped if we don't h ave enough history
		     }
		     total += this_rateflag->counts[k];
		  }
		  double rate = (double)total / (stride * (double)(2 * this_rateflag->nStride + 1));
		  Time tStride0 = t0 - stride * double(this_rateflag->nStride);
		  if(!skip && rate >= this_rateflag->rateThreshold) {
		     sendRateSeg(this_rateflag, tStride0, 
				 tStride0 + Interval(stride), 1);
		     if(Debug()) cout << i->uniqueID << " setting flag " 
				      << tStride0 << " "
				      << tStride0 + Interval(stride) 
				      << " 1 rate: " << rate << endl;
		  } else if(!skip) {
		     sendRateSeg(this_rateflag, tStride0,
				 tStride0 + Interval(stride), 0);
		     if(Debug()) cout << i->uniqueID << " setting flag " 
				      << tStride0 << " "
				      << tStride0 + Interval(stride) 
				      << " 0 rate: " << rate << endl;
		  } else {
		     if(Debug()) cout << "skipped rateflag not enough strides" 
				      << endl;
		  }
	       }
	    }
	    // output ASCII events
	    copy(events.begin(), events.end(), 
		 ostream_iterator<Event>(outfile));

	    // loop over the events to construct xml vetoes and triggers
	    for(event_iter j=events.begin(); j != events.end(); ++j) {
	       // generate veto segments
	       vetoseg_iter this_veto = mVetoList.find(j->channelName);
	       if( (this_veto != mVetoList.end()) && 
		   (j->significance > this_veto->second) ) {

		  // create the inactive and active segments
		  Time start_sec((Time::ulong_t)floor(j->startTime),0);
		  Time end_sec((Time::ulong_t)ceil(j->endTime),0);
		  if ( lastVetoSeg[this_veto->first] != start_sec ) {
		     // create an inactive segment from the last time to the present
		     sendSeg(this_veto, lastVetoSeg[this_veto->first],
			     start_sec, 0);
		  }
		  // create the active segment and update the last segment time
		  sendSeg(this_veto, start_sec, end_sec, 1);
		  lastVetoSeg[this_veto->first] = end_sec;
	       }

	       // fill trigger table
	       double dStart = j->startTime;
	       Time::ulong_t sec(dStart);
	       Time tStart(sec, Time::ulong_t(1e9*(dStart-sec)));
	       double f0 = j->frequency;
	       fComplex xfc;
	       i->lpefilter->Xfer(xfc, f0);
	       trig::SBTrigger trigger("kleineWelle",
				       i->channel, 
				       tStart, 
				       Interval(j->endTime - dStart),
				       f0,
				       0.50 * f0,
				       sqrt(j->normEnergy - j->npts)
				       );
	       trigger.ifo(j->channelName.substr(0,2));
	       trigger.setSignificance(j->significance);
	       trigger.setConfidence(j->significance);
	       trigger.amplitude(sqrt(j->energy/xfc.MagSq()));
	       trigger.setAvgOffset(j->centralTime - dStart);
	       trigger.setPeakOffset(j->centralTime - dStart);
	       trigger.peak_frequency(f0);
	       trigger.setPixelCount(j->npts);
	       mTrigClient.sendTrigger(trigger);
	    }
	 }

	 // clean up the veto segments
	 Time segment_end(t0.getS() + (Time::ulong_t)stride.GetSecs(),0);
	 for(vetoseg_iter this_veto=mVetoList.begin(); 
	     this_veto != mVetoList.end(); 
	     ++this_veto) {
	    if(lastVetoSeg[this_veto->first] != segment_end) {
	       sendSeg(this_veto, lastVetoSeg[this_veto->first],
		       segment_end, 0);
	    }
	 }

	 // output XML version
	 file.replace(file.size() - 3, 3, "xml");
	 // cout << "writing out XML files " << file << " and " << segfile 
	 //      << endl;
	 mTrigClient.setTableFile(file.c_str());
	 mTrigClient.flush(t0, t0+stride);
	 mSegClient.setTableFile(segfile.c_str());
	 mSegClient.flush(t0, t0+stride);
	 outfile.close();
	 outTrendFile.close();
      } else {
	 cout << "suppressing trigger output at break in data, expected " 
	      << lasttime << " found " << t0 << endl;
	 for(channel_iter i=mChanList.begin(); i != mChanList.end(); ++i) {
	    i->prefilter.reset();
	    if(i->lpefilter)
	       i->lpefilter->reset();
	    event_vect events = i->process(getDacc());
	 }
	 // reset rate flag counts
	 for(rateflag_iter i=mRateFlagList.begin(); i != mRateFlagList.end(); i++) {
	    for(int j = 0; j < 2*i->nStride + 1; j++) {
	       i->counts[j] = -1;
	    }
	 }
      }
      lasttime = getDacc().getCurrentTime(); 
   } else {
      Interval wait = stride - Interval(rem);
      Time t2 = t0 + wait;
      cout << "aligning with stride skipping " << wait 
	   << " seconds to " << t2 << endl;
      getDacc().seek(t2);
      lasttime = Time(0);
   }
   time ( &rawtime );
   timeinfo = gmtime ( &rawtime );
   cout << "# finished processing stride - " << asctime(timeinfo);
}

//======================================  Event null constructor
Event::Event(const std::string& name)
   : startTime(0), endTime(0), centralTime(0), frequency(0),
     energy(0), normEnergy(0), npts(0), significance(0),
     channelName(name)
{}

//======================================  Add an element
Event&
Event::operator += (const Element& el) {
   if (npts++) {
      startTime = min(startTime, el.startTime);
      endTime   = max(endTime, el.endTime);
   } else {
      startTime = el.startTime;
      endTime   = el.endTime;
   }
   double norm = el.normEnergy;
   centralTime += norm * (el.startTime + el.endTime)/2.;
   frequency += el.scale * norm;
   energy += el.energy;
   normEnergy += norm;
   return *this;
}

//======================================  Calculated event parameters
void
Event::average(const Time& toff, double fs) {
   if (!npts) return;
   //1/centralTime = centralTime / normEnergy;
   centralTime = toff.totalS() + centralTime / normEnergy;
   startTime += toff.totalS();
   endTime   += toff.totalS();
   //1/----------------------------------------
   significance = -gammqln(npts/2.0, normEnergy/2.0);
   frequency = fs / pow(2.0, frequency / normEnergy);
}

//======================================  Event printout operator.
std::ostream&
operator<<(std::ostream& output, const Event& a) {
   char line[256];
   sprintf(line, "%16.6f  %16.6f  %16.6f  %4.0f  %9.3e  %6.2f  %3d  %6.2f  %s",
	   a.startTime, a.endTime, a.centralTime, a.frequency, a.energy,
	   a.normEnergy, a.npts, a.significance, a.channelName.c_str());
   output << line << endl;
   return output;
}
