#include "gamma.hh"
#include <iostream>
#include <iomanip>
#include <stdlib.h>
#include <math.h>

/****
Numerical Recipies adaptations by Tom Loredo (loredo@spacenet.tn.cornell.edu)
gserln, gcfln, gammpln, gammqln added by Lindy Blackburn (lindy@mit.edu)
poisson significance can be calculated: -gammpln(measured number, expected) = -log(1 - poisscdf(measured - 1, expected) in matlab
chisq significance can be calculated: -gammqln(degrees-of-freedom/2, measurement/2) = -log(1 - chi2cdf(measurement, dof)) in matlab
matlab examples can be calculated more accurtely with gamminc, though the log functions are necessary for very small probabilities
port to C++ by Junwei Li (lijw04@gmail.com)
****/

using namespace std;
double rt2=sqrtl(2.0);
double gammln_cof[6]={76.18009173,-86.50532033,24.01409822,-1.231739516e0,0.120858003e-2,-0.536382e-5};
double gammln_stp=2.50662827465;


double gammln(double xx)  //Logarithm of the gamma function.
{
	double x=xx-1.0;
	double tmp=x+5.5;
	tmp=(x+0.5)*logl(tmp)-tmp;
	double ser=1.0;
	for(int j=0;j<6;j++)
	{
		x=x+1.0;
		ser=ser+gammln_cof[j]/x;
	}
	return tmp+logl(gammln_stp*ser);
}

double gser(double a,double x,int itmax,double eps)  //Series approximation to the incomplete gamma function.
{
	double gln=gammln(a);
	if(x<0.0)
	{
	cout<<"Bad argument: x."<<endl;
	return 0.0;
	}
	if(x==0.0)
	return 0.0;
	double ap=a;
	double sum=1.0/a;
	double delta=sum;
	for(int n=1;n<=itmax;n++)
	{
		ap=ap+1;
		delta=delta*x/ap;
		sum=sum+delta;
		if(fabsl(delta)<fabsl(sum)*eps)
			return sum*expl(-x+a*logl(x)-gln);
	}
	cout<<"max_iters:"<<"("<<fabsl(delta)<<","<<fabsl(sum)*eps<<")"<<endl;
	return 0.0;
}

double gserln(double a,double x,int itmax,double eps)  //Series approximation to the logarithm of incomplete gamma function.
{
	double gln=gammln(a);
	if(x<0.0)
	{
		cout<<"Bad argument: x."<<endl;
		return 0.0;
	}
	if(x==0.0)
	return 0.0;
	double ap=a;
	double sum=1.0/a;
	double delta=sum;
	for(int n=1;n<=itmax;n++)
	{
		ap=ap+1;
		delta=delta*x/ap;
		sum=sum+delta;
		if(fabsl(delta)<fabsl(sum)*eps)
		return logl(sum)+(-x+a*logl(x)-gln);
	}
	cout<<"max_iters:"<<"("<<fabsl(delta)<<","<<fabsl(sum)*eps<<")"<<endl;
	return 0.0;
}

double gcf(double a,double x,int itmax,double eps)  //Continued fraction approximation of the incomplete gamma function.
{
	double gln=gammln(a);
	double g = 0;
	double gold=0.0;
	double a0=1.0;
	double a1=x;
	double b0=0.0;
	double b1=1.0;
	double fac=1.0,anf;
	double an,ana;
	for(int n=1;n<=itmax;n++)
	{
		an=n;
		ana=an-a;
		a0=(a1+a0*ana)*fac;
		b0=(b1+b0*ana)*fac;
		anf=an*fac;
		a1=x*a0+anf*a1;
		b1=x*b0+anf*b1;
		if(a1!=0.0)
		{
			fac=1.0/a1;
			g=b1*fac;
			if(fabsl((g-gold)/g)<eps)
				return (g*expl(-x+a*logl(x)-gln));
			gold=g;
		}
	}
	cout<<"max_iters:"<<fabsl((g-gold)/g)<<endl;
	return 0.0;
}

double gcfln(double a,double x,int itmax,double eps)  //Continued fraction approximation of the logarithm of incomplete gamma function.
{
	double gln=gammln(a);
	double g = 0;
	double gold=0.0;
	double a0=1.0;
	double a1=x;
	double b0=0.0;
	double b1=1.0;
	double fac=1.0,anf;
	double an,ana;
	for(int n=1;n<=itmax;n++)
	{
		an=n;
		ana=an-a;
		a0=(a1+a0*ana)*fac;
		b0=(b1+b0*ana)*fac;
		anf=an*fac;
		a1=x*a0+anf*a1;
		b1=x*b0+anf*b1;
		if(a1!=0.0)
		{
			fac=1.0/a1;
			g=b1*fac;
			if(fabsl((g-gold)/g)<eps)
				return (logl(g)+(-x+a*logl(x)-gln));
			gold=g;
		}
	}
	cout<<"max_iters:"<<fabsl((g-gold)/g)<<endl;
	return 0.0;
}

double gammp(double a,double x)  //lower incomplete gamma function
{
	if(a==0)
		return 1.0;
	if(x<0.0||a<=0.0)
		cout<<"ValueError:"<<"("<<a<<","<<x<<")"<<endl;
	if(x<(a+1.0))
		return gser(a,x);
	else
		return 1.0-gcf(a,x);
}

double gammpln(double a,double x)  //the logarithm of lower incomplete gamma function
{
	if(a==0)
		return 0.0;
	if(x<0.0||a<=0.0)
		cout<<"ValueError:"<<"("<<a<<","<<x<<")"<<endl;
	if(x<(a+1.0))
		return gserln(a,x);
	else
		return logl(1.0-gcf(a,x));
}

double gammq(double a,double x)  //upper incomplete gamma function
{
	if(a==0)
		return 0.0;
	if(x<0.0||a<=0.0)
		cout<<"ValueError:"<<"("<<a<<","<<x<<")"<<endl;
	if(x<(a+1.0))
		return 1.0-gser(a,x);
	else
		return gcf(a,x);
}

double gammqln(double a,double x)  //the logarithm of upper incomplete gamma function
{
	if(a==0)
	{
		cout<<"Logarithm is infinite."<<endl;
		return 0.0;
	}
	if(x<0.0||a<=0.0)
		cout<<"ValueError:"<<"("<<a<<","<<x<<")"<<endl;
	if(x<(a+1.0))
		return logl(1.0-gser(a,x));
	else
		return gcfln(a,x);
}

