/* -*- mode: c++; c-basic-offset: 4; -*- */
//
//    File: dvTest
//
//    Bit Test monitor class definition.
//
#include "DMTBase.hh"
#include "Time.hh"
#include "TrigClient.hh"
#include <list>
#include <iosfwd>
#include <string>

class ChanDV;
class Time;

/**  @name dvTest
  *  dvTest prints out times (resolutiuon 1 s) with dataValid on a 
  *  specified channel.
  *
  *  The dvTest command line syntax is as follows:
  *
  *  \begin{verbatim}
         dvTest [-conf <cfile>] [-log] [-trig] [-raw]
     \end{verbatim}
  *
  *  Where:
  *  <table>
  *  <tr>
  *      <td>-conf <cfile> </td>
  *      <td>Specify the configuration file name [dvTest.cfg].</td>
  *  </tr>
  *  <tr>
  *      <td>-log </td>
  *      <td>Indicates that dvTest is to log all errors found.</td>
  *  </tr>
  *  <tr>
  *      <td>-raw </td>
  *      <td>All channels are in FrAdcData structures.</td>
  *  </tr>
  *  <tr>
  *      <td>-trig </td>
  *      <td>Enable trigger generation.</td>
  *  </tr>
  *  </table>
  *
  *  The configuration file contains one channel name per line plus options:
  *
  *  <table>
  *  <tr>
  *      <td>-[no]log </td>
  *      <td>Indicates that dvTest is to log errors on this channel.</td>
  *  </tr>
  *  <tr>
  *      <td>-procdata </td>
  *      <td>This channel is in a FrProccData structures.</td>
  *  </tr>
  *  <tr>
  *      <td>-[no]trigger </td>
  *      <td>Enable trigger generation for this channel.</td>
  *  </tr>
  *  </table>
  *
  *  @memo Look for dataValid errors in frames.
  *  @author J. Zweizig 
  *  @version 1.0; Modified August 10, 2010
  */
//@{
//@}

//======================================  Monitor class definition
class dvTest : public DMTBase {
public:
    dvTest(int argc, const char *argv[]);
    ~dvTest();

    //----------------------------------  Print channel status information
    void PrintStatus(std::ostream& out) const;

    void ProcessFrame(DMTBase::frame_ptr_type frame);
    bool ReadConfig(void);

private:
    std::string mConfig;
    std::string mStatusFile;
    typedef std::list<ChanDV> ChanList;
    typedef ChanList::iterator channel_iter;
    typedef ChanList::const_iterator const_chan_iter;
    ChanList mAdcChan;
    ChanList mProcChan;
    TrigClient mTrig;
    bool mGblLog;
    bool mGblTrig;
    bool mRawData;
    Time mLastPrint;
};

//
//    ChanBit Class
//
//    The ChanBit class carries the status of each channel to be followed
//    by the bit tester.
//
class ChanDV {
public:
    ChanDV(const char *chan, TrigClient& tc);
    ~ChanDV(void);

    //----------------------------------  Print status page.
    static void PrintHeader(std::ostream& out);
    void PrintStatus(std::ostream& out) const;

    //----------------------------------  Scan the data
    void Scan(fradcdata_pointer  data, const Time& t0);
    void Scan(frprocdata_pointer data, const Time& t0);
    void Scan(frvect_pointer     data, const Time& t0);
    void segment(const Time& t0, Interval dT, short val);
    const char* getChannel(void) const;
    void setTrigEnable(bool te);
    void setLog(bool le);
    void writeSegment(void);

public:
    std::string mChanName;
    bool mTrigEnable;
    bool mLogEnable;
    TrigClient& mTrig;
    short  mCurVal;
    Time   mStart;
    Time   mCurrent;

    //-----------------------------------  Statistics.
    long mCount;
    Time mLast;
};

inline const char*
ChanDV::getChannel(void) const {
    return mChanName.c_str();
}
