//
//    SixtyHertzMon Processing Routine
//    Author: Junyi Zhang (jyzhang@umich.edu)
#include "SixtyHertzMon.hh"
#include "FilterDesign.hh"
#ifndef __CINT__
      
//-->  The next three lines are needed if you are going to generate triggers.
//     The text in PIDTITLE should describe the monitor function.
#define PIDCVSHDR "$Header: https://redoubt.ligo-wa.caltech.edu/svn/gds/trunk/Monitors/SixtyHertzMon/SixtyHertzMon.cc 6294 2010-08-13 03:18:16Z john.zweizig@LIGO.ORG $"
#define PIDTITLE  "Sixty Hertz line monitor"
// #include "ProcIdent.hh"

#include "Dacc.hh"
#include <iostream>
#include <fstream>
#include <string>
#include <stdlib.h>
#include "Hanning.hh"
#include "fft.hh"
#include <math.h>
#include <fstream>

using namespace std;

EXECDAT(DatTemplate)


#endif               // !def(__CINT__)


//======================================  Skeleton object constructor.
DatTemplate::DatTemplate(int argc, const char *argv[]) 
  : DatEnv(argc, argv), maxFrame(9),mStep(60.00), mPipe(0), mstep(60.0),
     nfile(1) , mOSC(getDacc()) , ncf(0),TotalLoopnum(1)

{ 
    //----------------------- parameters initilization
    db=0;
    nhf=0;
    NB=1000.0;
    bool dvIndex = true;
    TrendValue   =0.0;
    DispOnline   = 1;
    for(int i=0;i<10;i++) tspd[i] = 0.0;
    current_month= "NA";
    string version = "debug";
    char* html_dir = "./";
    htmldir        = html_dir;
    if( (html_dir = getenv("DMTHTMLOUT")) ) {
      if( strlen(html_dir) ){
	if( html_dir[strlen(html_dir)-1] == '/' )
	  htmldir = html_dir;
	else
	  { htmldir = html_dir;htmldir=htmldir+"/";}       
      }
    }
    else 
      cout << "environmental variable DMTHTMLOUT unset, set to default directory ./" << endl;
    
    SignalChannel      = "H1:LSC-DARM_ERR";  //the signal channel

    //-------------------read arguments from command line
    string cfstr[10];
    bool syntax    = false;
    for (int i=1 ; i<argc; i++) {
        string argi = argv[i];
        if (argi == "-c") {
	    SignalChannel = argv[++i];
	} else if (argi == "-dvindex") {
	    dvIndex = true;
	} else if (argi == "-filter") {
	    mFilter = argv[++i];
	} else if (argi == "-maxframes") {
	    maxFrame = strtol(argv[++i], 0, 0);
	} else if (argi == "-db") {
	    db = strtol(argv[++i], 0, 0);
	} else if (argi == "-t") {
	    mStep=mstep = strtod(argv[++i], 0);
	} else if (argi == "-settle") {
	    mSettle = strtod(argv[++i], 0);
 	} else if (argi == "-cf") {
            cfstr[ncf] =argv[i+1];
            cf[ncf] = strtod(argv[++i],0);
            ncf++;
        } else if (argi == "-ver") {
	    version=argv[++i];
        } else if (argi == "-N") {
	    NB=strtod(argv[++i], 0);
        } else if (argi == "-html") {
	    htmldir=string(argv[++i]);
        } else if (argi == "-hf") {
            hfstr[nhf] =argv[i+1];
	    HeteroFre[nhf] = strtod(argv[++i],0);
            nhf++;
        } else if (isDatEnvArg(argv[i])) {
	    i++;
	} else {
	    syntax = true;
	    cerr << "Unrecognized argument: " << argi << endl;
	}
    }
    
    //-----------------------  Bail out if command line not correct
    if (syntax) {
        cerr << "Command syntax:" << endl;
	cerr << argv[0] 
	     << " [-c <Signal Channel>] [-frames <nMax>] [-t <Segment Length>] \\" 
	     << endl
	     << "      [-filter <filter>] [-settle <s-time>] [-dvIndex] \\"
	     << endl
             << "      [-cf <frequencies bands to display>] [-hetfre <Heterodyne Frequency> ] \\"
             << endl
             << "      [-html <html output directory> ] [-N <recent-days number> ] \\"
             << endl
	     << "      [<DatEnv-args>]" 
	     << endl;
	finish();
	return;
    }
    cout<<"Signal Channel:" << SignalChannel << endl;

    //----------------------------------  Make sure there's a valid channel
    if (SignalChannel[0]== 'H');   
      VoltageChannel      = "H0:PEM-LVEA2_V1";
    if (SignalChannel[0]== 'L')
      VoltageChannel      = "L0:PEM-LVEA_V1";
    if (SignalChannel.substr(0,2)== string("H1")) cali_fre=46.70;
    if (SignalChannel.substr(0,2)== string("H2")) cali_fre=54.10;
    if (SignalChannel.substr(0,2)== string("L1")) cali_fre=54.70;
    BothArmLocked = SignalChannel.substr(0,2)+string(":Both_arms_locked_strict_cm");
    string SigSub=SignalChannel.substr(0,2); 
 
    HtmlFilename  = htmldir+"index.html";
    cout <<"HtmlFilename:" << HtmlFilename << endl;

    //----------------------------------  set the data accessor
    getDacc().setStride(mStep);  
    getDacc().addChannel(VoltageChannel.c_str());
    getDacc().addChannel(SignalChannel.c_str());
    getDacc().setIgnoreMissingChannel(true);
   
    
    //----------------------------------  Specify DMT Viewer channel names
  
    string title_dmtviewer = "SixtyHertzMon";
    title_dmtviewer        = title_dmtviewer+"_"+version;
    title_dmtviewer.c_str();
    setServerName( title_dmtviewer.c_str() );
    mSigmaName = "H1:DMT-SXTY_60Hz_Freq";
    cout << "Trying to define trend channel with name" << mSigmaName << endl;
  
    //----------------------------------  Set up the trender
    string trendfilename = "SixtyHertzMon_";
    trendfilename = trendfilename+SignalChannel.substr(0,2);
    mTrend.setName( trendfilename.c_str() );    //  Set trend frame name
    mTrend.setFrameCount(1);          //  Set frames per file
    mTrend.setType(Trend::kMinute);   //  Set trend type (minute trends)
    mTrend.setAutoUpdate(false);      //  Avoid automatic screwup.
    mTrend.addChannel(mSigmaName.c_str());
 
    if (dvIndex) {
      string filename;
      const char* dmtout = getenv("DMTOUTPUT");
      if (dmtout) {
	filename = dmtout;
	filename += "/";
      }
      filename += "channel.cfg";
      mTrend.writeIndex(filename.c_str());
    }
    
    for(int hfindex=0;hfindex<nhf;hfindex++) {
    
       myObj[hfindex]=SixtyHertzMon(HeteroFre[hfindex],db,ncf,VoltageChannel,SignalChannel, NB,cf,HeteroFre);
     
       //---------------------------------   setup the dmtviewer    
       //Signal channels
       for (int i=0;i<ncf;i++) {
          std::stringstream sstr1,sstr2;
          sstr1 << cf[i]-5.0+HeteroFre[hfindex]-HeteroFre[0];
          sstr2 << cf[i]+5.0+HeteroFre[hfindex]-HeteroFre[0];
          std::string  sufix1 = sstr1.str();
          std::string  sufix2 = sstr2.str();
          string name = SignalChannel+"_NHTR_"+hfstr[hfindex]+sufix1+"_"+sufix2; 
          serveData(name.c_str(), &myObj[hfindex].UnHeteroedSigView[i], 0);
          name = SignalChannel+"_"+hfstr[hfindex]+sufix1+"_"+sufix2;
          serveData(name.c_str(),&myObj[hfindex].HeteroedSigView[i],0);
       }
    }     


    //----------------make necessary directories
    string command;
    command = "mkdir ";
    command = command+htmldir+"history/";
    system(command.c_str());
    //----------------setup lock status indicator
    cout << " call mosc.setstride with mstep:"<<mstep<<endl; 
    mOSC.ignoreAllExcept(SigSub);
    mOSC.setStride(int(mstep));
    mOSC.readConfig("LockLoss.conf");
    cout <<"contructor complete"<<endl;
}

//======================================  Skeleton object destructor.
DatTemplate::~DatTemplate() 
{   
    cout << "SixtyHertzMon is finished" << endl;
}

//======================================  Frame processing function.
void
DatTemplate::ProcessData(void) {
  
    if (db >= 1) cout << "loop " << TotalLoopnum << " begins. " << endl;
    //----------------------------------  Get pointers to the current data.
    TSeries* ts = getDacc().refData(VoltageChannel.c_str());
    TSeries* ts2 = getDacc().refData(SignalChannel.c_str());
    TSeries ts3=*ts2;
    if (!ts || ts->isEmpty()) {
        cout << "Channel: " << VoltageChannel << " was not found." << endl;
        return;
    }
    DecimateBy2 lowfilter(3);          //filter and downsample the singal to cut off high frequency part 
    SingalTSeries[1]  = lowfilter.apply(ts3);  //SingalTSeries[] contain the signal channel data
    VoltageTSeries[1] = *ts;                   //VoltageTSeries[] contain the voltage channel data 
    if (db >= 1) cout <<"loop start time:" << ts->getStartTime().getS()
		      << endl;
    Tnow = ts->getStartTime();

    Hanning win(20); 

    //----------------------create a new file for a new day
    if (newfile()) {
      if (db >= 1) cout<< "newfile:"<<1<<endl;
      //make a new folder for a new month if neccessary
      if (newfolder()) {
	string command;
	command = "mkdir ";
	command = command+htmldir+"history/"+HistFolder;
	system(command.c_str());
      }

      for(int hfindex=0;hfindex<nhf;hfindex++) {
         //decide the file name for the new file
         std::ofstream current;
         std::ofstream history;
         char utc[40];
         TimeStr(Tnow+Interval(8*3600),utc,"%d");
         string day(utc);
         if (day.length()==1) day=string("0")+day;
         TimeStr(Tnow+Interval(8*3600),utc,"%m");
         string mon(utc);      
         if (mon.length()==1) mon=string("0")+mon;
         TimeStr(Tnow+Interval(8*3600),utc,"%Y_");
         mon=string(utc)+mon+"_"+day;
         historyfile = htmldir+"history/"+HistFolder+"/"+mon+"_"+hfstr[hfindex]+"Hz"+".html";
         myObj[hfindex].historyfile=historyfile;
      
         //initilize the new html file,history file and current file simultaneously
         history.open(historyfile.c_str());
         if (!history.is_open()) cout<<"can't open the file"<<endl;
         history <<"<head><title>SixtyHerzMon History Data</title></head>"<<endl;
         history <<"<body bgcolor= \"wheat\">"<<endl;
         history <<"<center><big>Channel Name: "<<SignalChannel<<" with heterodyne frequecy:"<<hfstr[hfindex]<<"Hz"<<"</big></center>"<<endl;
         history <<"<br><br>"<<endl;
         history.close();
   
         string currentfile = htmldir+"current_"+hfstr[hfindex]+"Hz"+".html";
         myObj[hfindex].currentfile=currentfile;
         current.open(currentfile.c_str());
         if (!current.is_open()) cout<<"can't open the file"<<endl;
         current <<"<head><title>SixtyHerzMon Current Data</title></head>"<<endl;
         current <<"<body bgcolor= \"wheat\">"<<endl;
         current <<"<center><big>Channel Name: "<<SignalChannel<<" with heterodyne frequecy:"<<hfstr[hfindex]<<"Hz"<<"</big></center>"<<endl;
         current <<"<br><br>"<<endl;
         current.close();
      }
    }

    for(int hfindex=0;hfindex<nhf;hfindex++) myObj[hfindex].LoopInit(Tnow,TotalLoopnum,cali_fre);
    bool locked;
    if ( (locked=mOSC.satisfied(BothArmLocked.c_str())) )
      {            
        //-----------------------------------------------the voltage channel
        if (db >= 1) cout <<"now for the voltage channel"<<endl;
	int sampf = int(1.0/VoltageTSeries[1].getTStep());
	PadLen    = 1024;   //the length of data segment after zero padding
	VoltageTSeries[0] = TSeries(VoltageTSeries[1].getStartTime()-Interval((PadLen-int(mstep))/2),Interval(1.0/sampf),5,tspd);
	VoltageTSeries[0].extend(VoltageTSeries[1].getStartTime());
	VoltageTSeries[2] = win.apply(VoltageTSeries[1]);
	TSeries  joints = VoltageTSeries[0];   //the voltage channel time series after zero padding
 	joints.Append( VoltageTSeries[2] );    
	joints.extend( joints.getStartTime() + Interval(PadLen) );
        if (db >= 1) cout <<"joints.getInterval:"<<joints.getInterval()<<" joints.getMax()"
                          <<joints.getMaximum()<<" mstep:"<<int(mstep)<<endl;
	ff      = FSeries(joints);
	ss      = FSpectrum(ff);       //the power spectrum of the zero-padded voltage time series

        for(int hfindex=0;hfindex<nhf;hfindex++) myObj[hfindex].CalVolFreq(ss);
        if ( abs(myObj[0].volfreq-HeteroFre[0])<0.5 ) {
           TrendValue =myObj[0].volfreq;       
	   mTrend.trendData(mSigmaName.c_str(), SingalTSeries[1].getStartTime(), TrendValue);
	   mTrend.Update(); 

        //----------------------------------------------DARM_ERR channel     
	   sampf    = int(1.0/SingalTSeries[1].getTStep());
	   PadLen   = 512;
	   if (db >= 1) cout << "samf:"<<sampf<<endl;
	   SingalTSeries[0] = TSeries(SingalTSeries[1].getStartTime()-Interval((PadLen-int(mstep))/2),Interval(1.0/sampf),5,tspd);
	   SingalTSeries[0].extend(SingalTSeries[1].getStartTime());
	   SingalTSeries[2] = win.apply(SingalTSeries[1]);
	   joints   = SingalTSeries[0];    // the zero-padded signal time series
	   joints.Append(SingalTSeries[2]);
	   joints.extend( joints.getStartTime() + Interval(PadLen) );
	   if (db >= 1) cout <<"joints.getInterval:"<<joints.getInterval()<<" joints.getMax()"<<joints.getMaximum()<<" mstep:"<<int(mstep)<<endl;
	   ff = FSeries(joints);
	   ss = FSpectrum(ff);
 
	   for(int hfindex=0;hfindex<nhf;hfindex++) myObj[hfindex].Heterodyne(ss,PadLen);
	}
	else { locked=0;cout<<"voltage channel problem"<<endl;}

      }
      
    else  
      //cout <<"UNLOCKED"<<endl;

    for(int hfindex=0;hfindex<nhf;hfindex++) myObj[hfindex].SidebandSeek(PadLen,locked);
    
    //------------update index file,regardless if the last data segment is good or not
    char buf[40];
    char utc[40];
    indexfile.open(HtmlFilename.c_str());
    if (!indexfile.is_open()) cout<<"can't open the file"<<endl;
    indexfile <<"<head><title>SixtyHerzMon index</title></head>"<<endl;
    indexfile <<"<body bgcolor= \"wheat\">"<<endl;
    indexfile <<"<center><big>SixtyHerzMon Statistics Index</big></center>"<<endl;
    indexfile <<"<center>Last updated:</center><center>GPStime:"<<long(Tnow.totalS())<<"</center><center>";
    string timezone;
    if ( SignalChannel[0] == 'H' ) timezone="PT";
    else timezone="CT"; 
    LocalStr(Tnow,buf,"%w");
    if (db >=1) cout <<"buf:  "<<buf<<endl;
    string dayofweek;
    if (buf[0]=='1') dayofweek="Mon";
    if (buf[0]=='2') dayofweek="Tue";
    if (buf[0]=='3') dayofweek="Wed";
    if (buf[0]=='4') dayofweek="Thu";
    if (buf[0]=='5') dayofweek="Fri";
    if (buf[0]=='6') dayofweek="Sat";
    if (buf[0]=='0') dayofweek="Sun";
    indexfile <<" LocalTime:"<<dayofweek<<LocalStr(Tnow,buf," %M %d, %Y at %H:%N:%S");
    // cout<<"local time:"<<LocalStr(Tnow,buf," %M %d, %Y at %H:%N:%S")<<" utc time:"<<TimeStr(Tnow,utc," %M %d, %Y at %H:%N:%S")<<endl;
    TimeStr(Tnow,utc," %M %d, %Y at %H:%N:%S");
    indexfile <<" "<<timezone;
    indexfile <<"</center>"<<endl;
    indexfile <<"<center>UTC Time:  "<<utc<<"</center><br>"<<endl;
    indexfile.close();

    for (int hfindex=0;hfindex<nhf;hfindex++) {
      indexfile.open(HtmlFilename.c_str(),ios::out|ios::app);
       indexfile <<"<center><big></B>Heterodyne Frequency:"<<hfstr[hfindex]<<"Hz"<<"</B></big></center>"<<endl;
       indexfile <<"<center>goodloop ratio in past 1 hour:";       //calculate the ratio of good data segments in the last 1,6,12 and 24 hours 
       int retro,span;
       if (TotalLoopnum < 60) { retro=0; span=TotalLoopnum; }
       else { retro=(TotalLoopnum-60)%1441; span=60; }
       indexfile << float(myObj[hfindex].GoodloopNum-myObj[hfindex].goodloop[retro])<<"/"<< float(span) <<"</center>"<<endl;  
       indexfile <<"<center>goodloop ratio in past 6 hour:";
       if (TotalLoopnum < 360) { retro=0;span=TotalLoopnum; }
       else { retro=(TotalLoopnum-360)%1441; span=360; }
       indexfile << float(myObj[hfindex].GoodloopNum-myObj[hfindex].goodloop[retro])<<"/"<< float(span) <<"</center>"<<endl;
       indexfile <<"<center>goodloop ratio in past 12 hour:";
       if (TotalLoopnum < 720) { retro=0;  span=TotalLoopnum; }
       else { retro=(TotalLoopnum-720)%1441; span=720; }
       indexfile << float(myObj[hfindex].GoodloopNum-myObj[hfindex].goodloop[retro])<<"/"<< float(span)<<"</center>"<<endl;
       indexfile <<"<center>goodloop ratio in past 24 hour:";
       if (TotalLoopnum < 1440) { retro=0; span=TotalLoopnum; }
       else { retro=(TotalLoopnum-1440)%1441; span=1440; }
       indexfile << float(myObj[hfindex].GoodloopNum-myObj[hfindex].goodloop[retro])<<"/"<< float(span) <<"</center>"<<endl;

       string url,linkaddr;
       string linktext = "current data";
       url = "http://www.ligo-wa.caltech.edu/";
       if (DispOnline)
         linkaddr = string("./current_")+hfstr[hfindex]+"Hz.html";
       else
         linkaddr = htmldir+"current.html";
       indexfile <<"<center><a href=\""<<linkaddr<<"\">"<<linktext<<"</a></ceter>"<<endl;
       indexfile.close();

       linktext = "history data";
       if (DispOnline)
         linkaddr = "./history/";
       else
         linkaddr = htmldir+"history";
       indexfile.open(HtmlFilename.c_str(),ios::out|ios::app);
       indexfile <<"<center><a href=\""<<linkaddr<<"\">"<<linktext<<"</a></ceter>"<<endl;
       indexfile <<"<br><br>"<<endl;
       indexfile.close();
    }
 
    if (db >= 1) {
       cout <<"loop ends"<<endl;
       cout <<endl;
    }
       TotalLoopnum++;
}

//===========================================
SixtyHertzMon::SixtyHertzMon () {}


//===========================================
SixtyHertzMon::SixtyHertzMon(float hf,int a,int b,string c,string d,float e,float* g,float* h) {
    HeteroFre=hf;
    TotalLoopnum=1; 
    InvalidNum=0;
    GoodloopNum=0;
    NoiseEffectNum=0;
    Q=0.82; 
    NB        = 1000.0;      //The number of most recent days whose data make the major contribution to the spectrum averaging   
    reloldwght_sum = 0.0;    //weight for time averging
    oldwght = 0.0;           //weight for noise level averging
    newght    = 0.0;         //weight for noise level averging
    for (int i=0;i<10;i++) check_sum[i] = check(Q);
    check_std    = check(Q);
    check_sdnum  = check(Q-0.2);
    blast        = 0; 
    goodloop[0]  = 0; 
    for(int i=0;i<40000;i++) HeteroedSignal[i]=UnHeteroedSignal[i]=0.0;
    db=a;
    ncf=b;
    VoltageChannel=c;
    SignalChannel=d;
    NB=e;
    for(int cfindex=0;cfindex<ncf;cfindex++) cf[cfindex]=g[cfindex]+HeteroFre-h[0];
   }


//===========================================
void SixtyHertzMon::LoopInit(Time t,long l,float cali) {
    gooddata = 0;
    NoiseEffect=0;
    TotalLoopnum=l;
    cali_fre=cali;
    Tnow=t;
    if ( (GoodloopNum>NB) || (GoodloopNum==0) ) expfactor=-1.0/NB;
    else expfactor=-1.0/GoodloopNum;
    oldwght_bckup = oldwght;
    reloldwght_sum_bckup = reloldwght_sum;
    reloldwght_sum = reloldwght_sum*exp(expfactor);
    N            = 1.0/(reloldwght_sum+1.0);
    reloldwght_sum = reloldwght_sum+1.0;    
    ite = 0;
    for (int i=0;i<10;i++) dat1[i] = 0.0;
    nsidebands = 0;   
}


void SixtyHertzMon::CalVolFreq(FSpectrum& ss) {
    display = ss.extract(HeteroFre-3.0,6.0); //the spectrum of voltage around hetrodyne frequency
    display.setT0(Tnow);
   
    //---------------------------------------------calculating the frequency from voltage channel
    j1 = display.getData(display.getNStep()+1,dat1);
    float max = 0.0,
          sum = 0.0,
          totweight = 0.0,
          position;
    int maxi = 0;
    for(int i=0;i<j1;i++) if ( dat1[i]>max ) { max=dat1[i]; maxi=i; }
    for(int i=0;i<21;i++) { sum = sum+dat1[maxi-10+i]*(maxi-10+i);
                       totweight = dat1[maxi-10+i]+totweight; 
    }
    position = sum/totweight;
    volfreq  = display.getBinF(int(position))*(int(position)+1-position)+display.getBinF(int(position)+1)*(position-int(position));
    if (db >= 1) cout <<"j1:"<<j1<<" display.getNStep:"
                 <<display.getNStep()<<" volfreq:"<<volfreq<<endl;

}

//===============================================
void SixtyHertzMon::Heterodyne(FSpectrum& ss,int PadLen) {
	spechet = ss.extract(HeteroFre-30.0,60.0); //the whole heterodyned averaged spectrum
	float  freqdiff = volfreq-HeteroFre;
	int bindiff;
	int cfindex;
	j3 = spechet.getData(spechet.getNStep()+1,dat3);
	bool isvalid=1;
          
        if (db >=2) for(int i=0;i<j3;i=i+20) cout<<30.0+float(i)/PadLen<<":"<<dat3[i]<<endl;
        //---------calculating the noise level 
        //The whole spectrum( with range of 60Hz) is divided into 20 sub-bands,each band has freqency range of 2 Hz
	float noisesum = 0.0;
	for (int i=5;i<25;i++) {
	  int ii;
	  int ledge,redge;
	  ledge=peakedge(&dat3[100+i*2*PadLen],2*PadLen,-1);
	  redge=peakedge(&dat3[100+i*2*PadLen],2*PadLen,+1);
	  for (ii=100+i*2*PadLen;ii<100+i*2*PadLen+ledge;ii++) noisesum=noisesum+dat3[ii];
	  for (ii=100+i*2*PadLen+redge;ii<100+(i+1)*2*PadLen;ii++) noisesum=noisesum+dat3[ii];
	}
	if (db >= 1) cout<<" noise level sum:"<<noisesum<<endl;
    
        //--------check the statistical validity of spectrum sum 
	for (int i=0;i<10;i++) {
	  float sum;
	  int start,end;
          int cut_start,cut_end;
          start= PadLen+i*5*PadLen;
          end =start + 5*PadLen;
          cut_start=int((cali_fre-0.15-HeteroFre+30.0)*float(PadLen));
	  if (db >=2 ) cout<< "cut_start:"<<cut_start<<endl;
          cut_end=int((cali_fre+0.15-HeteroFre+30.0)*float(PadLen));
          if (cut_start<start) cut_start=start;
          if (cut_end>end) cut_end=end;
      	  sum=arraysum(&dat3[100+i*5*PadLen],5*PadLen);
          if (cut_end>cut_start) sum=sum-arraysum(&dat3[cut_start],cut_end-cut_start);
	  // cout << "cut:"<< (cut_end>cut_start)<<" cut_start_fre:"<<HeteroFre-30.0+float(cut_start)/float(PadLen)<<"cut_end_fre:"<<HeteroFre-30.0+float(cut_end)/float(PadLen)<<"cali_fre:"<<cali_fre<<endl;
	  isvalid = isvalid && check_sum[i].datvalid(sum,oldwght,1/noisesum/noisesum,1,db);
	}	     
	//--------check the statistical validity of spectrum standard deviation
	float std =  arraystd(dat3,j3);
	isvalid = isvalid && check_std.datvalid( std,oldwght,1/noisesum/noisesum,1,db );

        if ( isvalid ) {
              for (int i=0;i<10;i++) check_sum[i].onhold = 0;
	      check_std.onhold = 0;
              //-------do the heterodyne manually
	      if ( freqdiff >= 0 )
		{
		  bindiff = int(freqdiff*PadLen+0.5);
		  for(int i=0;i<j3-bindiff;i++) dat1[i] = dat3[i+bindiff];
		  for(int i=j3-bindiff;i<j3;i++) dat1[i] = 0.0;
		}
	      else
		{
		  bindiff = int(freqdiff*PadLen-0.5);
		  for(int i=j3-1;i>(-bindiff-1);i--) dat1[i] = dat3[i+bindiff];
		  for(int i=-bindiff-1;i>=0;i--) dat1[i] = 0.0;
		}
              //-------do the modified exponential averaging
	      for(int i=0;i<j3;i++) {
	        HeteroedSignal_bckup[i] = HeteroedSignal[i];
      		UnHeteroedSignal_bckup[i] = UnHeteroedSignal[i];
                float M=1.0-N;
                newght = 1.0/noisesum/noisesum;          
	        if (0.1*(M/N)<(oldwght/newght)) oldwto1=M/N;
		else { oldwto1=oldwght/newght; NoiseEffect=1;}
      		HeteroedSignal[i]       = dat1[i]/(oldwto1+1)+HeteroedSignal[i]*oldwto1/(oldwto1+1);
		UnHeteroedSignal[i]       = dat3[i]/(oldwto1+1)+UnHeteroedSignal[i]*oldwto1/(oldwto1+1);
	      }
      	      oldwght         = oldwght*exp(expfactor)+newght; 
              if (db >= 1) cout <<"N:"<<N<<endl;
	      GoodloopNum++;
	      gooddata = 1;

              //send the spectrum to dmtviewer for displaying
	      for (cfindex=0;cfindex<ncf;cfindex++)
		{ j2 = 10*PadLen+1;
		  UnHeteroedSigView[cfindex] = ss.extract(cf[cfindex]-5.0,10.0);
		  UnHeteroedSigView[cfindex].setData( j2, &UnHeteroedSignal[int((cf[cfindex]-5.0-(HeteroFre-30.0))*PadLen+0.1)] );
		  UnHeteroedSigView[cfindex].setT0(Tnow);
		  HeteroedSigView[cfindex] = ss.extract(cf[cfindex]-5.0,10.0);
		  HeteroedSigView[cfindex].setData( j2, &HeteroedSignal[int((cf[cfindex]-5.0-(HeteroFre-30.0))*PadLen+0.1)] );
		  HeteroedSigView[cfindex].setT0(Tnow);
		  if (db >= 1) cout <<" bindiff:"<<bindiff<<endl;
		  // if (db == 1) cout <<BothArmLocked<<":"<< mOSC.satisfied(BothArmLocked.c_str());
		}

	}
	else {
	  cout <<"BAD DATA,DISCARDED"<<endl; 

	  for (cfindex=0;cfindex<ncf;cfindex++)
	    { int j2 = 10*PadLen+1;
	      Time Tpast;
              Tpast=HeteroedSigView[cfindex].getStartTime();
	      UnHeteroedSigView[cfindex] = ss.extract(cf[cfindex]-5.0,10.0);
	      UnHeteroedSigView[cfindex].setData( j2, &UnHeteroedSignal[int((cf[cfindex]-5.0-(HeteroFre-30.0))*PadLen+0.1)] );
	      UnHeteroedSigView[cfindex].setT0(Tpast);
              HeteroedSigView[cfindex] = ss.extract(cf[cfindex]-5.0,10.0);
	      HeteroedSigView[cfindex].setData( j2, &HeteroedSignal[int((cf[cfindex]-5.0-(HeteroFre-30.0))*PadLen+0.1)] );
	      HeteroedSigView[cfindex].setT0(Tpast);
	    }

	  cfindex=ncf;InvalidNum++;
	}
		
}

//=========================================
void SixtyHertzMon::SidebandSeek(int PadLen,bool locked) {
    float threshold; 
    if( (GoodloopNum>0) && (gooddata==1) ) {
      //-------try to identify sidebands candidates in the spectrum that are not due to random fluctuation
      int scan,margin;
      margin = 20*int(HeteroFre/60.0);
   
      float fstprtsum1=0.0;
      float sndprtsum1=0.0;
      float fstprtsum2=0.0;
      float sndprtsum2=0.0;
      if (GoodloopNum<NB+1) {
	if (GoodloopNum == 1) sumexp1[GoodloopNum]=sumexp2[GoodloopNum]=0.0;
	if (GoodloopNum == 2) {sumexp1[GoodloopNum]=exp(-1.0); sumexp2[GoodloopNum]=exp(-2.0);}
	if (GoodloopNum > 2) 
	  { sumexp1[GoodloopNum]=(sumexp1[GoodloopNum-1]+1.0)*exp(-1.0/float(GoodloopNum-1));
	  sumexp2[GoodloopNum]=(sumexp2[GoodloopNum-1]+1.0)*exp(-2.0/float(GoodloopNum-1));
	  }
	fstprtsum1=sumexp1[GoodloopNum]+1.0;
	fstprtsum2=sumexp2[GoodloopNum]+1.0;
      }
      else {
	sumexp1[int(NB+1.1)]=(sumexp1[int(NB+0.1)]+1.0)*exp(-1.0/float(NB));
	sumexp2[int(NB+1.1)]=(sumexp2[int(NB+0.1)]+1.0)*exp(-2.0/float(NB));
	fstprtsum1=exp((GoodloopNum-NB)*expfactor)*sumexp1[int(NB+1.1)];
	fstprtsum2=exp((GoodloopNum-NB)*2*expfactor)*sumexp2[int(NB+1.1)];
	sndprtsum1=(1-exp((GoodloopNum-NB)*expfactor))/(1-exp(expfactor));
	sndprtsum2=(1-exp((GoodloopNum-NB)*2*expfactor))/(1-exp(2*expfactor));
      }
      threshold=5.0/( (fstprtsum1+sndprtsum1)/sqrt(fstprtsum2+sndprtsum2) );
      if (db >= 2) cout<<" fstprtsum1,fstprtsum2:"<<  sumexp1[GoodloopNum]<<","<< sumexp2[GoodloopNum]<<endl; 
      for (scan=margin;scan<j3-margin;scan++) {
	if ( ( HeteroedSignal[scan] > HeteroedSignal[scan-1]) &&
	     ( HeteroedSignal[scan] > HeteroedSignal[scan+1]) &&
	     ( HeteroedSignal[scan] >= UnHeteroedSignal[scan]))
	  {
	    int leftedge,rightedge;
	    int counter = scan;
	    while ( (counter<j3-margin) && (HeteroedSignal[counter+1]<HeteroedSignal[counter]) ) counter++;
	    rightedge = counter;
	    counter = scan;
	    while ( (HeteroedSignal[counter-1]<HeteroedSignal[counter]) && (counter>margin) ) counter--;
	    leftedge = counter;
	    float level = 0.5*(HeteroedSignal[leftedge]+HeteroedSignal[rightedge]);

            //if a certain peak is larger than the threshold,then it considered statistical significant
	    if ( (HeteroedSignal[scan]-level)/level > threshold ) {
              
	      float sum = 0.0,
		totweight = 0.0,
		position;
	      for(int i=0;i<11;i++){ sum = sum+HeteroedSignal[scan-5+i]*(scan-5+i);
	      totweight = HeteroedSignal[scan-5+i]+totweight;
	      }
	      position = sum/totweight;
	      peaks[ite] = HeteroFre-30.0+float(position)/float(PadLen);
	      ite++;
	    }
	  }
      }
      if (db >= 1) cout <<endl<<"ite:"<<ite<<endl;
      if (db >= 2) cout<<"threshold:"<<threshold<<endl;

      //--do a check on the statistical validity of the number of peaks found
      bool isvalid;
      isvalid = check_sdnum.datvalid(float(ite),oldwght,newght,0,db);
      if ( !isvalid ) {
	GoodloopNum--;
        gooddata = 0;
        int i;
        blast++;
        cout <<"DISCARDED,SIDEBAND BLAST"<<endl;
        for (i=0;i<j3;i++) { HeteroedSignal[i]=HeteroedSignal_bckup[i]; UnHeteroedSignal[i]=UnHeteroedSignal_bckup[i]; }
      }
     
      if (isvalid||1) { 
	//---try to identify if a certain peak is a sideband   
	for(scan=0;scan<ite;scan++){if (db >= 1) cout <<"peaks["<<scan<<"]:"<<peaks[scan]<<" ";}
        if (db >= 1) cout<<endl;
	for (int i=0;i<ite;i++) {
	  int j = i;
	  while ( (abs((peaks[i]+peaks[j])/2.0-HeteroFre)>0.002*(HeteroFre/60.0)) && (j<ite) ) j++;
	  if (j<ite) {
	    sidebands[nsidebands]      = abs(peaks[i]-HeteroFre);
            dsidebands[2*nsidebands]   = peaks[i];
            dsidebands[2*nsidebands+1] = peaks[j];
	    if (db >=1 ) cout <<"sidebands:"<<sidebands[nsidebands]<<endl;
	    nsidebands++;
	  }
	}
 
        //---update html file content
	if ((GoodloopNum % 1) == 0)
	  { //---update current file
	    std::ofstream current;
	    std::ofstream history;
	    current.open(currentfile.c_str(),ios::out|ios::app);
	    if (!(current.is_open())) cout<<"can't open current file"<<endl;
           
            current <<"<table border align=left><tr><th>GPS TIME</th><th>GoodloopNum</th><th># OF SIDEBANDS</th><th colspan="
                    <<nsidebands-1<<">SIDEBANDS(HZ)</th></tr>"<<endl;
            current <<"<tr><td rowspan=2>"<<long(Tnow.totalS())<<"</td><td rowspan=2>"
		    << long(GoodloopNum)<<","<<long(TotalLoopnum)<<"</td><td rowspan=2>"<<nsidebands*2-2<<"</td>";
	    for (int i=nsidebands-1;i>-1;i--)
              if ( abs(sidebands[i]) > 0.01 )
		current <<"<td>"<<HeteroFre<<"&plusmn;"<<sidebands[i]<<"</td>";
            current <<"</tr><tr>"<<endl;
            
	    for (int i=nsidebands-1;i>-1;i--)
              if ( abs(sidebands[i]) > 0.01 )
                current <<"<td>"<<dsidebands[2*i]<<","<<dsidebands[2*i+1]<<"</td>";
            current <<"</tr>"<<endl;
	    current <<"</table><br><br><br><br><br><br>"<<endl;
            if (db >= 2) cout<< "writing current file"<<endl;
	    current.close();

            //---update history file
            history.open(historyfile.c_str(),ios::out|ios::app);
            if (!(history.is_open())) cout<<"can't open history file"<<endl;      
       	    history <<"<table border align=left><tr><th>GPS TIME</th><th>GoodloopNum</th><th># OF SIDEBANDS</th><th colspan="
                    <<nsidebands-1<<">SIDEBANDS(HZ)</\th></tr>"<<endl;
            history <<"<tr><td rowspan=2>"<<long(Tnow.totalS())<<"</td><td rowspan=2>"
                    << long(GoodloopNum)<<","<<long(TotalLoopnum)<<"</td><td rowspan=2>"<<nsidebands*2-2<<"</td>";
            for (int i=nsidebands-1;i>-1;i--)
              if (abs(sidebands[i])>0.01)
                history <<"<td>"<<HeteroFre<<"&plusmn;"<<sidebands[i]<<"</td>";
            history <<"</tr><tr>"<<endl;

            for (int i=nsidebands-1;i>-1;i--)
              if ( abs(sidebands[i]) > 0.01)
                history <<"<td>"<<dsidebands[2*i]<<","<<dsidebands[2*i+1]<<"</td>";
            history <<"</tr>"<<endl;
            history <<"</table><br><br><br><br><br><br>"<<endl;
            history.close();
	  }
	
      }
    }
    
    if (!gooddata) {
      int i;
      //---if data is bad and discarded,restore the various averaging arguments to original value
      for (i=0;i<10;i++) check_sum[i].restore();
      check_std.restore();
      check_sdnum.restore();
      oldwght = oldwght_bckup;
      reloldwght_sum=reloldwght_sum_bckup;

      //---update current html files
      std::ofstream current;
      std::ofstream history;
      current.open(currentfile.c_str(),ios::out|ios::app);
      if (!(current.is_open())) cout<<"can't open current file"<<endl;
      current <<"<table border align=left><tr><th>GPS TIME</th><th>STATUS</th>"<<endl;
      current <<"<tr><td>"<<long(Tnow.totalS())<<"</td><td>";
      if ( !locked ) current <<	"UNLOCKED</td>";
      else current << "BAD DATA</td>";
      current <<"</tr>"<<endl;
      current <<"</table><br><br><br><br><br><br>"<<endl;
      current.close();
      //---update history html file
      history.open(historyfile.c_str(),ios::out|ios::app);
      if (!(history.is_open())) cout<<"can't open history file"<<endl;
      history <<"<table border align=left><tr><th>GPS TIME</th><th>STATUS</th>"<<endl;
      history <<"<tr><td>"<<long(Tnow.totalS())<<"</td><td>";
      if ( !locked ) history << "UNLOCKED</td>";
      else history << "BAD DATA</td>";
      history <<"</tr>"<<endl;
      history <<"</table><br><br><br><br><br><br>"<<endl;
      history.close();
   
    }
    else {
      for (int i=0;i<10;i++) check_sum[i].updt_temp();
      check_std.updt_temp();
      check_sdnum.updt_temp();
    }
    
    if (NoiseEffect) NoiseEffectNum++;
    if (db >= 1) {
      cout << " finoldwght:" << oldwto1/(oldwto1+1) << " finnewght:" 
	   << 1.0/(oldwto1+1) << "reloldwght:" << reloldwght_sum
	   << "expfactor:" <<expfactor << " noise effect:" << NoiseEffect
	   << " noiseeff totnum:" << NoiseEffectNum << endl;
      cout << " GoodloopNumber:" << GoodloopNum << " InvalidNum:" 
	   << InvalidNum <<" blast:" << blast << endl;
    }
    goodloop[(TotalLoopnum)%1441]=GoodloopNum;

}

//=======================================given the data array dat and its length n,find its highest point as the peak, and 
//return the right edge of the peak if direction is positive,or the left edge if the direction is negative
int SixtyHertzMon::peakedge(float* dat,int n,int direction){
  float max  = 0.0;
  int i,maxi = 0;
  for(i=0;i<n;i++) if(dat[i]>max) { maxi=i; max=dat[i]; }
  i = maxi;
  if (direction>0) {
    while( (dat[i]-dat[i+1]) > 0 ) i++;
    //  cout <<" i:"<<i<<endl;
    return i;
  }
  else
    { while( (dat[i]-dat[i-1]) > 0 ) i--;
    // cout<<" i:"<<i<<endl;
      return i;
    }
}

//========================================statistical validity checking procedure constructor
check::check (float Q) {
  q=Q;
  temp1=ave_of_x=0.0;
  temp2=ave_of_xsqr=0.0;
  temp3=powerq=1.0;
  onhold=0;
}

check::check (void) {}

//========================================check the statistical validity of a certain variable,such as the spectrum sum,
//spectrum standard deviation,or the number of sidebands found,using exponential averaging to give more weight to more 
//recent data
bool check::datvalid(float newx,float oldwght,float newght,bool weight,int db) {
  float N;
  float std_of_x;
  bool valid=1;
  bool a=0;
  bool b=0;
  temp1=ave_of_x;
  temp2=ave_of_xsqr;
  temp3=powerq;
  powerq=powerq*q;
  N=(1-q)/(1-powerq);
  //weighting paramenters
  float M=(q-powerq)/(1-powerq);
  float oldwto1;
  if (weight) {
    if (0.1*(M/N)<(oldwght/newght)) oldwto1=M/N;
    else oldwto1=oldwght/newght;
  }
  else oldwto1=M/N;
  //calculating the statistics
  if (db >= 1) cout <<" oldwto1:"<<oldwto1<<endl;
  ave_of_x=ave_of_x*oldwto1/(1+oldwto1)+newx/(1+oldwto1);
  ave_of_xsqr=ave_of_xsqr*oldwto1/(1+oldwto1)+(newx*newx)/(1+oldwto1);
  std_of_x=sqrt(abs(ave_of_xsqr-ave_of_x*ave_of_x));
  if (db >= 1) cout<<"change by std:"<<(newx-ave_of_x)/std_of_x<<endl;
  a=((newx-ave_of_x)>(2.0*std_of_x));
  b=((newx-ave_of_x)<(-2.5*std_of_x));
  if ( a || b ){
    valid=0;
    if (((newx-ave_of_x)/std_of_x<3.0)) onhold++;
  }
  if (db >= 1) cout<<"x:"<<newx<<" previous ave_of_x:"<<temp1<<" ave_of_x:"<<ave_of_x<<" std_of_x:"
      <<std_of_x<<" onhold:"<<onhold<<endl;
  if (db >= 1) cout<<"a:"<<a<<" b:"<<b<<" valid:"<<valid<<endl;
  return valid;
}

//===============================restore the existing statistical variables if the lastest segment is discarded 
void check::restore() {
  ave_of_x=temp1;
  ave_of_xsqr=temp2;
  powerq=temp3;
}

//==============================update the backup statistical variables if the lastest segment is acceptted
void check::updt_temp() {
  temp1= ave_of_x;
  temp2= ave_of_xsqr;
  temp3= powerq;
}

//==============================calculate the sum of the numbers stored in the arrary dat
float SixtyHertzMon::arraysum(float* dat,int n) {
  int i;
  float sum = 0;
  for (i=0;i<n;i++) sum = sum+dat[i];
  return sum;
}

//==============================calculate the sample standard deviation of the numbers stored in the arrary dat
float SixtyHertzMon::arraystd(float* dat,int n) {
  int ledge,redge;
  int pn;
  int i;
  float std;
  float psum  = 0.0;
  float sqsum = 0.0;
  ledge       = peakedge(dat,n,-1);
  redge       = peakedge(dat,n,+1);
  pn          = n-(redge-ledge);
  for(i=0;i<ledge+1;i++) psum  = psum+dat[i];
  for(i=redge;i<n;i++)   psum  = psum+dat[i];
  for(i=0;i<ledge+1;i++) sqsum = sqsum+(dat[i]-psum/pn)*(dat[i]-psum/pn);
  for(i=redge;i<n;i++)   sqsum = sqsum+(dat[i]-psum/pn)*(dat[i]-psum/pn);
  std = sqrt(sqsum/pn);
  return std;
}

//============================If a new month has began,then return true
//so that a new folder can be created by calling procedure
bool DatTemplate::newfolder() {
  char utc[40];
  TimeStr(Tnow+Interval(8*3600),utc,"%M");
  if ( !(string(utc)==current_month) ) {
    current_month=string(utc);
    TimeStr(Tnow+Interval(8*3600),utc,"%Y_%m");
    HistFolder = string(utc);
    return 1; 
  }
  else return 0;
}

//===========================If a new day cycle has began,then return true
//so that all the information can be stored in a new file
bool DatTemplate::newfile() {
  char utc[40];
  string renew_time;
  renew_time="16";
  TimeStr(Tnow,utc,"%H");
  if (db >=2 ) cout<<"utc:"<<utc<<" nfile:"<<nfile<<endl;
  if ( (string(utc)==renew_time ) || (TotalLoopnum==1) )
    if (nfile) { nfile=0; return 1;}
    else return 0;
  else {
    nfile=1;
    return 0;
  }
    
}

//======================================  Handle Messages
void
  DatTemplate::Attention(void) {
    MonServer::Attention();
}
