/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef SEGGENER_HH
#define SEGGENER_HH

#include "DatEnv.hh"
#include "OperStateCondList.hh"
#include "TrigClient.hh"
#include "Time.hh"
#include <list>
#include <string>

namespace trig {
    class Segment;
}

//
//  Segment condition definition class.
//
//  The segment condition class contains all the information needed to 
//  define a segment.
class SegCond {
public:
    typedef unsigned int mask_t;

public:
    /** Construct a null segment condition instance.
     *  \brief Default constructor
     */
    SegCond(void);

    /**
     *  Construct a segment that follows a signal condition.
     *  \brief Construct a follow segment condition.
     *  \param group Segment group name.
     *  \param v     Segment version number.
     *  \param on    Condition followed.
     */
    SegCond(const std::string& group, int v, const std::string& on);

    /**
     *  Construct a segment turned on by one condition and off by another.
     *  \brief Construct an on/off segment.
     *  \param group Segment group name.
     *  \param v     Segment version number.
     *  \param on    Transition to On condition.
     *  \param off   Transition to Off condition.
     */
    SegCond(const std::string& group, int v, const std::string& on, 
	    const std::string& off);
    
    /**
     *  Destroy a segment condition instance.
     *  \brief Destructor.
     */
    ~SegCond(void);

    /**
     *  Evaluate determines the current condition state and sets \c mInSegment 
     *  if the condition is asserted and \c mSegComplete when \c mInSegment 
     *  transitions from true to false or visa versa. It also counts the 
     *  segment live time.
     *  \brief Evaluate the segment condition.
     *  \param l  OSC list.
     *  \param t  Start time of the data to be evaluated.
     *  \param dT Duration of the data to be evaluated.
     *  \return True if the segment has transitioned or reached the 
     *               maximum length.
     */
    bool evaluate(OperStateCondList& l, const Time& t, Interval dT);

    //  Set up for segment generation with existing data. Returns true if a 
    //  valid segment exists.
    bool flush(void);

    //  Return the latest time.
    const Time& getCurrentTime(void) const;

    //  Return the segment type name
    const char* getGroup(void) const;

    //  Return the end-time
    Time getEndTime(void) const;

    //  Get the maximum segment length
    Interval getLimit(void) const;

    //  Get the total segment "on" time
    Interval getLiveTime(void) const;

    //  Get the number of segments generated to date.
    int getNSegment(void) const;

    //  Get the segment start condition name
    const char* getStartCond(void) const;

    //  Get the time of the last transition from off->on.
    Time getStartTime(void) const;
    Time getSegBegin(void) const;
    Time getSegEnd(void) const;
    const char* getStopCond(void) const;
    int getVersion(void) const;
    trig::Segment makeSegment(void);
    void reset(void);
    const std::string& refIfo(void) const;
    const std::string& refStartCond(void) const;
    void setComment(const std::string& comm);
    void setIfo(const std::string& ifo);
    void setLimit(Interval dT);
    void setSegFile(const std::string& file);

    /**  Test whether the segment condition is currently asserted.
     */
    bool testInSegment(void) const;

    //  testPrevious returns the segment state before the last completed
    //               evaluate.
    bool testPrevious(void) const;

    /**  testSegment returns true if mSegComplete is set, i.e. after the 
      *  segment starts.
      */
    bool testSegment(void) const;

    /**  Test whether the raw data needed to evaluate the segment are 
      *  available read out correctly.
      */
    bool testValid(void) const;

    /**  Test whether the conditions upon which the segment is calculated are
     *   well defined by the data.
     */
    bool valid(OperStateCondList& l);

    /**  Write the segment to a file.
     */
    void writeFile(const trig::Segment& s) const;

private:
    //------------------------------------  Parameters
    std::string mGroup;
    int         mVersion;
    Interval    mLimit;
    enum {
	kUnknown,
	kStartCond,  // Segment from "on" until "off"
	kOnCond      // Segment while "on"
    }           mType;
    std::string mOnCond;
    std::string mOffCond;
    std::string mSegFile;
    std::string mComment;
    std::string mIfo;

    //------------------------------------  State
    bool mDefined;
    bool mPrevious;
    bool mInSegment;

    //  Time of last transition from Off -> On.
    Time mStart;

    //  Time of last transition from On -> Off.
    Time mStop;

    //  End time of last evaluated segment
    Time mCurrent;

    //------------------------------------  Statistics
    Interval    mLiveTime;
    int         mNSegment;
};

//======================================  Inline functions
inline const Time& 
SegCond::getCurrentTime(void) const {
    return mCurrent;
}

inline const char* 
SegCond::getGroup(void) const {
    return mGroup.c_str();
}

inline Time 
SegCond::getEndTime(void) const {
    return mStop;
}

inline Interval
SegCond::getLimit(void) const {
    return mLimit;
}

inline Interval
SegCond::getLiveTime(void) const {
    return mLiveTime;
}

inline int
SegCond::getNSegment(void) const {
    return mNSegment;
}

inline Time
SegCond::getSegBegin(void) const {
    if (mPrevious) return mStart;
    return mStop;
}

inline Time
SegCond::getSegEnd(void) const {
    if (mPrevious) return mStop;
    return mStart;
}

inline Time 
SegCond::getStartTime(void) const {
    return mStart;
}

inline const char*
SegCond::getStartCond(void) const {
    return mOnCond.c_str();
}

inline const char*
SegCond::getStopCond(void) const {
    if (mType == kOnCond) return "-";
    return mOffCond.c_str();
}

inline int 
SegCond::getVersion(void) const {
    return mVersion;
}

inline const std::string& 
SegCond::refIfo(void) const {
    return mIfo;
}

inline const std::string& 
SegCond::refStartCond(void) const {
    return mOnCond;
}

inline bool 
SegCond::testInSegment(void) const {
    return mInSegment;
}

inline bool 
SegCond::testValid(void) const {
    return mDefined;
}


//=======================================  Monitor class

class SegGener : public DatEnv, TrigClient {
public:
    SegGener(int argc, const char* argv[]);
    ~SegGener(void);
    void ProcessData(void);
    bool readConfig(const char* file);
    void writeTable(void) const;

private:
    bool OSCexists(const std::string& s) const;

private:
    typedef std::list<SegCond> SegList;
    typedef SegList::const_iterator const_iter;
    typedef SegList::iterator       list_iter;

private:
    std::string mConfig;
    std::string mOSCfile;

    SegList mList;
    OperStateCondList mOSC;
    bool    mRecord;
    Time    mFirst;
    Time    mLast;
    std::string mHtFile;
    Time  mSpan0, mSpan1;
};

inline bool
SegGener::OSCexists(const std::string& s) const {
    return (mOSC.find(s) != mOSC.end());
}

#endif  //  SEGGENER_HH
