#ifndef WPIPE_TRANSFORM_HH
#define WPIPE_TRANSFORM_HH
// WTRANSFORM Compute the discrete coherent Q transform of the data
//
// WTRANSFORM applies the discrete Q transform described by the
// predetermined tiling structure to frequency domain data from
// multiple detectors, potentially after transformation into a new
// basis of virtual data streams for coherent analysis.  The tiling
// structure contains the transform parameters and is generated by the
// WTILE function.  The input data should be the fourier transform of
// the time series data to be analyzed.  However, the input frequency
// series should only extend from zero frequency to the Nyquist
// frequency.  As a result, the input frequency series should be of
// length N / 2 + 1, where N is the length of the original input time
// series.
//
// usage: transforms = wtransform(data, tiling, outlierFactor, 
//               analysisMode, channelNames, coefficients, coordinate);
//
// The following input variables are required:
//
//   data                 cell array of input frequency series data
//   tiling               discrete Q tranform tiling structure from WTILE
//   outlierFactor        Tukey whisker multiplier for outlier rejection
//   analysisMode         what type of analysis to do
//   channelNames         cell array of single detector channel names
//   coefficients         cell array of filter coefficients from WCONDITION
//   coordinate           sky position
//
// If just the first three or four parameters are present, an
// analysisMode of 'independent' is assumed or required.
//
// The output is:
//
//   transforms           cell array of discrete Q transform structures
//
// The sky position should be specified as a two component vector of
// the form [theta phi] as used by WTILESKY and WSKYMAP.  The
// coordinate theta is a geocentric colatitude running from 0 at the
// North pole to pi at the South pole, and the coordinate phi is the
// geocentric longitude in Earth fixed coordinates with 0 on the prime
// meridian.  The units are radian, the range of theta is [0, pi] and
// the range of phi is [0, 2 pi).
//
// The resulting discrete Q transform structures are parallel to the structure
// returned by WTILE and contain the following supplemental fields for each
// frequency row.
//
//   meanEnergy            mean of tile energies
//   normalizedEnergies    vector of normalized tile energies
//
// See also WTILE, WCONDITION, WTHRESHOLD, WSELECT, and WSEARCH.

// ***** See documentation for QH1H2.
// ***** This requires modifying WEVENTGRAM to also display incoherent energies.

// Shourov K. Chatterji <shourov@ligo.mit.edu>
// Antony C. Searle <acsearle@ligo.caltech.edu>
// Jameson Rollins <jrollins@phys.columbia.edu>

#include "wtypes.hh"
#include "TSeries.hh"

namespace containers {
  class DFT;
}


namespace wpipe {
  class wtile;
  class qrow;

  struct trow {
    trow(void);
    trow(double mean, const TSeries& normal);
    void dump(std::ostream& out) const;
    void setNormalized(const TSeries& energies, double transientDt, 
		       double outlierFactor, const qrow& tiling);
    double  meanEnergy;
    TSeries normalizedEnergies;
  };

  struct tplane {
    tplane(int nRow);
    void dump(std::ostream& out) const;
    const trow& rows(int iRow) const;
    std::vector<trow> _rows;
  };

  inline const trow& 
  tplane::rows(int iRow) const {
    return _rows[iRow];
  }

  /**  The qTransform class contains the q-transform results for a single 
    *  channel. A single channel transform is created with the constructor
    *  or by calling the transform method.
    */
  struct qTransform {
  public:
    qTransform(void);
    qTransform(const containers::DFT& data, const wtile& tiling, 
	       double outlierFactor, const std::string& chanName="");

    void addPlane(int nRows);
    void dump(std::ostream& out) const;
    const std::string& id(void) const;
    const tplane& planes(int iPlane) const;

    /** Perform a single channel(incoherent) transform.
     */
    void transform(const containers::DFT& data, const wtile& tiling, 
		   double outlierFactor, const std::string& chanName="");

    std::string _channelName;
    std::vector<tplane> _planes;
  };

  inline const tplane& 
  qTransform::planes(int iPlane) const {
    return _planes[iPlane];
  }

  /**  The wtransform class is a multi-channel q-transform container class.
   */
  class wtransform {
  public:
    wtransform(const containers::DFT& data, 
	       const wtile& tiling, 
	       double outlierFactor, 
	       const std::string& channelName="");

    wtransform(const dft_vect& data, 
	       const wtile& tiling, 
	       double outlierFactor, 
	       const std::string& analysisMode="independent", 
	       const str_vect& channelNames=str_vect(), 
	       const dft_vect& coefficients=dft_vect(), 
	       const dble_vect& coordinate=dble_vect());

    void dump(std::ostream& out) const;
    void init(const dft_vect& data, 
	      const wtile& tiling,
	      double outlierFactor, 
	      const std::string& analysisMode, 
	      const str_vect& channelNames, 
	      const dft_vect& coefficients, 
	      const dble_vect& coordinate);
    const TSeries& normE(int chan, int plane, int Row) const;
    int numberOfChannels(void) const;
    const qTransform& operator[](int i) const;
  private:
    std::vector<qTransform> transforms;
  };

  inline const TSeries&
  wtransform::normE(int chan, int plane, int row) const {
    return transforms[chan].planes(plane).rows(row).normalizedEnergies;
  }

  inline int 
  wtransform::numberOfChannels(void) const {
    return transforms.size();
  }

  inline const qTransform& 
  wtransform::operator[](int i) const {
    return transforms[i];
  }
} // namespace wpipe
#endif // !defined(WPIPE_TRANSFORM_HH)
