#ifndef WPIPE_FIGURE_HH
#define WPIPE_FIGURE_HH

#include "wtypes.hh"
#include "GDSPlot.hh"
#include <string>

class TSeries;
class Time;
class Interval;

namespace wpipe {
  struct qTransform;
  class  wtile;
  class  weventlist;

  class wfigure {
  public:
    wfigure(void);

    void clear(void);

    void open(void);

    /**  Set the canvas size.
     */
    void set_size(int nw=21, int nh=15);

    /**  Set the smoothing option.
     */
    void set_smooth(int nSmooth, const std::string& opt);

    /**  Set the file format
     */
    void set_format(const std::string& fmt);

    void wtimeseries(const TSeries& ts, const Time& centerTime, 
		     Interval before, Interval after, 
		     const std::string& channelName);


    /**  WSPECTROGRAM displays multi-resolution time-frequency spectrograms of
      *  normalized tile energy produced by WTRANSFORM.  A separate figure is 
      *  produced for each channel and for each time-frequency plane within the
      *  requested range of Q values.  The resulting spectrograms are 
      *  logarithmic in frequency and linear in time, with the tile color 
      *  denoting normalized energy.
      *
      *  The user can focus on a subset of the times and frequencies available 
      *  in the transformed data by specifying a desired time and frequency 
      *  range. Ranges are specified as two-component vectors, consisting of 
      *  the minimum and maximum value.  Additionally, WSPECTROGRAM can be 
      *  restricted to plot only a subset of the available Q planes by 
      *  specifying a single Q or a range of Qs.  If a single Q is specified, 
      *  WSPECTROGRAM displays the time-frequency plane which has the nearest
      *  value of Q in a logarithmic sense to the requested value. If a range 
      *  of Qs is specified, WSPECTROGRAM displays all time-frequency planes 
      *  with Q values within the requested range. By default, all available 
      *  channels, times, frequencies, and Qs are plotted.  The default values 
      *  can be obtained for any argument by passing an empty vector.
      *
      *  To define the range of times to plot, WSPECTROGRAM uses the start 
      *  time of the transformed data, a reference time, and relative time 
      *  range.  Both the start time and reference time should be absolute 
      *  quantities, but the range of times to plot should be specified 
      *  relative to the requested reference time.  The specified reference 
      *  time is used as the time origin in the resulting spectrograms and 
      *  is also reported in the title of each plot.
      *
      *  If only one time-frequency plane is requested, it is plotted in the 
      *  current figure window.  If more than one spectrogram is requested, 
      *  they are plotted in separate figure windows starting with figure 1.
      *
      *  The optional normalizedEnergyRange specifies the range of values to 
      *  encode using the colormap.  By default, the lower bound is zero and 
      *  the upper bound is autoscaled to the maximum normalized energy 
      *  encountered in the specified range of time, frequency, and Q.
      *  The optional cell array of channel names are used to label the 
      *  resulting figures.
      *
      *  The optional horizontal resolution argument specifies the number data 
      *  points in each frequency row of the displayed image.  The vector of 
      *  normalized energies in each frequency row is then interpolated to 
      *  this resolution in order to produce a rectangular array of data for 
      *  displaying.  The vertical resolution is directly determined by the 
      *  number of frequency rows in the transform data. By default, a 
      *  horizontal resolution of 2048 data points is assumed, but a higher 
      *  value may be necessary if (the zoom feature will be used to magnify 
      *  the image.  For aesthetic purposes, the resulting image is also 
      *  displayed with interpolated color shading enabled.
      *
      *  \param transforms            Single channel q transform structure
      *  \param tiling                q transform tiling structure
      *  \param referenceTime         reference time of plot
      *  \param timeRange             vector range of relative times to plot
      *  \param frequencyRange        vector range of frequencies to plot
      *  \param qRange                scalar Q or vector range of Qs to plot
      *  \param normalizedEnergyRange vector range of normalized energies for 
      *                               colormap
      *  \param horizontalResolution  number of data points across image
      */
    void wspectrogram(const qTransform& transform, const wtile& tiling, 
		      const Time& centerTime, const dble_vect& tRange,
		      const dble_vect& fRange, const dble_vect& Qrnge,
		      const dble_vect& normERange, 
		      long horizontalResolution=2048);

    /**  WEVENTGRAM Displays filled boxes corresponding to the time-frequency 
      *  boundary of statistically significant events.  WEVENTGRAM takes as 
      *  input a cell array of event matrices, one per channel. A separate
      *  figure is produced for each channel. The resulting spectrograms are 
      *  logarithmic in frequency and linear in time, with the tile color 
      *  denoting normalized energy of tiles or clusters.
      *
      *  WEVENTGRAM expects a single channel transform event structure.
      *  If the event structure tyupe field is "cluster" the normalized 
      *  energies are assume to be cluster energies.
      *
      *  The user can focus on a subset of the times and frequencies available
      *  in the original transform data by specifying the desired time and 
      *  frequency ranges. Ranges should be specified as a two component 
      *  vector, consisting of the minimum and maximum value. By default, the
      *  full time and frequency range specified in the tiling is displayed.
      *  The default values can be obtained for any argument by passing an 
      *  empty vector.
      *
      *  To determine the range of times to plot, WEVENTGRAM also requires a 
      *  reference time in addition to the specified time range.  The range of 
      *  times to plot are specified relative to the reference time.  The 
      *  specified reference time is used as the time origin in the resulting 
      *  eventgrams and is also reported in the title of each plot.  A 
      *  reference time of zero is assumed by default and????
      *
      *  The optional durationInflation and bandwidthInflation arguments are
      *  multiplicative scale factors that are applied to the duration and 
      *  bandwidth of displayed events.  If not specified, these parameters 
      *  both default to unity such that the resulting events have unit 
      *  time-frequency area.
      *
      *  The optional normalizedEnergyRange argument specifies the range of 
      *  values to encode using the colormap.  By default, the lower bound is 
      *  zero and the upper bound is autoscaled to the maximum normalized 
      *  energy encountered in the specified range of time and frequency.
      *
      *  The optional channel name string is used to label the resulting
      *  figure.
      * \brief Display statistically significant time-frequency events
      *
      * \param events                Time-frequency event list
      * \param tiling                q transform tiling structure
      * \param referenceTime         reference time of plot
      * \param timeRange             vector range of relative times to plot
      * \param frequencyRange        vector range of frequencies to plot
      * \param durationInflation     multiplicative factor for tile durations
      * \param bandwidthInflation    multiplicative factor for tile bandwidths
      * \param normalizedEnergyRange vector range of normalized energies for 
      *                              colormap
      */
    void weventgram(const weventlist& events, const wtile& tiling, 
		    const Time& referenceTime, const dble_vect& timeRange,
		    const dble_vect& frequencyRange, 
		    double durationInflation, double bandwidthInflation, 
		    const dble_vect& normalizedEnergyRange);

    void wprintfig(const std::string& figureBase, bool thumb=true) const;

  private:
    int         _wx;
    int         _wy;
    std::string _format;
    GDSPlot     _plot;
  };
}
#endif // !defined(WPIPE_FIGURE_HH)
