#include "param_list.hh"
#include "wtypes.hh"
#include "matlab_fcs.hh"

using namespace std;
using namespace wpipe;

//======================================  Constructor
param_list::param_list(void) {
}

//======================================  Destructor
param_list::~param_list(void) {
}

//======================================  Add int parameter
void 
param_list::add_param(const std::string& name, int& var) {
  param newp;
  newp._name  = name;
  newp._type  = tInt;
  newp._par.i = &var;
  _list.push_back(newp);
}

//======================================  Add double parameter
void 
param_list::add_param(const std::string& name, double& var) {
  param newp;
  newp._name  = name;
  newp._type  = tDouble;
  newp._par.d = &var;
  _list.push_back(newp);
}

//======================================  Add string parameter
void 
param_list::add_param(const std::string& name, string& var) {
  param newp;
  newp._name  = name;
  newp._type  = tString;
  newp._par.s = &var;
  _list.push_back(newp);
}

//======================================  Add numeric vector parameter
void 
param_list::add_param(const std::string& name, dble_vect& var) {
  param newp;
  newp._name   = name;
  newp._type   = tNumVec;
  newp._par.dv = &var;
  _list.push_back(newp);
}

//======================================  Add string vector parameter
void 
param_list::add_param(const std::string& name, str_vect& var) {
  param newp;
  newp._name   = name;
  newp._type   = tStrVec;
  newp._par.sv = &var;
  _list.push_back(newp);
}

//======================================  Add Time parameter
void 
param_list::add_param(const std::string& name, Time& var) {
  param newp;
  newp._name  = name;
  newp._type  = tTime;
  newp._par.t = &var;
  _list.push_back(newp);
}

//======================================  Add boolish parameter
void 
param_list::add_param(const std::string& name, bool& var) {
  param newp;
  newp._name  = name;
  newp._type  = tBool;
  newp._par.b = &var;
  _list.push_back(newp);
}

//======================================  Display all settings
std::ostream& 
param_list::display(std::ostream& out, const std::string& pfx) const {
  //------------------------------------  Get field width
  size_t nCol = 0;
  for (const_param_iter i=_list.begin(); i != _list.end(); ++i) {
    if ((i->_name).size() > nCol) nCol = (i->_name).size();
  }
  nCol += 2;

  for (const_param_iter i=_list.begin(); i != _list.end(); ++i) {
    out << pfx << i->_name << ":";
    out << string(nCol - (i->_name).size(), ' ');
    switch (i->_type) {
    case tInt:
      out << *(i->_par.i);
      break;
    case tDouble:
      out << *(i->_par.d);
      break;
    case tString:
      out << "\"" << *(i->_par.s) << "\"";
      break;
    case tNumVec:
      wpipe::display(*(i->_par.dv), out);
      break;
    case tStrVec:
      wpipe::display(*(i->_par.sv), out);
      break;
    case tTime:
      out << (i->_par.t)->totalS();
      break;
    case tBool:
      out << boolstr_yn(*(i->_par.b));
      break;
    }
    out << endl;
  }
  return out;
}

//======================================  Find named parameter
param_list::const_param_iter
param_list::find(const std::string& name) const {
  for (const_param_iter i=_list.begin(); i != _list.end(); ++i) {
    if (i->_name == name) return i;
  }
  return _list.end();
}

//======================================  Find named parameter
param_list::param_iter
param_list::find(const std::string& name) {
  for (param_iter i=_list.begin(); i != _list.end(); ++i) {
    if (i->_name == name) return i;
  }
  return _list.end();
}

//======================================  Set a parameter
void 
param_list::set_param(const std::string& name, const std::string& value) {
  param_iter i = find(name);
  if (i == _list.end()) {
    cerr << "Unrecognized parameter: " << name << " not parsed" << endl;
    return;
  }
  switch (i->_type) {
  case tInt:
    *(i->_par.i) = strtol(value.c_str(), 0, 0);
    break;
  case tDouble:
    *(i->_par.d) = strtod(value.c_str(), 0);
    break;
  case tString:
    *(i->_par.s) = unquote(value);
    break;
  case tNumVec:
    *(i->_par.dv) = eval(value);
    break;
  case tStrVec:
    if (value[0] == '{' || value[0] == '"' || value[0] == '\'') {
      *(i->_par.sv) = eval_str(value);
    } 
    else {
      (i->_par.sv)->clear();
      string test = value;
      while (! test.empty()) {
	size_t ipos = test.find(',');
	if (ipos != string::npos) {
	  (i->_par.sv)->push_back(test.substr(0,ipos));
	  test.erase(0, ipos+1);
	} else {
	  (i->_par.sv)->push_back(test);
	  test.clear();
	}
      }
    }
    break;
  case tTime:
    {
      double t  = strtod(value.c_str(), 0);
      long   tS = long(t);
      *(i->_par.t) = Time(tS, long((t-tS) * 1e9));
    }
    break;
  case tBool:
    *(i->_par.b) = strtol(value.c_str(), 0, 0);
    break;
  }
}
