#include "medianmeanaveragespectrum.hh"
#include "quick_select.hh"
#include "matlab_fcs.hh"
#include "DVecType.hh"
#include "Hanning.hh"
#include "lcl_array.hh"
#include <algorithm>
// #include <iostream>

// results verified 2011.06.27 JGZ
// factor of 2 in calculation of DC and fNyquist to match omega/matlab bug.

using namespace containers;

namespace wpipe {

  double medianbiasfactor(int);

  PSD 
  medianmeanaveragespectrum(const TSeries& data, double fs, int N, Window* w) {
    /////////////////////////////////////////////////////////////////////
    //   Checks on input.
    /////////////////////////////////////////////////////////////////////

    // ---- Verify that sample rate is a positive integer.
    if (fs != int(fs + 0.5) || fs <= 0) {
      error("sample rate fs must be a positive integer.");
    }

    // ---- Verify that data length and FFT length are commensurate.
    int lData = data.getNSample();
    if (lData % N != 0) {
      error("data length must be an integer multiple of N.");
    }

    // ---- Assign window if (necessary.
    Window* win = w;
    if (!w) win = new Hanning(N);

    /////////////////////////////////////////////////////////////////////
    //   Segment data and FFT.
    /////////////////////////////////////////////////////////////////////

    // ---- Number of segments (FFTs).
    int Ns = 2*lData/N-1;  // -- always odd

    // ---- Divide time segments into two sets of non-overlapping segments.
    //      Will compute median PSD on each separately, then average results.
    int Ns_odd  = (Ns+1)/2;
    int Ns_even = (Ns-1)/2;

    // ---- Note that oddSegs always has 1 more element than evenSegs.  Drop an
    //      element from one so that both contain an odd number of elements. 
    if (Ns_odd % 2 == 0) Ns_odd--;
    else                 Ns_even--;

    int lpsd = N/2+1;
    lcl_array<double> S_even(Ns_even*lpsd);
    lcl_array<double> S_odd(Ns_odd*lpsd);

    // ---- Number of points by which consecutive segments overlap.
    //int Delta = N/2;
    Interval dT = double(N) * data.getTStep();
    double   dF = 1./double(dT);
    long iBin = 0;
    for (long i=0; i<Ns; ++i) {
      Time t = data.getBinT(iBin);
      int ioff = i/2;
      iBin += N/2;

      //-------------------------------- Process odd bins {1,3,...2N-1}
      if (i % 2 == 0) {
	if (ioff >= Ns_odd) continue;
	PSD Si(win->apply(data.extract(t, dT)));
	const DVectD& dvd = dynamic_cast<DVectD&>(Si.refDVect());
	for (int j=0; j<lpsd; j++) {
	  S_odd[j*Ns_odd+ioff] = dvd[j];
	}
      }

      //-------------------------------- Process odd bins {2,4,...2N}
      else {
	if (ioff >= Ns_even) continue;
	PSD Si(win->apply(data.extract(t, dT)));
	const DVectD& dvd = dynamic_cast<DVectD&>(Si.refDVect());
	for (int j=0; j<lpsd; j++) {
	  S_even[j*Ns_even+ioff] = dvd[j];
	}
      }
    }
    if (!w) delete win;
    win = 0;

    // ---- Compute median PSD over "odd" segments.
    DVectD dvd(lpsd);
    double* pS = S_odd;
    for (int i=0; i < lpsd; i++) {
      dvd[i] = quick_select(pS, Ns_odd);
      pS += Ns_odd;
    }

    // ---- Compute median PSD over "even" segments.
    if (Ns_even) {
      DVectD dvd_even(lpsd);
      pS = S_even;
      for (int i=0; i < lpsd; i++) {
	dvd_even[i] = quick_select(pS, Ns_even);
	pS += Ns_even;
      }

      // ----  Weighted average of two medians.
      double nTotal(Ns_even + Ns_odd);
      double fac_odd  = Ns_odd  / medianbiasfactor(Ns_odd);
      double fac_even = Ns_even / medianbiasfactor(Ns_even);
      for (int i=0; i < lpsd; i++) {
	dvd[i] = (dvd[i] * fac_odd + dvd_even[i] * fac_even) / nTotal;
      }
    }
    else {
      dvd *= 1.0/medianbiasfactor(Ns_odd);
    }

    //-----------------------------  Normalize: Done by PSD!
    //   The normalized PSD is: PSD(f) = delta/N (h^2(f) + h^2(-f))
    //                                 = 2/(N*fs) h^2(f)
    //dvd *= 2/(N*fs);


#ifdef OMEGA_MATLAB_BUGS
    //-----------------------------  Reproduce the omega matlab bug!!!
    //                               The DC and Nyquist bins shoul not 
    //                               have a factor of 2!
    dvd[0] *= 2.0;
    dvd[lpsd-1] *= 2.0;
#endif

    //-----------------------------  Contruct PSD to return to caller.
    PSD ret_psd;
    static_cast<fSeries&>(ret_psd) = 
      fSeries(0, dF, data.getStartTime(), data.getInterval(), dvd);
    // std::cout << "returned PSD is: " << std::endl;
    // ret_psd.Dump(std::cout);
    return ret_psd;
  }

  /////////////////////////////////////////////////////////////////////////////
  //                            Helper functions.                            //
  /////////////////////////////////////////////////////////////////////////////

  double medianbiasfactor(int n) {
    // MEDIANBIASFACTOR - Compute bias factor for median estimate of mean.
    //
    // usage:
    //
    //   alpha = medianbiasfactor(n)
    //
    //  n        Scalar.  Number of samples used to estimate median.  Must be a
    //           positive, odd integer.  
    //  alpha    Scalar.  Factor by which median must be divided to give an
    //           unbiased estimate of the mean, assuming an exponential
    //           distribution.    

    // ---- Verify that n is a positive, odd, integer scalar.
    if (n < 0 || n%2 != 1) {
      error("medianbiasfactor: n must be a positive, odd, integer scalar.");
    }

    // ---- Compute bias factor alpha.
    double alpha = 0;
    double toggle = 1.0;
    for (int ii=1; ii<=n; ++ii) {
      alpha += toggle/double(ii);
      toggle = -toggle;
    }
    return alpha;
  }
} //namespace wpipe
