/***************************************************************************
    File        : Singleton.h
    Description : Singleton pattern
 ---------------------------------------------------------------------------
    Begin       : Tue Nov 8 2005
    Author(s)   : Roberto Grosso
 ***************************************************************************/

#ifndef __SINGLETON_H
#define __SINGLETON_H

/*!
 * \file Singleton.h
 * Implements the well known Singleton design pattern.
 * \brief Singleton design pattern.
 */


// Libs
#include <cstdlib>
#include <iostream>
#include <list>
#include <string>


// Project
#include "Types.h"
#include "Numeric.h"

namespace gwd {

  class Singleton
  {
  public:
    //! Iterator 
    typedef std::list<std::string*>::iterator Iterator;

    //! Return a pointer to the unique Singleton in the application.
    static Singleton* exemplar();

    //! Iterator pointing to the begin of the list of strings.
    Iterator begin() { return mMessage.begin(); }
    //! Iterator pointing to the end the list of strings.
    Iterator end() { return mMessage.end();   }

    //! check if there are some messages in the list
    bool empty() { return mMessage.empty(); }

    //! Append a string message to the end of the list.
    void AppendMessage(const std::string& message)  { std::string* msg = new std::string(message); mMessage.push_back(msg); }
    
    //! Append a const char* message at the end of the list.
    void AppendMessage(const char* message)         { std::string* msg = new std::string(message); mMessage.push_back(msg); }

    //! Delete all strings in the message list and clear list.
    void ClearMessages() {
      std::list<std::string*>::iterator p;
      for (p = mMessage.begin(); p != mMessage.end(); p++) delete *p;
      mMessage.clear();
    }
  private:
    //! The pointer to the unique instance of the Singleton.
    static Singleton *instanz;

    //! Private constructor to avoid a creation of a Singleton from outside.
    Singleton() {}

    //! Copy constructor to avoid the compiler to create a public copy constructor.
    Singleton( const Singleton& sing );

    //! Private destructor for deleting Singleton from the heap.
    ~Singleton() { ClearMessages(); }

    //! Guardian class which deletes the unique Singleton from the application.
    class Guardian {
      public:
        Guardian() {}
        Guardian(const Guardian& g) { }
        ~Guardian() { if( Singleton::instanz != 0 ) delete Singleton::instanz; }
    };

    //! Declare Guardian private to be able to access private Singleton destructor.
    friend class Guardian;

    //! List containing the global messages in form of pointer to string.
    std::list<std::string*> mMessage;
  };

} // namespace gwd
#endif // __SINGLETON_H

