/***************************************************************************
    File        : RunningMedian.cpp
    Description : Implementation file for RunningMedian.h
 ---------------------------------------------------------------------------
    Begin       : Fri Jan 2 2004
    Author(s)   : Roberto Grosso
 ***************************************************************************/

#include "RunningMedian.h"




template <typename T>
class Median {
public:

  inline Median(const int& filtersize) : mFilterSize(filtersize), mMedianPos(filtersize>>1)
  {
    int i,n;
    for(i = 0,n = mMedianPos; i < n; i++ )
    {
      mOldestElements.push_front((T)i);
      mLeftOfMedian.insert((T)i);
    }
    mOldestElements.push_front((T)i);
    mMedian = (T)i++;
    for(n = mFilterSize;i < n; i++ )
    {
      mOldestElements.push_front((T)i);
      mRightOfMedian.insert((T)i);
    }
  }
  ~Median() { }

  inline void AddSortedAndRemoveOldest( T const& value );
  inline std::ostream & Dump(std::ostream& out );
  inline T GetMedian() { return mMedian; }

private:
  unsigned int mFilterSize;
  unsigned const int mMedianPos;

  std::list<T>   mOldestElements;
  std::multiset<T>    mLeftOfMedian;
  std::multiset<T>    mRightOfMedian;
  T mMedian;
};

// 
template<typename T> inline void
Median<T>::AddSortedAndRemoveOldest( T const& value )
{
  typename std::multiset<T>::iterator pos;
  T oldval = mOldestElements.back();
  mOldestElements.pop_back();
  mOldestElements.push_front(value);

  if( (oldval < mMedian) && (value <= mMedian) )
  {
    pos = mLeftOfMedian.find(oldval);
    if(pos!=mLeftOfMedian.end())
    {
      mLeftOfMedian.erase(mLeftOfMedian.find(oldval));
      mLeftOfMedian.insert(value);
    }
  }
  else if( (oldval > mMedian) && (value >= mMedian) )
  {
    pos = mRightOfMedian.find(oldval);
    if(pos!=mRightOfMedian.end())
    {
      mRightOfMedian.erase(pos);
      mRightOfMedian.insert(value);
    }
  }
  else if( (oldval < mMedian) && (value > mMedian) )
  {
    pos = mLeftOfMedian.find(oldval);
    if(pos!=mLeftOfMedian.end())
    {
      mLeftOfMedian.erase(pos);
      mLeftOfMedian.insert(mMedian);
      if( value > *mRightOfMedian.begin() )
      {
        mMedian = *mRightOfMedian.begin();
        mRightOfMedian.erase(mRightOfMedian.begin());
        mRightOfMedian.insert(value);
      }
      else
      {
        mMedian = value;
      }
    }
  }
  else if( (oldval > mMedian) && (value < mMedian) )
  {
    pos = mRightOfMedian.find(oldval);
    if(pos!=mRightOfMedian.end())
    {
      mRightOfMedian.erase(pos);
      mRightOfMedian.insert(mMedian);
      if( value < *(--mLeftOfMedian.end() ) )
      {
        mMedian = *(--mLeftOfMedian.end());
        mLeftOfMedian.erase(--mLeftOfMedian.end());
        mLeftOfMedian.insert(value);
      }
      else
      {
        mMedian = value;
      }
    }
  }
  else if( (oldval == mMedian) )
  {
    if( value > *mRightOfMedian.begin() )
    {
      mMedian = *mRightOfMedian.begin();
      mRightOfMedian.erase(mRightOfMedian.begin());
      mRightOfMedian.insert(value);
    }
    else if( value < *(--mLeftOfMedian.end()) )
    {
      mMedian = *(--mLeftOfMedian.end());
      mLeftOfMedian.erase(--mLeftOfMedian.end());
      mLeftOfMedian.insert(value);
    }
    else
    {
      mMedian = value;
    }
  }
}

// Output
template<typename T>  inline std::ostream&
Median<T>::Dump(std::ostream& out )
{
  typename std::multiset<T>::const_iterator it,eit;
  it = mLeftOfMedian.begin();
  eit = mLeftOfMedian.end();
  while(it!=eit)
  {
    out << *it << ' ';
    ++it;
  }
  out << mMedian << ' ';
  it = mRightOfMedian.begin();
  eit = mRightOfMedian.end();
  while(it!=eit)
  {
    out << *it << ' ';
    ++it;
  }
  out << std::endl;
  return out;
}



bool
gwd::RunningMedian::ComputeMedians( const int filtersz, Vector const& in,Vector& out )
{
  mFilterSize = static_cast<Vector::size_type>(filtersz);
  if( in.empty() )
  {
    std::cerr << "RunningMedian::ComputeMedians: empty input array" << std::endl;
    return false;
  }
  if( !mFilterSize )
  {
    std::cerr << "RunningMedian::ComputeMedians: zero filter size" << std::endl;
    return false;
  }
  if (!IsOdd((unsigned int)mFilterSize))
    mFilterSize++;

  // compute best filter size;
  Vector::size_type filtersize =  mFilterSize;
  if( in.size() <= filtersize )
  {
    filtersize = in.size();
  }

  // Create Median
  Median<double> mFilter((int)filtersize);
  
  // fill filter
  for(Vector::size_type i=0,n=filtersize;i<n;i++ )
  {
    mFilter.AddSortedAndRemoveOldest(in[i]);
  }

  Vector::size_type cnt = 1+in.size()-filtersize;
  out.resize(cnt);
  Vector::iterator act_out = out.begin();
  Vector::const_iterator act_in = in.begin()+filtersize;
  while( cnt-- )
  {
    *act_out++ = mFilter.GetMedian();
    if( cnt )
    {
      mFilter.AddSortedAndRemoveOldest(*act_in++);
    }
  }
  return true;
}







