// $Id: LIGOLwMon.cc 5580 2009-03-24 18:22:05Z john.zweizig $
#include "TSeries.hh"
#include "TriggerFiles.hh"
#include "LIGOLwMon.hh"
#include "xsil/MetaIO.hh"
#include <iostream>
#include <glob.h>
#include <cstring>

EXECSVR(LIGOLwMon)

using namespace std;

//======================================  Construct the server program
LIGOLwMon::LIGOLwMon(int argc, const char* argv[]) 
  : DMTServer(argc,argv), mDelay(0), mStep(60.0)
{
  string mPath;

  // parse the command line arguments
  bool syntax = false;
  for ( int i = 1; i < argc; ++i )
  {
    string argi = argv[i];
    if (argi == "-path")
    {
      mPath = argv[++i];
    } 
    else if (argi == "-name")
    {
      mMonName = std::string(argv[++i]);
    }
    else if (argi == "-delay")
    {
      mDelay = strtod(argv[++i], 0);
    }
    else if (argi == "-step")
    {
      mStep = strtod(argv[++i], 0);
    }
    else if (isServerArg(argv[i]))
    {
      ++i;
    } 
    else
    {
      cerr << "Unrecognized command line argument: " << argi << endl;
      syntax = true;
    }
  }

  //---------------------------------  Check the command line syntax
  if ( mPath.empty() || mMonName.empty() )
  {
    cerr << "-path and -name must be specified" << endl;
    finish();
  }
  if ( syntax )
  {
    cerr << "Syntax error on command line" << endl;
    finish();
  }

  //---------------------------------  Set the server name
  string mLIGOLwMonName = "LIGOLwMon_" + mMonName;
  LIGOLwMon::setServerName( mLIGOLwMonName.c_str() );
  cerr << "Set server name to " << mLIGOLwMonName << endl;

  //---------------------------------  Initialize updates
  cerr << "Delay from present time " << mDelay << " seconds" << endl;
  mLength = 12.0 * 60.0 * 60.0;
  mLastUpdate = Now() - mDelay - mLength;

  //---------------------------------  Create the fixed length time series
  int tSeriesLength(int(mLength/mStep));
  
  float *zeroData = new float[tSeriesLength];
  memset( zeroData, 0, tSeriesLength * sizeof(float) );

  mTS_time.setMaxLen(mLength);
  mTS_number.setMaxLen(mLength);
  mTS_amp_mean.setMaxLen(mLength);
  mTS_amp_min.setMaxLen(mLength);
  mTS_amp_max.setMaxLen(mLength);
  mTS_cnf_mean.setMaxLen(mLength);
  mTS_cnf_min.setMaxLen(mLength);
  mTS_cnf_max.setMaxLen(mLength);
  mTS_frq_mean.setMaxLen(mLength);
  mTS_frq_min.setMaxLen(mLength);
  mTS_frq_max.setMaxLen(mLength);

  mTS_time.fixedAppend(mLastUpdate, mStep, zeroData, tSeriesLength );
  mTS_number.fixedAppend(mLastUpdate, mStep, zeroData, tSeriesLength );
  mTS_amp_mean.fixedAppend(mLastUpdate, mStep, zeroData, tSeriesLength );
  mTS_amp_min.fixedAppend(mLastUpdate, mStep, zeroData, tSeriesLength );
  mTS_amp_max.fixedAppend(mLastUpdate, mStep, zeroData, tSeriesLength );
  mTS_cnf_mean.fixedAppend(mLastUpdate, mStep, zeroData, tSeriesLength );
  mTS_cnf_min.fixedAppend(mLastUpdate, mStep, zeroData, tSeriesLength );
  mTS_cnf_max.fixedAppend(mLastUpdate, mStep, zeroData, tSeriesLength );
  mTS_frq_mean.fixedAppend(mLastUpdate, mStep, zeroData, tSeriesLength );
  mTS_frq_min.fixedAppend(mLastUpdate, mStep, zeroData, tSeriesLength );
  mTS_frq_max.fixedAppend(mLastUpdate, mStep, zeroData, tSeriesLength );

  delete[] zeroData;

  //---------------------------------  Initialize the trigger files
  mTrigFiles = TriggerFiles(mPath, mLastUpdate.getS());

  //---------------------------------  Serve the time series
  serveData((mMonName + "_Analyzed_Data").c_str(),      &mTS_time);
  serveData((mMonName + "_Number_of_Triggers").c_str(), &mTS_number);
  serveData((mMonName + "_Amplitude_Mean").c_str(),     &mTS_amp_mean);
  serveData((mMonName + "_Amplitude_Max").c_str(),      &mTS_amp_max);
  serveData((mMonName + "_Amplitude_Min").c_str(),      &mTS_amp_min);
  serveData((mMonName + "_Confidence_Mean").c_str(),    &mTS_cnf_mean);
  serveData((mMonName + "_Confidence_Max").c_str(),     &mTS_cnf_max);
  serveData((mMonName + "_Confidence_Min").c_str(),     &mTS_cnf_min);
  serveData((mMonName + "_Frequency_Mean").c_str(),     &mTS_frq_mean);
  serveData((mMonName + "_Frequency_Max").c_str(),      &mTS_frq_max);
  serveData((mMonName + "_Frequency_Min").c_str(),      &mTS_frq_min);
}

//======================================  Termination entry
LIGOLwMon::~LIGOLwMon(void)
{
}

void 
LIGOLwMon::ProcessData(const Time& t)
{
  Time t0 = t - mDelay;
  if ( mDelay ) mTrigFiles.setEndTime(t0.getS());

  cerr << "Starting cycle at " << t << endl;

  //---------------------------------  Extend the time series to now
  mTS_time.padUntil(t0, mStep);
  mTS_number.padUntil(t0, mStep);
  mTS_amp_mean.padUntil(t0, mStep);
  mTS_amp_min.padUntil(t0, mStep);
  mTS_amp_max.padUntil(t0, mStep);
  mTS_cnf_mean.padUntil(t0, mStep);
  mTS_cnf_min.padUntil(t0, mStep);
  mTS_cnf_max.padUntil(t0, mStep);
  mTS_frq_mean.padUntil(t0, mStep);
  mTS_frq_min.padUntil(t0, mStep);
  mTS_frq_max.padUntil(t0, mStep);

  cerr << "Coverage is " << mTS_time.getStartTime() << " to " << 
    mTS_time.getEndTime() << endl;

  //---------------------------------  Create the ts data pointers
  float* mTS_time_data   = (float *) mTS_time.refData();
  float* mTS_number_data = (float *) mTS_number.refData();
  tsdatum_t mTS_amp_data;
  tsdatum_t mTS_frq_data;
  tsdatum_t mTS_cnf_data;
  mTS_amp_data.mean = (float *) mTS_amp_mean.refData();
  mTS_amp_data.max  = (float *) mTS_amp_max.refData();
  mTS_amp_data.min  = (float *) mTS_amp_min.refData();
  mTS_frq_data.mean = (float *) mTS_frq_mean.refData();
  mTS_frq_data.max  = (float *) mTS_frq_max.refData();
  mTS_frq_data.min  = (float *) mTS_frq_min.refData();
  mTS_cnf_data.mean = (float *) mTS_cnf_mean.refData();
  mTS_cnf_data.max  = (float *) mTS_cnf_max.refData();
  mTS_cnf_data.min  = (float *) mTS_cnf_min.refData();

  //---------------------------------  Map containing the output
  mOutputSeries.insert(mmts_t::value_type("amplitude",  mTS_amp_data));
  mOutputSeries.insert(mmts_t::value_type("confidence", mTS_cnf_data));
  mOutputSeries.insert(mmts_t::value_type("frequency",  mTS_frq_data));

  //---------------------------------  Update list of trigger files
  vector <string> mFileList = mTrigFiles.newFiles();

  //---------------------------------  Process the new files
  xsil::MetaIO mio;
  for ( vector< string >::const_iterator iFile = mFileList.begin(); 
      iFile != mFileList.end(); ++iFile )
  {
    string mFile = *iFile;

    // read the segment table to get the analyzed times
    mio.open(mFile.c_str(), "segment");

    if ( ! mio.is_open() )
    {
      cerr << "Unable to open segment table in file " << mFile << endl;
      finish();
    }

    while ( mio.getRow() )
    {
      Time tSegStart = 
        Time(mio.getInt("start_time", 0), mio.getInt("start_time_ns", 0));
      Time tSegEnd = 
        Time(mio.getInt("end_time", 0), mio.getInt("end_time_ns", 0));
      Time tNow;

      if ( tSegStart.getS() == 0 || tSegEnd.getS() == 0 )
      {
        cerr << "start_time or end_time is missing from segment table" <<
        endl;
        finish();
      }

      for ( Time tNow = tSegStart; tNow < tSegEnd; tNow += mStep )
        mTS_time_data[long(mTS_time.getBin(tNow))] = 1.0;
    }

    mio.close();

    // read in the triggers (if they exist)
    mio.open(mFile.c_str(), "ligolw_mon");
    if ( mio.is_open() )
    {
      // loop over all the triggers in the file
      while( mio.getRow() )
      {
        Time trigTime = Time(mio.getInt("time"), mio.getInt("time_ns"));
        
        // find the time series bin corresponding to this trigger
        long bin = long(mTS_number.getBin(trigTime));

        // count the triggers
        ++mTS_number_data[bin];

        // iterate over the time series
        mmts_t::iterator iter;
        for ( iter = mOutputSeries.begin(); iter != mOutputSeries.end();
          ++iter )
        {

          // return zero if the column is missing
          float datum = mio.getFloat(iter->first.c_str(), 0.0);

          // compute the mean
          if ( iter->second.mean[bin] )
            iter->second.mean[bin] = (iter->second.mean[bin] + datum) / 2.0;
          else
            iter->second.mean[bin] = datum;

          // store the max
          if ( datum > iter->second.max[bin] ) iter->second.max[bin] = datum;

          // store the min snr
          if ( ! iter->second.min[bin] )
            iter->second.min[bin] = datum;
          else
          {
            if ( datum < iter->second.min[bin] )
              iter->second.min[bin] = datum;
          }
        } // end loop over time series
      } // end loop over rows

      mio.close();
    }
  }

  //---------------------------------  Discard files older than 12 hours
  mTrigFiles.flushOld((t0-mLength).getS());

  cerr << "Finished cycle at " << Now() << endl;
}
