/* -*- mode: c++; c-basic-offset: 4; -*- */
#include "Cumulus.hh"
#include "LSCSegFind.hh"
#include "ParseLine.hh"
#include "SegEval.hh"
#include <string>
#include <iostream>
#include <stdexcept>

using namespace std;

//======================================  Generate main function
EXECSVR(Cumulus)

//======================================  Monitor constructor
Cumulus::Cumulus(int argc, const char* argv[]) 
  : DMTServer(argc, argv)
{
    //----------------------------------  set default dictionary values
    mDict.addPar("Update",   double(300));
    mDict.addPar("MonName", string("Cumulus"));
 
   //----------------------------------  Parse command line
    bool syntax=false;
    for (int i=1; i<argc; ++i) {
        std::string argi = argv[i];
	if (argi == "-conf") {
	    mConfFile = argv[++i];
	} else if (isServerArg(argi)) {
	    ++i;
	} else {
	    cerr << "Unrecognized keyword: " << argi << endl;
	    syntax = true;
	    break;
	}
    }

    
    //----------------------------------  Check for command line syntax error
    if (mConfFile.empty()) syntax = true;
    if (syntax) {
        cerr << "Command syntax: " << endl;
	cerr << "   Cumulus -config <cfile>" << endl;
	finish();
	return;
    }

    //----------------------------------  Parse configuration
    readConfig();

    //----------------------------------  Parse configuration
    setServerName(mDict.getString("MonName").c_str());
    Interval update = mDict.getDouble("Update");
    if (update <= Interval(5)) {
	cerr << "Invalid update time (<5s), set to 300s." << endl;
	update = 300.0;
    }
    setReschedule(update);
}

//======================================  Monitor destructor
Cumulus::~Cumulus(void) {
}

//======================================  Fill some segments
void 
Cumulus::ProcessData(const Time& t) {
    if (Debug()) cerr << "ProcessData dispatched. Time=" << t << endl;
    for (dsp_iter i=mDspVect.begin(); i!=mDspVect.end(); ++i) {
        try {
	    i->fill(mSegMap);
	} catch (exception& e) {
	    cerr << "Exception: " << e.what() << " caught filling display: "
		 << i->getName() << endl;
	}
    }
}

//======================================  Handle attention interrupt
void 
Cumulus::Attention(void) {
    MonServer::Attention();
}

//======================================  Read configuration file
void 
Cumulus::readConfig(void) {
    ParseLine pl(mConfFile.c_str());
    if (!pl.isOpen()) {
        cerr << "Unable to open configuration file: " << mConfFile << endl;
        finish();
	return;
    }

    while(pl.getLine() >= 0) {
        int nArg = pl.getCount();
	if (!nArg) continue;

	//-------------------------------  Decode command string
	bool syntax=false;
	string cmd = pl[0];
	if (cmd == "Display") {
	    string name=pl[1];
	    if (Debug()) cerr << "Configure display: " << name << endl;
	    string seg =name;
	    Time tStart(815150000);
	    Interval dT = Interval(60*60*24);  // default interval = 1 day
	    for (int i=2; i<nArg; ++i) {
	        string argi=pl[i];
		if (argi == "-dt") {
		    dT = pl.getDouble(++i);
		} else if (argi == "-segment") {
		    seg = pl[++i];
		} else if (argi == "-start") {
		    tStart = Time(pl.getInt(++i));
		} else {
		    cerr << "Unrecognized segment key: " << argi << endl;
		    syntax = true;
		    break;
		}
	    } // end loop over options
	    if (syntax) continue;

	    long tsec=tStart.getS();
	    long dsec=long(dT);
	    tsec -= tsec%dsec;
	    segdsply sd(name, seg, Time(tsec), dT);
	    if (Debug()) sd.setDebug(1);
	    mDspVect.push_back(sd);

	//-------------------------------  Configure segment
	} else if (cmd == "Parameter") {
            for (int i=1 ; i<nArg ; i+=2) {
                if (!mDict.exists(pl[i])) {
                    cerr << "No such parameter: " << pl[i] << endl;
                    break;
                } else if (i+1 >= nArg)  {
                    cerr << "No value specified for parameter: " 
			 << pl[i] << endl;
                    break;
                } else {
                    if (Debug()) cout << "Set parameter: " << pl[i]
                                      << " to: " << pl[i+1] << endl;
                    mDict.setPar(pl[i], pl[i+1]);
                }
            }

	//-------------------------------  Configure segment
	} else if (cmd == "Segment") {
	    string name=pl[1];
	    if (segExists(name)) {
	        cerr << "Segment: " << name << " already exists!" << endl;
		continue;
	    }

	    if (Debug()) cerr << "Define segment: " << name << endl;
	    int inx = 2;
	    string seg = name;
	    if (nArg > inx && *pl[inx] != '-') seg = pl[inx++];

	    string ifo, url, file, server;
	    for (int i=inx; i<nArg; ) {
	        string argi=pl[i++];
		if (argi == "-ifo") {
		    ifo = pl[i++];
		} else if (argi == "-url") {
		    url = pl[i++];
		} else if (argi == "-file") {
		    file = pl[i++];
		} else if (argi == "-server") {
		    if (i == nArg || *pl[i] == '-') server.clear();
		    else                            server = pl[i++];
		} else {
		    cerr << "Unrecognized segment key: " << argi << endl;
		    syntax = true;
		    break;
		}
	    } // end loop over options
	    if (syntax) continue;
	    if (ifo.empty() && seg[2] == ':') {
	        ifo = seg.substr(0,2);
		seg.erase(0,3);
	    }
	    LSCSegFind lscseg(seg, ifo);
	    if (!file.empty()) {
	        lscseg.setSource(LSCSegFind::kFile, file);
		if (!url.empty()) lscseg.setUrl(url);
	    } else {
	        lscseg.setSource(LSCSegFind::kDataBase, server);
	    }
	    if (Debug()) lscseg.setDebug(1);
	    mSegMap[name] = lscseg;

	//-------------------------------  Command error
	} else {
	    cerr << "Unrecognized config command: " << cmd << endl;
	    syntax = true;
	    break;
	} 
    }  // end loop over config lines

    //-----------------------------------  Serve all the defined displays
    for (dsp_iter i=mDspVect.begin(); i!=mDspVect.end(); ++i) {
        i->book(*this);
    }
}


//======================================  Display constructor
Cumulus::segdsply::segdsply(const std::string& name, const std::string& seg, 
			    const Time& start, Interval dt) 
  : _name(name), _seg(seg), _start(start), _dt(dt), 
    _hist(name.c_str(), 101, 0, 1.01), _debug(0)
{
}

//======================================  Serve monitor objects
void 
Cumulus::segdsply::book(MonServer& svr) {
    string objname = _name + "Cumulate";
    if (_debug) cerr << "Book TSeries: " << objname << endl;
    svr.serveData(objname.c_str(), &_ts);
    objname = _name + "Histo";
    if (_debug) cerr << "Book Histogram: " << objname << endl;
    svr.serveData(objname.c_str(), &_hist);
}

//======================================  Calculate live fraction vs time
void
addLive(const LockSegList& seg, TSeries& tf) {
    int N = tf.getNSample();
    Interval dT(tf.getTStep());
    // cerr << "addLive: tf start " << tf.getStartTime() << " end time " 
    //      << tf. getEndTime() << endl;
    float* td = reinterpret_cast<float*>(tf.refData());
    for (unsigned int i=0; i<seg.size(); ++i) {
        Time tSeg1 = seg[i].getEndTime();
	if (tSeg1 <= tf.getStartTime()) continue;
        Time tSeg0 = seg[i].getStartTime();
	if (tSeg0 <  tf.getStartTime()) tSeg0 = tf.getStartTime();
	//cerr << "addLive: segment " << i << " start " << tSeg0
	//     << " end " << tSeg1 << endl;
        int fBin=tf.getBin(tSeg0);
        int lBin=tf.getBin(tSeg1);
	//cerr << "addLive: segment " << i << " first bin " << fBin 
	//     << " last bin " << lBin << endl;
	if (lBin >= N) lBin = N-1;
	if (fBin == lBin) {
	    td[fBin] += (tSeg1 - tSeg0) / dT;
	} else {
	    td[fBin] += (tf.getBinT(fBin+1) - tSeg0) / dT;
	    td[lBin] += (tSeg1 - tf.getBinT(lBin))   / dT;
	    for (int iBin=fBin+1; iBin<lBin; ++iBin) td[iBin] += 1.0;
	}
    }
}

//======================================  Set the display objects.
void 
Cumulus::segdsply::fill(Cumulus::segmap& smap) {
    SegEval eval(smap, _seg);
    const LockSegList& seg(eval.eval());

    //----------------------------------  Fill fraction live per bin.
    float seed(0);
    TSeries tf(_start, _dt, 1, &seed);
    tf.extend(Now());
    if (_debug) cerr << "Time series: " << _name << " allocated. nEntries: " 
		     << tf.getNSample() << " tStart: " << tf.getStartTime() 
		     << endl;
    addLive(seg, tf);
    if (_debug) cerr << "fill: Segment " << _name << " filled from "
		     << tf.getStartTime() << " to " << tf.getEndTime()
		     << " average live: " << tf.getAverage() << endl;

    //----------------------------------  Fill cumulative times
    int N = tf.getNSample();
    float* td = reinterpret_cast<float*>(tf.refData());    
    Histogram1 htmp(_name.c_str(), 101, 0, 1.01);
    htmp.Fill(td[0]);
    for (int i=1; i<N; ++i) {
        htmp.Fill(td[i]);
        td[i] += td[i-1];
    }

    //----------------------------------  Serve the data
    _ts   = tf;
    _hist = htmp;
}

//======================================  Set the display objects.
void 
Cumulus::segdsply::setDebug(int lvl) {
    _debug = lvl;
}
