/* -*- mode: c++; c-basic-offset: 4; -*- */
//
//    File: BitTest.hh
//
//    Bit Test monitor class definition.
//
#ifndef BITTEST_HH
#define BITTEST_HH

#include "DMTBase.hh"
#include "TrigClient.hh"
#include "AlarmClient.hh"
#include <list>
#include <iosfwd>
#include <string>

class ChanBit;

/**  @name BitTest
  *  BitTest monitors specified channels for bits that remain in one state 
  *  and for channels that repeat a value twice or more in succession. The 
  *  channels to be monitored may produce either integer or floating point 
  *  data, but the bit checking facility functions only for the integer 
  *  channels. BitTest accumulates statistics on all requested channels for
  *  a fixed number of frames, until a specified time or until an external 
  *  interrupt is received. Any detected problems are reported at the end 
  *  of the period and the accumulation cycle is restarted.
  *
  *  The BitTest error reports are produced in two sections with each 
  *  section written to a separate file. The first section lists the channels 
  *  found to be missing or to have data errors. The second section gives 
  *  detailed statistics on each channel.
  *
  *  Triggers may be generated for each erroneous channel found. One 
  *  trigger is generated for each flagged channel in each statistics 
  *  accumulation period. The generation of all triggers may be enabled
  *  or disabled with a command line argument.
  *
  *  \begin{center}{\bf {\Large The BitTest Configuration File}}\end{center}
  *
  *  The BitTest configuration file specifies the channels to be monitored 
  *  and parameters affecting the generation of triggers for each channel. 
  *  Each line of the configuration file specifies the parameters for a 
  *  single channel and contains the following fields:
  *
  *  <channel-name> <bit-range> <max-repetition>
  *
  *  The meaning of each field is as follows:
  *
  *  \begin{center}
  *  \begin{tabular}{rr}
  *    <channel-name> & Name of the channel to be monitored. \\
  *    <bit-range> & Mask of bits to be monitored. \\
  *    <max-repetition> & Maximum number of times a value may be repeated.
  *  \end{tabular}
  *  \end{center}
  *
  *  The bits in the mask need not be adjacent. If the bit mask is zero, an 
  *  automatic error detection algorithm is used that requires that any 
  *  stuck bits be adjacent high-order bits. Thus, any bit that is stuck in 
  *  one state, is not the most significant bit and is not adjacent to a 
  *  higher order stuck bit will generate a trigger. This will give the 
  *  desired result in cases where the channel measures a signal that varies 
  *  smoothly over a given range. This condition is satisfied by most LIGO
  *  signals.
  *
  *  Repetition count triggers may be disabled by setting the maximum 
  *  repetition count to zero.
  *  The configuration file is reread each time a SIGUSR1 signal is received. 
  *  This means that the monitor process need not be restarted in order to 
  *  change the configuration. Instead, the configuration file can be 
  *  modified, and the process reconfigured with a "kill -USR1 <pid>" 
  *  command.
  *
  *  \begin{center}{\bf {\Large Running BitTest}}\end{center}
  *
  *  The syntax of the BitTest command is as follows:
  *
  *  \begin{verbatim}
BitTest [-partition <pname>] [-infile <file>] [cfile <config>] \
        [ofile <out-file>] [reset <nsec>] [synch hh[:mm[:ss]]] \
        [-debug <dbg-level>] [+trig[ger]] [-toc]
     \end{verbatim}
  *
  *  Where the arguments have the following meaning:
  *
  *  \begin{center}
  *  \begin{tabular}{ll}
  *    <pname>     & Shared memory partition name with data to be read \\
  *    <file>      & Input frame file(s) (exclusive of <pname>) \\
  *    <dbg-level> & Debug level \\
  *    <config>    & Configuration file name. \\
  *    <nsec>      & Accumulation time in seconds \\
  *    hh:mm:ss    & Time (in current UTC day) to generate first report \\
  *    <out-file>  & Root output file name (defaults to "BitTest.junk")
  *  \end{tabular}
  *  \end{center}
  *
  *  The partition name is mutually exclusive with the input frame file 
  *  name. If both are specified, data are read from the specified shared 
  *  memory partition. The debug level defaults to 0 (no debug messages). 
  *  Any other value for <dbg-level> will cause debugging messages to be 
  *  printed to cout/cerr. Reports are produced at hh:mm:ss and every
  *  <nsec> seconds after that. If <nsec> is not specified, BitTest 
  *  will continue to accumulate statistics until it catches either a 
  *  SIGTERM or SIGUSR1 signal. If hh:mm:ss is not specified, BitTest will 
  *  produce a report after <nsec> frames have been received.
  *
  *  {\bf Modifying the online configuration}
  *
  *  The BitTest configuration file can be modified while BitTest is running.
  *  This is accomplished by editing the current BitTest configuration file,
  *  usually {\bf ~ops/pars/BitTest.conf}. Once the file has been modified, 
  *  BitTest can be made to read in the new configuration by signaling the
  *  running process with SIGUSR1. When the signal is caught by the process,
  *  BitTest will write out the status files with whatever statistics have 
  *  already been collected, read the configuration file and restart 
  *  processing. The SIGUSR1 signal is delivered with the following command:
  *
  *  \begin{center}#kill -USR1 <pid>#\end{center}
  *
  *  where <pid> is the ID of the BitTest process. The process ID(s) can be 
  *  found with, e.g. 
  *
  *  \begin{center}# ps -eopid,comm | grep BitTest #\end{center}
  *
  *  \begin{center}{\bf {\Large BitTest Output}}\end{center}
  *
  *  {\bf Triggers}
  *
  *  BitTest generates a trigger for each flagged channel at the end of each
  *  statistics accumulation period (nominally 20 minutes). The trigger has
  *  a trigger ID of {\bf BitTest} and a sub-ID of the channel name. The 
  *  trigger user data contains the following double precision float fields:
  *  \begin{enumerate}
  *  \item Number of words read
  *  \item Maximum repetition count
  *  \item Number of readout errors
  *  \item Mask of bits always set
  *  \item Mask of bits always zero.
  *  \item Number of Overflows/Underflows
  *  \end{enumerate}
  *  Triggers will not be produced unless the "+trig[ger]" option is 
  *  specified on the command line.
  *
  *  {\bf Alarms}
  *
  *  BitTest generates an alarm for each channel with a bit error or data
  *  overflow error at the end of each statistics accumulation period. Bit
  *  errors are indicated by a #Bit_is_Stuck# alarm and data overflows
  *  are indicated by a #Overflow# alarm. The Alarms' short descriptions
  *  (displayed by holding your pointer over the severity ball) give the
  *  channel names and for stuck bit errors, the hex masks of the stuck
  *  bits.
  *
  *  In general, #BitTest# alarms do not indicate serious problems. 
  *  Nevertheless, channels with consistent alarms should be investigated
  *  when time permits.
  *
  *  {\bf Reports and Other Output}
  *
  *  The BitTest reports are divided into two sections. The first section 
  *  is a list all channels that were found to have errors. This list is 
  *  stored in the file named by "<out-file>.Errors" where the file root 
  *  name is specified on the command line. The error list is further divided 
  *  into categories containing channels with the following errors:
  *
  *  \begin{itemize}
  *  \item
  *    {\bf Channels with errors:} List of channels that either have 
  *    one or more stuck bits or have a repetition count greater than the 
  *    maximum specified.
  *  \item 
  *    {\bf Channels with all one value:} List of channels that never 
  *    changed value during the accumulation period. Channels are listed here 
  *    even if the repetition count test is disabled by setting the maximum 
  *    to zero
  *  \item 
  *    {\bf Channels not read out:} List of channels that were requested 
  *    in the configuration file, but never found in the data frames.
  *  \end{itemize}
  *
  *  The second section is a table with statistics and status information 
  *  for all configured channels. The table is written to a file named 
  *  "<out-file>.Statistics" and contains the following information for 
  *  each channel:
  *  \begin{itemize}
  *  \item
  *    {\bf Error Flag:} A flag (****) is printed for Channels failing the  
  *    bit or repetition test.
  *  \item
  *    {\bf Channel Name:} Obvious
  *  \item
  *    {\bf Frames:} The number of frames in which the channel was seen.
  *  \item 
  *    {\bf On Bits:} A hex mask containing a 1 in each bit position in
  *    which the data was always 1.
  *  \item
  *    {\bf Off Bits:} A hex mask containing a 1 in each bit position in
  *    which the data was always 0.
  *  \item
  *    {\bf Repeat Count:} Largest number of consecutive samples containing 
  *    the same data value.
  *  \item
  *    {\bf Average value:} Average of all accumulated samples.
  *  \item
  *    {\bf Sigma:} Standard deviation from the mean of all samples during 
  *    the accumulation period.
  *  \item
  *    {\bf Minimum:} The smallest (signed) value found during the 
  *    accumulation period.
  *  \item
  *    {\bf Maximum:} The largest (signed) value found during the 
  *    accumulation period.
  *  \end{itemize}
  *  @memo Look for stuck channels or bits.
  *  @author J. Zweizig 
  *  @version 2.0; Modified April 11, 2001
  */
//@{
//@}

//======================================  Monitor class definition
class BitTest : public DMTBase, TrigClient {
  public:
    BitTest(int argc, const char *argv[]);
    ~BitTest();
  void ProcessFrame(DMTBase::frame_ptr_type frame);
    void Attention(void);
    bool ReadConfig(void);
    void Report(void);
    void Reset(void);

  private:
    AlarmClient   mAlarm;
    unsigned int  mReset;
    unsigned int  mSynch;
    unsigned int  mProcessed;
    std::string   mConfig;
    std::string   mOutput;
    std::list<ChanBit> mChannel;
    typedef std::list<ChanBit>::iterator channel_iter;
    bool          mGenTrig;
    Time          mLastProcessed;
};

//
//    ChanBit Class
//
//    The ChanBit class carries the status of each channel to be followed
//    by the bit tester.
//
class ChanBit {
  public:
     typedef unsigned int mask_t;
     typedef unsigned int counter_t;
     typedef unsigned int uint_t;
    //----------------------------------  Constructors, Destructors.
    ChanBit(const char *, mask_t ign = 0, counter_t maxct=1);
    ~ChanBit();

    //----------------------------------  Scan the data
    void Scan(uint_t nw, const short *data);
    void Scan(uint_t nw, const float *data);
    void Print(std::ostream& ostr, bool phdr=false) const;
    void Status(std::ostream& ostr, bool phdr=false) const;
    void incReadOutError(void) {mReadOutErr++;}
    void setDebug(int debug);
    void setReported(bool card);
    void Reset (void);
    bool BitError(void) const;
    bool RepeatError(void) const;
    bool OverflowError(void) const {return mOverFlow != 0;}

    /*  Test whether the channel has a flagable error. A flagable error 
     *  is considered to be a stuck bit, a repeat count error, a readout 
     *  error or an overflow error.
     *  @memo Test for a flagable error.
     */
    bool Error(void) const;
    bool isReported(void) const {return mReported;}

    //----------------------------------  Accessors
    mask_t      getStuckOn()   const;
    mask_t      getStuckOff()  const;
    mask_t      getStuckAdj()  const;
    const char* getChannel()   const {return mChanName.c_str();}
    counter_t   getNFrames()   const {return mNFrames;}
    counter_t   getNWords()    const {return counter_t(mNWords);}
    double      getAvgLen()    const;
    double      getAverage()   const;
    double      getSigma()     const;
    double      getMinimum()   const {return mMinimum;}
    double      getMaximum()   const {return mMaximum;}
    counter_t   getMaxRpt(void) const {return mMaxRpt;}
    counter_t   getLongRpt()   const {return mLongR;}
    mask_t      getLastValue()  const {return mValue;}
    mask_t      getHiBits() const;
    mask_t      getDBits()  const {return mDBits;}
    counter_t   getNOverFlow(void) const {return mOverFlow;}
    counter_t   getNReadOutError(void) const {return mReadOutErr;}


  private:
    std::string mChanName;
    counter_t mMaxRpt;
    counter_t mNFrames;
    mask_t    mSelect;
    counter_t mDBits;
    int       mDebug;
    double    mNWords;
    double    mAverage;
    double    mSigma;
    double    mMinimum;
    double    mMaximum;
    mask_t    mStuckOn;
    mask_t    mStuckOff;
    mask_t    mStuckAdj[4];
    mask_t    mValue;
    double    mFValue;
    counter_t mRepeat;
    counter_t mLongR;
    bool      mReported;
    bool      mInteger;
    counter_t mOverFlow;
    counter_t mReadOutErr;
};

//====================================  Inline ChanBit Methods
#ifndef __CINT__

//------------------------------------  Test if bits are stuck on/off
inline ChanBit::mask_t
ChanBit::getStuckOn()  const {
    return  (mStuckOn & mDBits);
}

inline ChanBit::mask_t
ChanBit::getStuckOff()  const {
    return  (~mStuckOff & mDBits);
}

//----------------------------------  Get the running average.
inline double 
ChanBit::getAverage() const {
    if (mNWords == 0) return 0.0;
    return mAverage/mNWords;
}

//----------------------------------  Get the running average length
inline double 
ChanBit::getAvgLen() const {
    if (mNFrames == 0) return 0.0;
    return mNWords/mNFrames;
}

//----------------------------------  Get the sigma
inline double 
ChanBit::getSigma() const {
    if (mNWords == 0) return 0.0;
    double Avg = mAverage/mNWords;
    double Var = mSigma/mNWords - Avg*Avg;
    if (Var <= 0) return 0.0;
    return ::sqrt(Var);
}

inline void 
ChanBit::setDebug(int debug) {
    mDebug = debug;
}

inline void 
ChanBit::setReported(bool card) {
    mReported = card;
}

#endif // __CINT__

#endif // BITTEST_HH
