// BicoDir -- Bicoherence (Direct Method) Class
// 
//  -- Steven Penn - 10 Sep 03 - penn@hws.edu
// --------------------------------------------
//
// DESCRIPTION:
// ------------
//    A class for the bicoherence of a time series.  The direct method
//    for bicoherence calculation is used.  Functions include setting
//    the calculation configuration record, performing the calculation,
//    and integrating over given regions.
//
//      
// VERSION HISTORY:
// ----------------
// Version 0.1 -- 10 Sep 2003 -- Steve Penn
//    - I am splitting my Bicoherence code into code for a background monitor
//       and a foreground viewer.  This class will be an object that can be 
//       used in either for calculating the bicoherence.
//
//
//---------------------------------------------------------------------------- 
//

// C++ headers
#include <sys/types.h>
#include <cstdlib>
#include <cmath>
#include <cstdio>

// DMT/GDS headers
#include <Time.hh>
#include <Hanning.hh>
#include "DVector.hh"
#include "DVecType.hh" 
#include "Dacc.hh"


#include "BicoDir.hh"

using namespace std;


// DECLARE CONSTANTS
// -----------------
const Int_t kFreqBinMin = 64 ;            // Min Number of Freq Bins



// **********************************************************
// *                                                        *
// *   ---  BicoMeas: Bicoherence measurement Record  ---   *
// *                                                        *
// **********************************************************

// Constructor
// -----------
BicoMeas::BicoMeas () 
  : fLabel (""), fFreq1 (0), fFreq2 (0), fRad (0), fNConf (0)
{
}


// Destructor
// ----------
BicoMeas::~BicoMeas ()   
{
}



// ********************************************************
// *                                                      *
// *   ---  BicoDirConf: Bicoherence Configuration  ---   *
// *                                                      *
// ********************************************************

// Constructor
// -----------
BicoDirConf::BicoDirConf () 
  : fNew (false), fValid (false), fPause (false), 
    fChanName1 (""), fChanName2 (""), fChanName3 (""), 
    fChanNum1 (0), fChanNum2 (0), fChanNum3 (0), fNChans (0),
    fRate (0), fRate1 (0), fRate2 (0), fRate3 (0), 
    fDecim (0), fDecim1 (0), fDecim2 (0), fDecim3 (0), 
    fDecOrder1(0), fDecOrder2(0), fDecOrder3(0),
    fFreqRes (0), fFreqMax (0), fNBins (0), fNPoints (0), fNFFT (0), 
    fNWinRad (0), fOverlap (0), fWindow1D (false), fAverage (0), fNTotal (0),
    fSpan (0), fStride (0), fNSlices (0), fWindow2D (false)
{
}


// Destructor
// ----------
BicoDirConf::~BicoDirConf ()   
{
}





// *******************************************************
// *                                                     *
// *   ---  BicoDir:  Bicoherence (Direct Method)  ---   *
// *                                                     *
// *******************************************************

// Constructor
// -----------
BicoDir::BicoDir ()
	:	fBicoDirData (0),  
   	windowNorm (0.0), 
      ttemp1 (0),ttemp2 (0),ttemp3 (0),
      fftOut1 (0),fftOut2 (0),fftOut3 (0),
      ftemp1 (0),ftemp2 (0),ftemp3 (0),
      fft_plan1 (0), fft_plan2 (0), fft_plan3 (0),
      ptemp1 (0), ptemp2 (0), ptemp3 (0),
      bsd_data (0), bsdWindow (0), corr_norm (0), bic_norm (0), 
      psd_temp (0), bc_temp (0), bs_temp (0)
{
   fConf.fNew = false ;    // Make false until new configuration set
   fConf.fValid = false ;  // Make false until valid configuration set
}


// Destructor
// ----------
BicoDir::~BicoDir()
{
   BicoDir::DeleteDynamicArrays();   
}



// ************************************
// *                                  *
// *   ---  BicoDir:  TestData  ---   *
// *                                  *
// ************************************

bool BicoDir::TestData(int fChanNum, const TSeries* fDataPtr1, const TSeries* fDataPtr2, const TSeries* fDataPtr3)
{
   bool fDataValid = true ;
   
   switch (fChanNum) {
    case 3:
      fDataValid = (fDataValid && fDecFilter3.isDataValid(*fDataPtr3));
    case 2:
      fDataValid = (fDataValid && fDecFilter2.isDataValid(*fDataPtr2));
    case 1:
      fDataValid = (fDataValid && fDecFilter1.isDataValid(*fDataPtr1));
      break;
    default: 
     cout<<"ERROR: Channel Number not 1-3"<<endl;
     return false;
   }
   
   return fDataValid;
}



// ************************************
// *                                  *
// *   ---  BicoDir:  TestData  ---   *
// *                                  *
// ************************************

bool BicoDir::TestData(int fChanNum, const TSeries& fDataNew1, const TSeries& fDataNew2, const TSeries& fDataNew3)
{
   bool fDataValid = true ;
   
   switch (fChanNum) {
    case 3:
      fDataValid = (fDataValid && fDecFilter3.isDataValid(fDataNew3));
      break;
    case 2:
      fDataValid = (fDataValid && fDecFilter2.isDataValid(fDataNew2));
      break;
    case 1:
      fDataValid = (fDataValid && fDecFilter1.isDataValid(fDataNew1));
      break;
    default: 
     cout<<"ERROR: Channel Number not 1-3"<<endl;
     return false;
   }
   
   return fDataValid;
}



// ***********************************
// *                                 *
// *   ---  BicoDir:  SetData  ---   *
// *                                 *
// ***********************************

void BicoDir::SetData(int fDataNum, const TSeries* fDataPtr, bool fOnline)
{
   if (fOnline) {
      switch (fDataNum) {
       case 3:
         fData3 = fDataPtr->decimate(fConf.fDecOrder3);
         break;
       case 2:
         fData2 = fDataPtr->decimate(fConf.fDecOrder2);
         break;
       case 1:
         fData1 = fDataPtr->decimate(fConf.fDecOrder1);
         break;
       default: 
         cout<<"ERROR: Channel Number not 1-3"<<endl;
         return;
      }
   } else {
      switch (fDataNum) {
       case 3:
         fData3 = fDecFilter3.apply(*fDataPtr);
         break;
       case 2:
         fData2 = fDecFilter2.apply(*fDataPtr);
         break;
       case 1:
         fData1 = fDecFilter1.apply(*fDataPtr);
         break;
       default: 
         cout<<"ERROR: Channel Number not 1-3"<<endl;
         return;
      }
   }
}



// ***********************************
// *                                 *
// *   ---  BicoDir:  SetData  ---   *
// *                                 *
// ***********************************

void BicoDir::SetData(int fDataNum, const TSeries& fDataNew, bool fOnline)
{
   if (fOnline) {
      switch (fDataNum) {
       case 3:
         fData3 = fDataNew.decimate(fConf.fDecOrder3);
         break;
       case 2:
         fData2 = fDataNew.decimate(fConf.fDecOrder2);
         break;
       case 1:
         fData1 = fDataNew.decimate(fConf.fDecOrder1);
         break;
       default: 
         cout<<"ERROR: Channel Number not 1-3"<<endl;
         return;
      }
   } else {
      switch (fDataNum) {
       case 3:
         fData3 = fDecFilter3.apply(fDataNew);
         break;
       case 2:
         fData2 = fDecFilter2.apply(fDataNew);
         break;
       case 1:
         fData1 = fDecFilter3.apply(fDataNew);
         break;
       default: 
         cout<<"ERROR: Channel Number not 1-3"<<endl;
         return;
      }
   }
}
   


// *************************************************
// *                                               *
// *   ---  BicoDir:  Set New Configuration  ---   *
// *                                               *
// *************************************************

void BicoDir::SetConfig(const BicoDirConf& fConfNew)
{
   // Read in Basic Parameters
   // (Chan Names, Chan Rates, Freq Max, Freq Res, Overlap, Span)
   // -----------------------------------------------------------
   fConf = fConfNew;
   
   // Calculate Remaining Parameters
   // ------------------------------
   if (!BicoDir::CalcConfigParams()) {
      cout << " Error configuring monitor. Exiting."<<endl;
//       finish();
   }

   
   // Generate the Decimation Filters
   // -------------------------------
   fDecFilter1 = DecimateBy2(fConf.fDecOrder1, 3);
   fDecFilter2 = DecimateBy2(fConf.fDecOrder2, 3);
   fDecFilter3 = DecimateBy2(fConf.fDecOrder3, 3);
   

   
	#ifdef BD_DEBUG
      switch (fConf.fNChans) {
       case 3:
         cout << "fConf.fChanName3 = " << fConf.fChanName3 << endl;
         cout << "fConf.fRate3     = " << fConf.fRate3 << endl;
         cout << "fConf.fDecim3    = " << fConf.fDecim3 << endl;
         cout << "fConf.fDecOrder3 = " << fConf.fDecOrder3 << endl;
       case 2:
         cout << "fConf.fChanName2 = " << fConf.fChanName2 << endl;
         cout << "fConf.fRate2     = " << fConf.fRate2 << endl;
         cout << "fConf.fDecim2    = " << fConf.fDecim2 << endl;
         cout << "fConf.fDecOrder2 = " << fConf.fDecOrder2 << endl;
       case 1:
         cout << "fConf.fChanName1 = " << fConf.fChanName1 << endl;
         cout << "fConf.fRate1     = " << fConf.fRate1 << endl;
         cout << "fConf.fDecim1    = " << fConf.fDecim1 << endl;
         cout << "fConf.fDecOrder1 = " << fConf.fDecOrder1 << endl;
         break;
       default:
         cout << "ERROR setting configuration data." << endl;
      }
      cout << "fConf.fNChans    = " << fConf.fNChans << endl;
      cout << "fConf.fRate      = " << fConf.fRate << endl;
      cout << "fConf.fDecim     = " << fConf.fDecim << endl;
      cout << "fConf.fFreqRes   = " << fConf.fFreqRes << endl;
      cout << "fConf.fFreqMax   = " << fConf.fFreqMax << endl;
      cout << "fConf.fNBins     = " << fConf.fNBins << endl;
      cout << "fConf.fNPoints   = " << fConf.fNPoints << endl;
      cout << "fConf.fNFFT      = " << fConf.fNFFT << endl;
      cout << "fConf.fNWinRad   = " << fConf.fNWinRad << endl;
      cout << "fConf.fOverlap   = " << fConf.fOverlap << endl;
      cout << "fConf.fWindow1D  = " << fConf.fWindow1D << endl;
      cout << "fConf.fAverage   = " << fConf.fAverage << endl;
      cout << "fConf.fNTotal    = " << fConf.fNTotal << endl;
      cout << "fConf.fSpan      = " << fConf.fSpan << endl;
      cout << "fConf.fStride    = " << fConf.fStride << endl;
      cout << "fConf.fNSlices   = " << fConf.fNSlices << endl;
      cout << "fConf.fWindow2D  = " << fConf.fWindow2D << endl;
      cout << "fConf.fNew       = " << fConf.fNew << endl;
      cout << "fConf.fValid     = " << fConf.fValid << endl;
   #endif
}
   


// *********************************************
// *                                           *
// *   ---  BicoDir:  Get Configuration  ---   *
// *                                           *
// *********************************************

BicoDirConf BicoDir::GetConfig()
{
   // Return the Configuration record
   return fConf;
}




// ************************************************************************************
//
//   CALCULATE CONFIGURE PARAMETERS: Calculate remaining parameters from those read in.
//
// ************************************************************************************
//
bool BicoDir::CalcConfigParams()
{
   Int_t fWindow2DRange = 64 ;  // 1/Range = fraction of fNFFT bins covered by Window 
   
   
   fConf.fRate = 16384 ;
   switch (fConf.fNChans) {
    case 3:
      fConf.fRate = (fConf.fRate < fConf.fRate3) ? fConf.fRate : fConf.fRate3 ;
    case 2:
      fConf.fRate = (fConf.fRate < fConf.fRate2) ? fConf.fRate : fConf.fRate2 ;
    case 1:
      fConf.fRate = (fConf.fRate < fConf.fRate1) ? fConf.fRate : fConf.fRate1 ;
   }
   
   
   // Calculate the remaining parameters
   // ----------------------------------
   float fFreqNyq = fConf.fRate / 2 ;   // Nyquist Frequency
   fConf.fFreqMax = (fConf.fFreqMax > fFreqNyq) ? fFreqNyq : fConf.fFreqMax ; //Max Freq
   
   // Calculate decimation and create decimation filter
   fConf.fDecim = int(fFreqNyq / fConf.fFreqMax) ;
   fConf.fRate = fConf.fRate / fConf.fDecim ;
   fConf.fDecim1 = int(fConf.fRate1 / fConf.fRate); 
   fConf.fDecim2 = int(fConf.fRate2 / fConf.fRate); 
   fConf.fDecim3 = int(fConf.fRate3 / fConf.fRate); 
   fConf.fDecOrder1 = int(std::log(float(fConf.fDecim1)) / std::log(2.0)); 
   fConf.fDecOrder2 = int(std::log(float(fConf.fDecim2)) / std::log(2.0)); 
   fConf.fDecOrder3 = int(std::log(float(fConf.fDecim3)) / std::log(2.0)); 

         
   if (fConf.fFreqRes > 0.0) {
      fConf.fNBins = int(fConf.fFreqMax / fConf.fFreqRes) ;
      if (fConf.fNBins < kFreqBinMin) {
         fConf.fNBins = kFreqBinMin ;
         fConf.fFreqRes = fConf.fFreqMax / fConf.fNBins ;
      } 
   } else {
      fConf.fNBins = kFreqBinMin ;
      fConf.fFreqRes = fConf.fFreqMax / fConf.fNBins ;
   }
   
   fConf.fNPoints = 2 * fConf.fNBins ;
   fConf.fNFFT = fConf.fNBins + 1;
   fConf.fNWinRad = fConf.fNFFT / fWindow2DRange ;
   fConf.fWindow1D = (fConf.fOverlap > 0.0);  // if no overlap then don't window in 1D
//    fConf.fWindow2D = false ;
   
   // Set Span to Max span for all configurations
//    fConf.fSpan = fSpanMax ;
   fConf.fNTotal = fConf.fSpan * fConf.fRate ;
   Int_t fNonOverlapPoints = int((1-fConf.fOverlap)*fConf.fNPoints) ;
   fConf.fNSlices = 1 + (fConf.fNTotal - fConf.fNPoints) / fNonOverlapPoints;
   fConf.fAverage = fConf.fNSlices ;
   fConf.fStride = fConf.fSpan ;  
   
   // Validate configuration correctly read in
   fConf.fValid = true ;    
   
   
   return true;
}




// *************************************************************
// *                                                           *
// *   ---  BicoDir:  Create Data Containers and Arrays  ---   *
// *                                                           *
// *************************************************************
void BicoDir::CreateDataObjects()
{
   // To create the DMT Objects, data arrays, decimation rates, etc., we  
   // require information from the requested data channels.  Rather than create
   // and delete these objects with each frame, we sample the data channels
   // and create thse objects once.
   // 
   // In SetupDataObjects we:
   // -----------------------
   //  -- allocate the DMT Objects
   //  -- allocate the data arrays
   //  -- create the FFTW plan
   //  -- create the plotting histograms and canvases
   //  
      
   // Allocate DMT Objects
   // --------------------
	#ifdef BD_DEBUG
   	cout<<"  Allocating DMT Objects  ... "<<endl;
   #endif
   
   fFrameStartTime = fData1.getStartTime();
   Interval dT = fData1.getTStep();    // Data time interval
   dT *= fConf.fNPoints;                    // Time interval for each slice
   
   switch (fConf.fNChans) {
    case 3:
//      tDataSlice3 = new TSeries(fData3.extract(fFrameStartTime, dT)); 
      tDataSlice3 = fData3.extract(fFrameStartTime, dT); 
    case 2:
//      tDataSlice2 = new TSeries(fData2.extract(fFrameStartTime, dT)); 
      tDataSlice2 = fData2.extract(fFrameStartTime, dT); 
    case 1:
//      tDataSlice1 = new TSeries(fData1.extract(fFrameStartTime, dT)); 
      tDataSlice1 = fData1.extract(fFrameStartTime, dT); 
     break;
    default: 
     cout<<"ERROR: Channel Number not 1-3"<<endl;
     return;
   }
   
   

   // Allocate Data Arrays
   // --------------------
   
	#ifdef BD_DEBUG
   	cout<<"  Allocating Data Arrays  ... "<<endl;
   #endif
      
   switch (fConf.fNChans) {
    case 3:
     ptemp1 = new float[fConf.fNFFT];      // Power Spectrum Data
     ptemp2 = new float[fConf.fNFFT];   
     ptemp3 = new float[fConf.fNFFT];
     ftemp1 = new std::complex<float>[fConf.fNFFT];      // FFT Data
     ftemp2 = new std::complex<float>[fConf.fNFFT];   
     ftemp3 = new std::complex<float>[fConf.fNFFT];
     break;
    case 2:
     ptemp1 = new float[fConf.fNFFT];      // Power Spectrum Data
     ptemp2 = new float[fConf.fNFFT];   
     ptemp3 = ptemp2;
     ftemp1 = new std::complex<float>[fConf.fNFFT];      // FFT Data
     ftemp2 = new std::complex<float>[fConf.fNFFT];   
     ftemp3 = ftemp2;
     break;      
    case 1:
     ptemp1 = new float[fConf.fNFFT];      // Power Spectrum Data
     ptemp2 = ptemp1;
     ptemp3 = ptemp1;
     ftemp1 = new std::complex<float>[fConf.fNFFT];      // FFT Data
     ftemp2 = ftemp1;
     ftemp3 = ftemp1;
     break;
    default:  
     cout<<"ERROR: Channel Number not 1-3"<<endl;
     return;
   }

      
   // Direct Method Arrays
   // ----------------------
   fBicoDirData = new Float_t[fConf.fNBins*fConf.fNBins];  // Bicoherence Data
   
   switch (fConf.fNChans) {
    case 3:
     fftOut1 = (fftwf_complex*)fftwf_malloc(sizeof(fftwf_complex) * fConf.fNFFT);   // FFT Data (direct method)
     fftOut2 = (fftwf_complex*)fftwf_malloc(sizeof(fftwf_complex) * fConf.fNFFT);
     fftOut3 = (fftwf_complex*)fftwf_malloc(sizeof(fftwf_complex) * fConf.fNFFT);
     ttemp1 = (float*)fftwf_malloc(sizeof(float) * fConf.fNPoints); 
     ttemp2 = (float*)fftwf_malloc(sizeof(float) * fConf.fNPoints); 
     ttemp3 = (float*)fftwf_malloc(sizeof(float) * fConf.fNPoints); 
     fft_plan1 = fftwf_plan_dft_r2c_1d(fConf.fNPoints, ttemp1, fftOut1, FFTW_ESTIMATE);
     fft_plan2 = fftwf_plan_dft_r2c_1d(fConf.fNPoints, ttemp2, fftOut2, FFTW_ESTIMATE);
     fft_plan3 = fftwf_plan_dft_r2c_1d(fConf.fNPoints, ttemp3, fftOut3, FFTW_ESTIMATE);
     break;
    case 2:
     fftOut1 = (fftwf_complex*)fftwf_malloc(sizeof(fftwf_complex) * fConf.fNFFT);   
     fftOut2 = (fftwf_complex*)fftwf_malloc(sizeof(fftwf_complex) * fConf.fNFFT);
     fftOut3 = fftOut2;
     ttemp1 = (float*)fftwf_malloc(sizeof(float) * fConf.fNPoints); 
     ttemp2 = (float*)fftwf_malloc(sizeof(float) * fConf.fNPoints); 
     ttemp3 = ttemp2;
     fft_plan1 = fftwf_plan_dft_r2c_1d(fConf.fNPoints, ttemp1, fftOut1, FFTW_ESTIMATE);
     fft_plan2 = fftwf_plan_dft_r2c_1d(fConf.fNPoints, ttemp2, fftOut2, FFTW_ESTIMATE);
     break;      
    case 1:
     fftOut1 = (fftwf_complex*)fftwf_malloc(sizeof(fftwf_complex) * fConf.fNFFT);   
     fftOut2 = fftOut1;
     fftOut3 = fftOut1;
     ttemp1 = (float*)fftwf_malloc(sizeof(float) * fConf.fNPoints); 
     ttemp2 = ttemp1;
     ttemp3 = ttemp1;
     fft_plan1 = fftwf_plan_dft_r2c_1d(fConf.fNPoints, ttemp1, fftOut1, FFTW_ESTIMATE);
     break;
    default:  
     cout<<"ERROR: Channel Number not 1-3"<<endl;
     return;
   }
   
   // Creating FFTW Plan
   // ------------------   
	#ifdef BD_DEBUG
   	cout<<"  Creating FFTW Plan  ... "<<endl;
   #endif
      
  
   // Create the Bicoherence and Bispectral Data Arrays
   // -------------------------------------------------
   fBicoDirData = new float[fConf.fNBins*fConf.fNBins];
//   fBis_Data = new float[fConf.fNBins*fConf.fNBins];

   
   // Create the Optimal Bispectral Window
   // ------------------------------------
   if (fConf.fWindow2D) {
		#ifdef BD_DEBUG
			cout<<"Generating Optimal Bispectral Frequency Window ..."<<endl;
      #endif
     
     int Wr = 2*fConf.fNWinRad + 1;           // Diameter, in bins, of window
     bsdWindow = new float[Wr*Wr] ;          // Bispectral Window
     for (int ix=0; ix<Wr; ++ix) {
        for (int iy=0; iy<=ix; ++iy) {
           int ixy = ix*Wr + iy;
           int iyx = iy*Wr + ix;
           float s1 = float(ix-fConf.fNWinRad) / float(fConf.fNWinRad) ;
           float s2 = float(iy-fConf.fNWinRad) / float(fConf.fNWinRad) ;
           bsdWindow[ixy] = optFreqWindow(s1,s2) ;
           if (ix != iy) {
              bsdWindow[iyx] = bsdWindow[ixy];
              windowNorm += 2*bsdWindow[ixy];
           } else {
              windowNorm += bsdWindow[ixy];
           }
        }
     }
     cout << "Window Normalization = " << windowNorm << endl;
	  #ifdef BD_DEBUG
        for (int ix=0; ix<Wr; ++ix) {
           int ixx = ix*(Wr/2) + ix;
           cout<< " Window["<<ix<<", "<<ix<<"] = "<<bsdWindow[ixx]<<endl;
        }
     #endif
   }
}




//###############################################################
//#                                                             #
//#  ---  BicoDir:  Calculate Bicoherence (Direct Method)  ---  #
//#                                                             #
//###############################################################

void BicoDir::Calculate(void) {
   //
   // I should make separate routines for autobicoherence and for crossbicoherence
   //
   // Direct Bicoherence Method
   // -------------------------
   //
   //      Calculate the bispectrum as a product of the 1D FFT's
   //
   //          Bis(f1,f2) = F(f1)*F(f2)*F(f1+f2)
   //
   //       which is averaged over the number of data slices of the time series
   //
   //       The bicoherence has multiple definitions depending on 
   //          1) whether the magnitude is squared, and
   //          2) whether the normalization is correlated.
   //
   //       The Normalization is defined as either 
   //          1)  N(f1,f2) = |P(f1)*P(f2)| * |P(f1 + f2)|      (Correlated)
   //          2)  N(f1,f2) = |P(f1)| * |P(f2)| * |P(f1 + f2)|  (UnCorrelated)
   //          where P is the Power Spectral Density and |P| implies that P 
   //          is averaged over data slices
   //
   //       The magnitude is defined as either
   //          1) Bic(f1,f2) = (Bis(f1,f2))^2 / N(f1,f2)
   //          2) Bic(f1,f2) = Bis(f1,f2) / sqrt(N(f1,f2))
   //
   //       
   //      For the bicoherence we average the bispectrum over the bin size
   //      and divide by the averaged power spectrum 
   //
   //   

	#ifdef BD_DEBUG
      cout<<" Calculating the Bicoherence, Direct Method ..."<<endl;
      if (fConf.fWindow2D) cout<<"    ... with Rao-Gabr windowing"<<endl;
   #endif
   
   if (!fConf.fValid) {
      cout << "ERROR: Invalid Configuration. Calculation Aborted." << endl ;
      return;
   }
   
   fFrameStartTime = fData1.getStartTime();    // Frame start time
	#ifdef BD_DEBUG
   	cout << "Start Time = " << fFrameStartTime << endl;
   #endif
   Interval dT = fData1.getTStep();    // Data time interval
   dT *= fConf.fNPoints;                    // Time interval for each slice
	#ifdef BD_DEBUG
   	cout << "dT = " << dT << endl;
   #endif
   Hanning hanWdw; 
   
   int fErrorLimit = -5;   // Limit the number of calculation errors issued
   
   
   // Create/Zero Data Containers
   // ---------------------------
   //
   // New Configuration
   //
   if (fConf.fNew) {
		#ifdef BD_DEBUG
      	cout << "Creating Data Objects" << endl;
      #endif
      BicoDir::CreateDataObjects() ;
      fConf.fNew = false;
	//
	// Old Configuration
	//
   } else {
   	// Grab initial data slice
      switch (fConf.fNChans) {
       case 3:
         tDataSlice3 = fData3.extract(fFrameStartTime, dT);    // Make times series of data slice            
       case 2:
         tDataSlice2 = fData2.extract(fFrameStartTime, dT);    // Make times series of data slice
       case 1:
         tDataSlice1 = fData1.extract(fFrameStartTime, dT);    // Make times series of data slice
         break;
       default:
         cout<<"ERROR: Channel Number not 1-3"<<endl;
         return;
      }
   }
   
         
   

   // Create the bispectral data arrays
   // ---------------------------------
   bsd_data = new complex<float>[fConf.fNBins*fConf.fNBins];	// Bispectrum (direct method)
   corr_norm = new Float_t[fConf.fNBins*fConf.fNBins];     		// Bicoherence Normalization
   
	// Clear Bicoherence Array
   for (Int_t ix=0; ix<fConf.fNBins*fConf.fNBins; ++ix) {
      fBicoDirData[ix] = 0.0 ;
      bsd_data[ix] = 0.0 ;
      corr_norm[ix] = 0.0 ;
   }

   // Zero PSD & FFT arrays
   for (Int_t ix=0; ix<fConf.fNBins; ++ix) {
      switch (fConf.fNChans) {
       case 3:
         ptemp3[ix] = 0.0;
         ftemp3[ix] = 0.0;
       case 2:
         ptemp2[ix] = 0.0;
         ftemp2[ix] = 0.0;
       case 1:
         ptemp1[ix] = 0.0;
         ftemp1[ix] = 0.0;
         break;
      }
   }      

   

   
   // Process Data Slices:  Calculate FFT, PSD, and Bispectrum
   // --------------------------------------------------------
   float fNorm = 1.0/float(fConf.fNSlices);
   
	#ifdef BD_DEBUG
   	cout << "Processing Data Slices ..." << endl;
   #endif
   
   for (Int_t ik=0; ik<fConf.fNSlices; ++ik) {
      
      // Skip Rereading first data slice
      //--------------------------------
      if (ik!=0) {
         fFrameStartTime += dT*(1.0-fConf.fOverlap);                 // increment start time
         switch (fConf.fNChans) {
          case 3:
            tDataSlice3 = fData3.extract(fFrameStartTime, dT);            // Make times series of data slice            
          case 2:
            tDataSlice2 = fData2.extract(fFrameStartTime, dT);            // Make times series of data slice
          case 1:
            tDataSlice1 = fData1.extract(fFrameStartTime, dT);            // Make times series of data slice
            break;
          default:
            cout<<"ERROR: Channel Number not 1-3"<<endl;
            return;
         }
      }

		#ifdef BD_DEBUG
      	if (ik == 0) {
            cout << "Detrend, FFT, Form PSD  ..." << endl;
      	}
      #endif
    
      // Detrend Data, Apply Window, Calculate FFT & PSD
      // -----------------------------------------------
      switch (fConf.fNChans) {
       case 3:
         tDataSlice3.Convert(DVecType<float>::getDataType()); // Make the data float (redundant but necessary)
         tDataSlice3 -= tDataSlice3.getAverage();            // Detrend (remove average)
         if (fConf.fWindow1D) {
            tDataSlice3 = hanWdw(tDataSlice3);               // Window time series if desired
         }
         tDataSlice3.getData(fConf.fNPoints,ttemp3);          // Extract data into array
         fftwf_execute(fft_plan3);                        		// fftOut3 = FFT(ttemp3)
         for (Int_t ix=0; ix<fConf.fNBins; ++ix) {
         	ftemp3[ix+1] = complex<float> (fftOut3[0][ix+1], fftOut3[1][ix+1]);
            ptemp3[ix] += norm(ftemp3[ix+1]) * fNorm ;     // ptemp3 = ftemp3 .* ~(ftemp3)
         }
         
       case 2:
         tDataSlice2.Convert(DVecType<float>::getDataType()); // Make the data float (redundant but necessary)
         tDataSlice2 -= tDataSlice2.getAverage();            // Detrend (remove average)
         if (fConf.fWindow1D) {
            tDataSlice2 = hanWdw(tDataSlice2);               // Window time series if desired
         }
         tDataSlice2.getData(fConf.fNPoints,ttemp2);          // Extract data into array
         fftwf_execute(fft_plan2);                        		// fftOut2 = FFT(ttemp2)
         for (Int_t ix=0; ix<fConf.fNBins; ++ix) {
            ftemp2[ix+1] = complex<float> (fftOut2[0][ix+1], fftOut2[1][ix+1]);
            ptemp2[ix] += norm(ftemp2[ix+1]) * fNorm ;     // ptemp2 = ftemp2 .* ~(ftemp2)
         }
         
       case 1:
         tDataSlice1.Convert(DVecType<float>::getDataType()); // Make the data float (redundant but necessary)
         tDataSlice1 -= tDataSlice1.getAverage();            // Detrend (remove average)
         if (fConf.fWindow1D) {
            tDataSlice1 = hanWdw(tDataSlice1);               // Window time series if desired
         }
         tDataSlice1.getData(fConf.fNPoints,ttemp1);          // Extract data into array
         fftwf_execute(fft_plan1);                        		// fftOut1 = FFT(ttemp1)
         for (Int_t ix=0; ix<fConf.fNBins; ++ix) {
            ftemp1[ix+1] = complex<float> (fftOut1[0][ix+1], fftOut1[1][ix+1]);
            ptemp1[ix] += norm(ftemp1[ix+1]) * fNorm ;     // ptemp1 = ftemp1 .* ~(ftemp1)
         }
         break;
         
       default:
         cout<<"ERROR: Channel Number not 1-3"<<endl;
         return;
      }
      
		#ifdef BD_DEBUG
         if (ik == 0) {								// First slice only
            for (Int_t ix=0; ix<5; ++ix) {
               cout << "FFT(" << ix << ") = " << ftemp1[ix+1] << "     PSD(" << ix << ") = " << ptemp1[ix] << endl;
               cout << endl;
               cout << "Calculate BSD  ..." << endl;
            }
         }
      #endif

       
      // Calculate the bispectrum (BSD)
      // -------------------------------
      //  NOTE: we must use all quadrants for X-bicoherence
      for (int ix=0; ix<fConf.fNBins; ++ix) {     // We can ignore the constant 0 channel
         for (int iy=0; iy<fConf.fNBins; ++iy) {
            Int_t ixy = ix + iy + 1 ;
            Int_t ixychan = ix*fConf.fNBins + iy;

            if (ixy <= fConf.fNFFT-1) {
               bs_temp = conj(ftemp3[ixy]);
            } else {
               bs_temp = ftemp3[fConf.fNPoints-ixy];
            }

            bs_temp = ftemp1[ix+1] * ftemp2[iy+1] * bs_temp ;
            corr_norm[ixychan] += norm(ftemp1[ix+1] * ftemp2[iy+1]) * fNorm ;
            
            // Increment -- No Smoothing Window           
            if (!fConf.fWindow2D) {
               bsd_data[ixychan] += bs_temp * fNorm ;
              
            // Increment -- Smoothing Window
            } else {
               int jxmin = (ix-fConf.fNWinRad < 0)? 0 : ix-fConf.fNWinRad;
               int jxmax = (fConf.fNBins < ix + fConf.fNWinRad)? fConf.fNBins : ix+fConf.fNWinRad ;
               int jymin = (iy-fConf.fNWinRad < 0)? 0 : iy-fConf.fNWinRad;
               int jymax = (fConf.fNBins < iy + fConf.fNWinRad)? fConf.fNBins : iy+fConf.fNWinRad ;
               for (int jx=jxmin; jx<jxmax; ++jx) {     
                  for (int jy=jymin; jy<jymax; ++jy) {
                     Int_t jxychan = jx*(fConf.fNBins) + jy;
                     Int_t wx = (ix - jx + fConf.fNWinRad) ;
                     Int_t wy = (iy - jy + fConf.fNWinRad) ;
                     Int_t wxychan = wx*(2*fConf.fNWinRad+1) + wy ;                     
                     bsd_data[jxychan] += bs_temp * bsdWindow[wxychan] / windowNorm ;
                  }
               }
            } // End of Window Test
            
         }
      }  // End of loop over x,y
                 
   }   // End of the Loop over Data Slices  
   
	#ifdef BD_DEBUG
   	cout << "Calculate Bicoherence  ..." << endl;
	#endif
      
   // Calculate the Bicoherence array
   for (Int_t ix=0; ix<fConf.fNBins; ++ix) {    
      for (Int_t iy=0; iy<fConf.fNBins; ++iy) {
         Int_t jx = ix + 1;
         Int_t jy = iy + 1;
         Int_t jxy = ix + iy + 1;
         jxy = (jxy < fConf.fNBins) ? jxy : fConf.fNPoints - jxy - 1 ;   
//         jxy = (jxy <= fConf.fNBins) ? jxy : fConf.fNPoints - jxy ;   
         Int_t ixy = fConf.fNBins*ix + iy;                // Ignore the constant 0 channels
         
         if (kBicoNormCorrelated) {
            bic_norm = corr_norm[ixy] * ptemp3[jxy] ;
         } else {
            bic_norm = ptemp1[jx] * ptemp2[jy] * ptemp3[jxy] ;
         }
         
         if (bic_norm > 0.0) {
            if (kBicoMagSquared) {
               fBicoDirData[ixy] = norm(bsd_data[ixy]) / bic_norm ;
            } else {
               fBicoDirData[ixy] = abs(bsd_data[ixy]) / sqrt(bic_norm) ;
            }
         } else {
            fBicoDirData[ixy] = 0.0;
            if (fErrorLimit < 0) {
               ++fErrorLimit;
               if (kBicoNormCorrelated) {
                  cout <<"ERROR: BicoNorm["<<ix<<"]["<<iy<<"] = 0, CorrNorm["<<ixy<<"]="<<corr_norm[ixy]<<", P3["<<jxy<<"]="<<ptemp3[jxy]<<"]"<<endl;
               } else {
                  cout <<"ERROR: BicoNorm["<<ix<<"]["<<iy<<"] = 0, P1["<<jx<<"]="<<ptemp1[jx]<<", P2["<<jy<<"]="<<ptemp2[jy]<<", P3["<<jxy<<"]="<<ptemp3[jxy]<<"]"<<endl;
               }
            }
            
         }

      }   // End of loop over y
   }   // End of loop over x
   
   #ifdef BD_DEBUG
      for (int ixy=0; ixy < 10; ++ixy) {
         cout << "Bicoherence[" << ixy<< "] = " << fBicoDirData[ixy] << endl;
      }
   #endif
   
   
	// Delete the Bicoherence Data Arrays  
	//-----------------------------------
   delete[] bsd_data;
   delete[] corr_norm;
   
}
      	


//#################################
//#                               #
//#  ---  BicoDir:   Sum (3) ---  #
//#                               #
//#################################
//
// Sum over the region centered at (f1,f2) with radius, dF
// 
float BicoDir::Sum(const BicoMeas& fMeasure) {
   float f1 = fMeasure.fFreq1;
   float f2 = fMeasure.fFreq2;
   float df = fMeasure.fRad;
   df = abs(df);
      
   return BicoDir::SumRegion(f1, f2, df) ;
}



//#################################
//#                               #
//#  ---  BicoDir:   Sum (2) ---  #
//#                               #
//#################################
//
// Sum over the region centered at (f1,f2) with radius, dF
// 
float BicoDir::Sum(float f1, float f2, float df) {

   return BicoDir::SumRegion(f1, f2, df) ;
}



//#################################
//#                               #
//#  ---  BicoDir:   Sum (1) ---  #
//#                               #
//#################################
//
// Sum over the entire unique region
// 
float BicoDir::Sum(void) {
   
   return BicoDir::SumAll();
}



//###################################
//#                                 #
//#  ---  BicoDir:   SumRegion ---  #
//#                                 #
//###################################
//
// Sum over the region centered at (f1,f2) with radius, dF
// 
float BicoDir::SumRegion(float f1, float f2, float df) {

	// SumAll if all inputs are 0
	// --------------------------
	if ((f1 == 0.0) && (f2 == 0.0) && (df == 0.0)) {
	   return BicoDir::SumAll();
	}

   float tempSum = 0.0 ;
   float tempBins = 0.0 ;
   df = abs(df);
   

   // Set Bin Limits. Sum is over square region.  Round down 1 for array bin
   // ----------------------------------------------------------------------
   int ixmin = int(floor(-0.5 + (f1 - df)/fConf.fFreqRes));  // Lower limit in x
   int ixmax = int(floor(0.5 + (f1 + df)/fConf.fFreqRes));   // Upper limit in x
   int iymin = int(floor(-0.5 + (f2 - df)/fConf.fFreqRes));  // Lower limit in x
   int iymax = int(floor(0.5 + (f2 + df)/fConf.fFreqRes));   // Upper limit in x
   
   ixmin = (ixmin > 0) ? ixmin : 0 ;
   ixmax = (ixmax < fConf.fNBins) ? ixmax : fConf.fNBins ;
   iymin = (iymin > 0) ? iymin : 0 ;
   iymax = (iymax < fConf.fNBins) ? iymax : fConf.fNBins ;

	
	// Sum over the region.
	// --------------------	
   for (int ix=ixmin; ix<ixmax; ++ix) {     	// Ignore the constant 0 channel
      for (int iy=iymin; iy<iymax; ++iy) {
         Int_t ixy = fConf.fNBins*ix + iy;		// Ignore the constant 0 channels
         tempSum += fBicoDirData[ixy] ;
         tempBins += 1.0 ;
      }
   }
   
   if (tempBins > 0.0) {
   	tempSum = tempSum/tempBins ;
   } else {
   	cout << " Error:  Zero Bicoherence Bins Summed." << endl;
   	tempSum = 0.0;
   }
   
   return tempSum;
}



//#################################
//#                               #
//#  ---  BicoDir:   SumAll  ---  #
//#                               #
//#################################
//
// Sum over the entire unique region
// 
float BicoDir::SumAll(void) {
   float tempSum = 0.0 ;
   float tempBins = 0.0 ;
   
   for (int ix=0; ix<fConf.fNBins; ++ix) {     	// Ignore the constant 0 channel
      int iymax = 2*(fConf.fNBins - ix) - 1 ;
      iymax = (ix+1 < iymax) ? ix : iymax ;
      for (int iy=0; iy<iymax; ++iy) {
         Int_t ixy = fConf.fNBins*ix + iy;		// Ignore the constant 0 channels
         if (iy == ix) {
            tempSum += 0.5 * fBicoDirData[ixy] ;
            tempBins += 0.5 ;
         } else {
            tempSum += fBicoDirData[ixy] ;
            tempBins += 1.0 ;
         }
      }
   }
   
   if (tempBins > 0.0) {
   	tempSum = tempSum/tempBins ;
   } else {
   	cout << " Error:  Zero Bicoherence Bins Summed." << endl;
   	tempSum = 0.0;
   }
   
   return tempSum;
}



//#######################################
//#                                     #
//#  ---  BicoDir:   CopyBicoData  ---  #
//#                                     #
//#######################################
//
// Copy the bicoherence array to the supplied
// 
// I am not fond of this method, but it will suffice for now.
// 
void BicoDir::CopyBicoData(float* fOutArray) {
   
	// Arrays must be same size or big errors.

   int fBins = fConf.fNBins*fConf.fNBins;
   
   for (int ik=0; ik < fBins; ++ik) {
      fOutArray[ik] = fBicoDirData[ik];
   }

   return ;
}


//########################################
//#                                      #
//#  ---  BicoDir:   CopyStartTime  ---  #
//#                                      #
//########################################
//
// Copy the Data Start Time
// 
// I am not fond of this method, but it will suffice for now.
// 
void BicoDir::CopyStartTime(Time fOutStartTime) {
   
   
   fOutStartTime = fFrameStartTime;
   
   #ifdef BD_DEBUG
      cout<<"Frame Start Time = "<< fFrameStartTime << endl;
      cout<<"Copy Start Time = "<< fOutStartTime << endl;
   #endif

   return ;
}



//########################################
//#                                      #
//#  ---  BicoDir:   GetStartTime  ---  #
//#                                      #
//########################################
//
// Get the Data Start Time
// 
// I am not fond of this method, but it will suffice for now.
// 
Time& BicoDir::GetStartTime(void) {
      
   #ifdef BD_DEBUG
      cout<<"Frame Start Time = "<< fFrameStartTime << endl;
   #endif

   return fFrameStartTime;
}




//######################################
//#                                    #
//#  ---  BicoDir:   GetBicoData  ---  #
//#                                    #
//######################################
//
// Direct supplied pointer to bicoherence array
// 
// I am not fond of this method, but it will suffice for now.
// 
void BicoDir::GetBicoData(float* fOutArray) {
   float* fTempPointer;
   
   // Arrays must be same size or big errors.

   fTempPointer = fOutArray;
   fOutArray = fBicoDirData;
   fBicoDirData = fTempPointer;
   
   return ;
}



//######################################
//#                                    #
//#  ---  BicoDir:   GetBicoData  ---  #
//#                                    #
//######################################
//
// Direct supplied pointer to bicoherence array
// 
// I am not fond of this method, but it will suffice for now.
// 
float* BicoDir::GetBicoData() {
   
   return fBicoDirData;
}



//######################################
//#                                    #
//#  ---  BicoDir:   GetPSDData  ---  #
//#                                    #
//######################################
//
// Direct supplied pointer to PSD array
// 
// I am not fond of this method, but it will suffice for now.
// 
void BicoDir::GetPSDData(int fChanNum, float* fOutArray) {
   float* fTempPointer;
   
   // Arrays must be same size or big errors.

   switch (fChanNum) {
    case 3:
      {      
         fTempPointer = fOutArray;
         fOutArray = ptemp3;
         ptemp3 = fTempPointer;
         break;
      }
    case 2:
      {
         fTempPointer = fOutArray;
         fOutArray = ptemp2;
         ptemp2 = fTempPointer;
         break;
      }
    case 1:
      {
         fTempPointer = fOutArray;
         fOutArray = ptemp1;
         ptemp1 = fTempPointer;
         break;
      }
    default:
      {
         cout<<"ERROR: PSD Channel Number not 1-3"<<endl;
         return;
      }
   }
    
   return ;
}



//######################################
//#                                    #
//#  ---  BicoDir:   GetPSDData  ---  #
//#                                    #
//######################################
//
float* BicoDir::GetPSDData(int fChanNum) {
   
   // Arrays must be same size or big errors.

   switch (fChanNum) {
    case 3:
      {      
         return ptemp3;
         break;
      }
    case 2:
      {
         return ptemp2;
         break;
      }
    case 1:
      {
         return ptemp1;
         break;
      }
    default:
      {
         cout<<"ERROR: PSD Channel Number not 1-3"<<endl;
         return 0;
      }
   }
    
   return 0 ;
}




//######################################
//#                                    #
//#  ---  BicoDir:   CopyPSDData  ---  #
//#                                    #
//######################################
//
void BicoDir::CopyPSDData(int fChanNum, float* fOutArray) {
   
   // Arrays must be same size or big errors.
   int fBins = fConf.fNBins;

   switch (fChanNum) {
    case 3:
      {      
         for (int ik=0; ik < fBins; ++ik) {
            fOutArray[ik] = ptemp3[ik];
         }
         break;
      }
    case 2:
      {
         for (int ik=0; ik < fBins; ++ik) {
            fOutArray[ik] = ptemp2[ik];
         }
         break;
      }
    case 1:
      {
         for (int ik=0; ik < fBins; ++ik) {
            fOutArray[ik] = ptemp1[ik];
         }
         break;
      }
    default:
      {
         cout<<"ERROR: PSD Channel Number not 1-3"<<endl;
         return;
      }
   }
    
   return;
}




//#######################################
//#                                     #
//#  ---  BicoDir:  optFreqWindow  ---  #
//#                                     #
//#######################################
//
// Function used to form the optimal freq window
// see Rao-Gabr _Intro to Bispectral Analysis_ P. 39-42
// 
// windowNorm = sqrt(3)/(pi^3)
// 
float BicoDir::optFreqWindow(float s1, float s2) {
   float windowNormTheory = 0.0558613 ;   
   float df2 = (s1*s1 + s2*s2 + s1*s2) ;
   if (df2 > 1.0) {
      return 0.0 ;
   } else {
      return ((1.0 - df2) * windowNormTheory );
   }
}



// *****************************************************
// *                                                   *
// *   ---  BicoDir:  Delete the Dynamic Arrays  ---   *
// *                                                   *
// *****************************************************
//
void BicoDir::DeleteDynamicArrays()
{
   
   //  cout<<"Deallocating Data Arrays ..."<<endl;  
   // Deleting the ptemp arrays causes the following error:
   //     Fatal in <operator delete>: storage area overwritten
   switch (fConf.fNChans) {
    case 3:
      delete [] ptemp3;
      delete [] ftemp3;
    case 2:
      delete [] ptemp2;
      delete [] ftemp2;
    case 1:
      delete [] ptemp1;
      delete [] ftemp1;
      break ;
    default:
      return ;  // Nothing created so nothing to destroy
   }

   // Deallocate Window Array
   if (fConf.fWindow2D) {
		#ifdef BD_DEBUG
      	cout<<"Deallocating Window Arrays ..."<<endl;
      #endif
      delete[] bsdWindow ;
   }

   #ifdef BD_DEBUG
   	cout<<"Deallocating Direct Method Data Arrays ..."<<endl;  
   #endif
   delete[] fBicoDirData;
   
   // Free the FFTW Arrays and Destroy the FFTW Plan 
   #ifdef BD_DEBUG
   	cout<<"Deallocating FFTW Data Arrays and Destroying the FFT plan..."<<endl; 
   #endif
   switch (fConf.fNChans) {
    case 3:
      fftwf_free(ttemp3);
      fftwf_free(fftOut3);
      fftwf_destroy_plan(fft_plan3);
    case 2:
      fftwf_free(ttemp2);
      fftwf_free(fftOut2);
      fftwf_destroy_plan(fft_plan2);
    case 1:
      fftwf_free(ttemp1);
      fftwf_free(fftOut1);
      fftwf_destroy_plan(fft_plan1);
   }
   
}



// ***********************************************
// *                                             *
// *   ---  BicoDir:  Delete DMT Objects   ---   *
// *                                             *
// ***********************************************
void BicoDir::DeleteDMTObjects()
{
//    switch (fConf.fNChans) {
//     case 3:
//       delete tDataSlice3;
//     case 2:
//       delete tDataSlice2;
//     case 1:
//       delete tDataSlice1;
//       break ;
//     default:
//       cout<<"ERROR: Channel Number not 1-3"<<endl;
//       return;
//    }
}




