#include <time.h>
#include "xml/XsilHistogram.hh"		// Associated header file
#include <strings.h>
#include <iostream>

   using namespace std;
namespace xml {


//////////////////////////////////////////////////////////////////////////
//                                                                      //
// xsilHistogram                                                        //
//                                                                      //
// XML histogram object                                                 //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   std::ostream& xsilHistogram::write (std::ostream &os) const 
   {
      if (!fH1 && !fH2) {
         return os;
      }
      int bintype = fH1 ? fH1->GetBinType() : fH2->GetBinType();
      if (bintype == 0) {
         return os;
      }
      // Begin & subtype
      os << xsilDataBegin (fH1 ? fH1->GetTitle() : fH2->GetTitle(),
                          xmlObjTypeHistogram, fLevel) << endl;
      int subtype = fH1 ? 0 : 2;
      if (bintype == 2) ++subtype;
      bool errorflag = fH1 ? fH1->IsErrorFlagON() : fH2->IsErrorFlagON();
      if (errorflag) subtype += 6;
      os << xsilParameter<int> ("Subtype", subtype) << endl;
      // Time & NData
      os << xsilTime ("t0", fH1 ? fH1->GetTime() : fH2->GetTime()) 
         << endl;
      os << xsilParameter<int> ("NData", fH1 ? fH1->GetNEntries() : 
                           fH2->GetNEntries()) << endl;
      // Title and Labels
      const char* p = fH1 ? fH1->GetTitle() : fH2->GetTitle();
      if (p && *p) os << xsilParameter<const char*> ("Title", p) << endl;
      p = fH1 ? fH1->GetXLabel() : fH2->GetXLabel();
      if (p && *p) os << xsilParameter<const char*> ("XLabel", p) << endl;
      p = fH1 ? 0 : fH2->GetYLabel();
      if (p && *p) os << xsilParameter<const char*> ("YLabel", p) << endl;
      p = fH1 ? fH1->GetNLabel() : fH2->GetNLabel();
      if (p && *p) os << xsilParameter<const char*> ("NLabel", p) << endl;
      // Weights
      Histogram1::stat_t stat[10];
      if (fH1) {
         fH1->GetStats (stat);
         os << xsilParameter<double> ("SumWeight", stat[0]) << endl;
         os << xsilParameter<double> ("SumWeightSqr", stat[1]) << endl;
         os << xsilParameter<double> ("SumWeightX", stat[2]) << endl;
         os << xsilParameter<double> ("SumWeightXSqr", stat[3]) << endl;
      }
      else {
         fH2->GetStats (stat);
         os << xsilParameter<double> ("SumWeight", stat[0]) << endl;
         os << xsilParameter<double> ("SumWeightSqr", stat[1]) << endl;
         os << xsilParameter<double> ("SumWeightX", stat[2]) << endl;
         os << xsilParameter<double> ("SumWeightXSqr", stat[3]) << endl;
         os << xsilParameter<double> ("SumWeightY", stat[4]) << endl;
         os << xsilParameter<double> ("SumWeightYSqr", stat[5]) << endl;
         os << xsilParameter<double> ("SumWeightXY", stat[6]) << endl;
      }
      // Bin number
      int xbins = fH1 ? fH1->GetNBins() : fH2->GetNBins (0);
      int ybins = fH1 ? 0 : fH2->GetNBins (1);
      // temporary array
      double* arr = new double [(xbins + 2) * (ybins + 2)];
      // Write bins
      os << xsilParameter<int> ("NBinx", xbins) << endl;
      if (fH1) {
         if (bintype == 1) {
            os << xsilParameter<double> ("XLowEdge", 
                                 fH1->GetBinLowEdge (0)) << endl;
            os << xsilParameter<double> ("XSpacing", 
                                 fH1->GetBinSpacing()) << endl;
         }
         else {
            fH1->GetBinLowEdges (arr);
            os << xsilArray<double> ("XBins", xbins + 1, arr) << endl;
         }
      }
      else {
         os << xsilParameter<int> ("NBiny", ybins) << endl;
         if (bintype == 1) {
            os << xsilParameter<double> ("XLowEdge", 
                                 fH2->GetBinLowEdge (0, 0)) << endl;
            os << xsilParameter<double> ("XSpacing", 
                                 fH2->GetBinSpacing (0)) << endl;
            os << xsilParameter<double> ("YLowEdge", 
                                 fH2->GetBinLowEdge (0, 1)) << endl;
            os << xsilParameter<double> ("YSpacing", 
                                 fH2->GetBinSpacing (1)) << endl;
         }
         else {
            fH2->GetBinLowEdges (arr, 0);
            os << xsilArray<double> ("XBins", xbins + 1, arr) << endl;
            fH2->GetBinLowEdges (arr, 1);
            os << xsilArray<double> ("YBins", ybins + 1, arr) << endl;
         }
      }
      // Data
      if (fH1) {
         fH1->GetBinContents (arr);
         os << xsilArray<double> ("Contents", xbins + 2, arr) << endl;
      }
      else {
         fH2->GetBinContents (arr);
         os << xsilArray<double> ("Contents", xbins + 2, 
                              ybins + 2, arr) << endl;
      }
      // Errors
      if (errorflag) {
         if (fH1) {
            fH1->GetBinErrors (arr);
            os << xsilArray<double> ("Errors", xbins + 2, arr) << endl;
         }
         else {
            fH2->GetBinErrors (arr);
            os << xsilArray<double> ("Errors", xbins + 2, 
                                 ybins + 2, arr) << endl;
         }
      }
      delete [] arr;
      os << xsilDataEnd<double>() << endl;
      return os;
   }



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// xsilHandlerHistogram                                                 //
//                                                                      //
// XML histogram handler                                                //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   xsilHandlerHistogram::xsilHandlerHistogram (
                     std::vector<Histogram1>* objs1, 
                     std::vector<Histogram2>* objs2,
                     const attrlist* attr, bool ignore)
   : xsilHandler (ignore), fDat1 (objs1), fDat2 (objs2),
   fSubtype (-1), fSec(0), fNsec(0), fNData (0), fSumWeight (0),
   fSumWeightSqr (0), fSumWeightXY (0), fErrors (0), fData (0) 
   {
      for (int i = 0; i < 3; ++i) {
         fNBin[i] = 0;
         fSumWeightDim[i] = 0;
         fSumWeightDimSqr[i] = 0;
         fLowEdge[i] = 0;
         fSpacing[i] = 1;
         fBins[i] = 0;
         fBinsLen[i] = 0;
         fErrorsDim[i] = 0;
         fDataDim[i] = 0;
      }
   }

//__________________________________________________________________________
   xsilHandlerHistogram::~xsilHandlerHistogram() 
   {
      // 1D histogram
      if ((fSubtype == 0) || (fSubtype == 1) ||
         (fSubtype == 6) || (fSubtype == 7)) {
         bool err = false;
         Histogram1 h1;
         // Setup bins
         if ((fSubtype == 0) || (fSubtype == 6)) {
            h1.SetBinLowEdges (fNBin[0], fLowEdge[0], fNBin[0]*fSpacing[0]);
         }
         else {
            if ((fNBin[0] == fBinsLen[0] - 1) && fBins[0]) {
               h1.SetBinLowEdges (fNBin[0], fBins[0]);
            }
            else {
               err = true;
            }
         }
         // Set labels
         h1.SetXLabel (fLabel[0].c_str());
         h1.SetNLabel (fLabel[3].c_str());
         // Set data
         if ((fNBin[0] == fDataDim[0] - 2) && fData &&
            (fDataDim[1] == 0) && (fDataDim[2] == 0)) {
            h1.SetBinContents (fData);
         }
         else {
            err = true;
         }
         // Set errors
         if ((fSubtype == 6) || (fSubtype == 7)) {
            if ((fNBin[0] == fErrorsDim[0] - 2) && fErrors && 
               (fErrorsDim[1] == 0) && (fErrorsDim[2] == 0)){
               h1.SetBinErrors (fErrors);
            }
            else {
               err = true;
            }
         }
         // Set weights
         Histogram1::stat_t stat[4];
         stat[0] = fSumWeight;
         stat[1] = fSumWeightSqr;
         stat[2] = fSumWeightDim[0];
         stat[3] = fSumWeightDimSqr[0];
         h1.PutStats (stat);
      	// Title & time & # of entries
         h1.SetTitle (fTitle.c_str());
         h1.SetTime (Time (fSec, fNsec));
         h1.SetNEntries (fNData);
         // Add histogram to array 
         if (!err && fDat1) fDat1->push_back (h1);
      }
      
      // 2D histogram
      else if ((fSubtype == 2) || (fSubtype == 3) ||
              (fSubtype == 8) || (fSubtype == 9)) {
         bool err = false;
         Histogram2 h2;
         // Setup bins
         if ((fSubtype == 2) || (fSubtype == 8)) {
            h2.SetBinLowEdges (fNBin[0], fLowEdge[0], fNBin[0]*fSpacing[0],
                              fNBin[1], fLowEdge[1], fNBin[1]*fSpacing[1]);
         }
         else {
            if ((fNBin[0] == fBinsLen[0] - 1) && fBins[0] &&
               (fNBin[1] == fBinsLen[1] - 1) && fBins[1]) {
               h2.SetBinLowEdges (fNBin[0], fBins[0], fNBin[1], fBins[1]);
            }
            else {
               err = true;
            }
         }
         // Set labels
         h2.SetXLabel (fLabel[0].c_str());
         h2.SetYLabel (fLabel[1].c_str());
         h2.SetNLabel (fLabel[3].c_str());
         // Set data
         if ((fNBin[0] == fDataDim[0] - 2) && 
            (fNBin[1] == fDataDim[1] - 2) && fData &&
            (fDataDim[2] == 0)) {
            h2.SetBinContents (fData);
         }
         else {
            err = true;
         }
         // Set errors
         if ((fSubtype == 8) || (fSubtype == 9)) {
            if ((fNBin[0] == fErrorsDim[0] - 2) &&
               (fNBin[1] == fErrorsDim[1] - 2) && fErrors && 
               (fErrorsDim[2] == 0)) {
               h2.SetBinErrors (fErrors);
            }
            else {
               err = true;
            }
         }
         // Set weights
         Histogram2::stat_t stat[7];
         stat[0] = fSumWeight;
         stat[1] = fSumWeightSqr;
         stat[2] = fSumWeightDim[0];
         stat[3] = fSumWeightDimSqr[0];
         stat[4] = fSumWeightDim[1];
         stat[5] = fSumWeightDimSqr[1];
         stat[6] = fSumWeightXY;
         h2.PutStats (stat);
      	// Title & time & # of entries
         h2.SetTitle (fTitle.c_str());
         h2.SetTime (Time (fSec, fNsec));
         h2.SetNEntries (fNData);
         // Add histogram to array 
         if (!err && fDat2) fDat2->push_back (h2);
      }
      
      // 3D histogram
      else if ((fSubtype == 4) || (fSubtype == 5) ||
              (fSubtype == 10) || (fSubtype == 11)) {
         // not supported yet
      }
   
      // free memory
      for (int i = 0; i < 3; ++i) {
         if (fBins[i]) delete [] fBins[i];
      }
      if (fErrors) delete [] fErrors;
      if (fData) delete [] fData;
   }

//__________________________________________________________________________
   bool xsilHandlerHistogram::HandleParameter(const std::string& name, 
                     const attrlist& attr, 
                     const int& p, int N) 
   {
      if (strcasecmp (name.c_str(), "Subtype") == 0) {
         fSubtype = p;
         return true;
      } 
      else if (strcasecmp (name.c_str(), "NBinx") == 0) {
         fNBin[0] = p;
         return true;
      } 
      else if (strcasecmp (name.c_str(), "NBiny") == 0) {
         fNBin[1] = p;
         return true;
      } 
      else if (strcasecmp (name.c_str(), "NBinz") == 0) {
         fNBin[2] = p;
         return true;
      } 
      else if (strcasecmp (name.c_str(), "NData") == 0) {
         fNData = p;
         return true;
      } 
      else {
         return false;
      }
   }

//__________________________________________________________________________
   bool xsilHandlerHistogram::HandleParameter(const std::string& name, 
                     const attrlist& attr, 
                     const double& p, int N) {
      if (strcasecmp (name.c_str(), "SumWeight") == 0) {
         fSumWeight = p;
         return true;
      } 
      else if (strcasecmp (name.c_str(), "SumWeightSqr") == 0) {
         fSumWeightSqr = p;
         return true;
      } 
      else if (strcasecmp (name.c_str(), "SumWeightX") == 0) {
         fSumWeightDim[0] = p;
         return true;
      } 
      else if (strcasecmp (name.c_str(), "SumWeightXSqr") == 0) {
         fSumWeightDimSqr[0] = p;
         return true;
      } 
      else if (strcasecmp (name.c_str(), "SumWeightY") == 0) {
         fSumWeightDim[1] = p;
         return true;
      } 
      else if (strcasecmp (name.c_str(), "SumWeightYSqr") == 0) {
         fSumWeightDimSqr[1] = p;
         return true;
      } 
      else if (strcasecmp (name.c_str(), "SumWeightXY") == 0) {
         fSumWeightXY = p;
         return true;
      } 
      else if (strcasecmp (name.c_str(), "XLowEdge") == 0) {
         fLowEdge[0] = p;
         return true;
      } 
      else if (strcasecmp (name.c_str(), "YLowEdge") == 0) {
         fLowEdge[1] = p;
         return true;
      } 
      else if (strcasecmp (name.c_str(), "ZLowEdge") == 0) {
         fLowEdge[2] = p;
         return true;
      } 
      else if (strcasecmp (name.c_str(), "XSpacing") == 0) {
         fSpacing[0] = p;
         return true;
      } 
      else if (strcasecmp (name.c_str(), "YSpacing") == 0) {
         fSpacing[1] = p;
         return true;
      } 
      else if (strcasecmp (name.c_str(), "ZSpacing") == 0) {
         fSpacing[2] = p;
         return true;
      } 
      else {
         return false;
      }
   }

//__________________________________________________________________________
   bool xsilHandlerHistogram::HandleParameter(const std::string& name, 
                     const attrlist& attr, 
                     const std::string& p) {
      if (strcasecmp (name.c_str(), "Title") == 0) {
         fTitle = p;
         return true;
      } 
      else if (strcasecmp (name.c_str(), "XLabel") == 0) {
         fLabel[0] = p;
         return true;
      } 
      else if (strcasecmp (name.c_str(), "YLabel") == 0) {
         fLabel[1] = p;
         return true;
      } 
      else if (strcasecmp (name.c_str(), "ZLabel") == 0) {
         fLabel[2] = p;
         return true;
      } 
      else if (strcasecmp (name.c_str(), "NLabel") == 0) {
         fLabel[3] = p;
         return true;
      } 
      else {
         return false;
      }
   }

//__________________________________________________________________________
   bool xsilHandlerHistogram::HandleTime (
                     const std::string& name, const attrlist& attr, 
                     unsigned long sec, unsigned long nsec) 
   {
      if (strcasecmp (name.c_str(), "t0") == 0) {
         fSec = sec;
         fNsec = nsec;
         return true;
      } 
      else {
         return false;
      }
   }

//__________________________________________________________________________
   bool xsilHandlerHistogram::HandleData (
                     const std::string& name, double* x, 
                     int dim1, int dim2, int dim3, int dim4) 
   {
      if ((strcasecmp (name.c_str(), "XBins") == 0) && 
         (dim2 == 0) && (dim3 == 0) && (dim4 == 0)) {
         if (fBins[0]) delete [] fBins[0];
         fBins[0] = x;
         fBinsLen[0] = dim1;
         return true;
      } 
      else if ((strcasecmp (name.c_str(), "YBins") == 0) && 
              (dim2 == 0) && (dim3 == 0) && (dim4 == 0)) {
         if (fBins[1]) delete [] fBins[1];
         fBins[1] = x;
         fBinsLen[1] = dim1;
         return true;
      } 
      else if ((strcasecmp (name.c_str(), "ZBins") == 0) && 
              (dim2 == 0) && (dim3 == 0) && (dim4 == 0)) {
         if (fBins[2]) delete [] fBins[2];
         fBins[2] = x;
         fBinsLen[2] = dim1;
         return true;
      } 
      else if ((strcasecmp (name.c_str(), "Errors") == 0) &&
              (dim4 == 0)) {
         if (fErrors) delete [] fErrors;
         fErrors = x;
         fErrorsDim[0] = dim1;
         fErrorsDim[1] = dim2;
         fErrorsDim[2] = dim3;
         return true;
      } 
      else if ((strcasecmp (name.c_str(), "Contents") == 0) &&
              (dim4 == 0)) {
         if (fData) delete [] fData;
         fData = x;
         fDataDim[0] = dim1;
         fDataDim[1] = dim2;
         fDataDim[2] = dim3;
         return true;
      } 
      else {
         return false;
      }
      return true; 
   }



//////////////////////////////////////////////////////////////////////////
//                                                                      //
// xsilHandlerHistogram                                                 //
//                                                                      //
// XML histogram handler                                                //
//                                                                      //
//////////////////////////////////////////////////////////////////////////
   xsilHandler* xsilHandlerQueryHistogram::GetHandler (
                     const attrlist& attr) {
      attrlist::const_iterator ti = attr.find (xmlType);
      if ((ti != attr.end()) &&
         (strcasecmp (ti->second.c_str(), xmlObjTypeHistogram) == 0)) {
         return new (nothrow) xsilHandlerHistogram (
                              fDat1, fDat2, &attr, false);
      } 
      else {
         return 0;
      }
   }


}
