//
//    Shared memory partition producer class.
//
//    Methods:
//        LSMP_PROD()
//            Construct a null producer (no data access)
//
//        LSMP_PROD(LSMP *part)
//            Construct a producer to partition *part.
//
//        ~LSMP_PROD()
//            Clean up and detach producer.
//
//        char *get_buffer();
//            Get a pointer to a new buffer.
//
//        void free_buffer();
//            Free up an allocated buffer.
//
#include "lsmp_prod.hh"
#include "lsmp_int.hh"
#include <sys/ipc.h>
#include <sys/sem.h>
#include "SysError.hh"

#include <iostream>
#include <time.h>
#include <unistd.h>

//======================================  Constructors
LSMP_PROD::LSMP_PROD(void)
  : ibuf(-1)
{}

LSMP_PROD::LSMP_PROD(const char *name)
  : LSMP(name), ibuf(-1)
{}

LSMP_PROD::LSMP_PROD(const char *name, int nbuf, int size)
  : LSMP(name, nbuf, size), ibuf(-1)
{}

//======================================  Destructor
LSMP_PROD::~LSMP_PROD(void) {
    if (ibuf >= 0) return_buffer();
}

//---------------------------------------------------------------------------
//
//    int get_buffer(int flags)
//
//    Get an empty buffer for a producer
//
//    Parameters:
//        flags   Set to NOWAIT to return immediatly if no buffers 
//                are available.
//
//    Returns:
//        0       Buffer allocation failed
//        !=0     Buffer address.
//
//---------------------------------------------------------------------------
char*
LSMP_PROD::get_buffer(int flags) {
    struct sembuf sbuf;

    //----------------------------------  Get the global partition pointer
    if (ibuf>=0)  return buffer_addr(ibuf);
    if (!valid()) throw std::logic_error("Partition not attached");

    //----------------------------------  Wait for request synchronization
    bool wait = testFlag(RQSYNCH);
    while (wait) {

	//------------------------------  Look for a waiting consumer
        if (!gate(true)) return (char*) 0;
	for (int i=0 ; i < LSMP_MAXCONS ; i++) {
	    if (conptr[i].flags & EVWAIT) wait = false;
	}
	gate(false);
	if (!wait) break;

	//------------------------------  Wait for the synch semaphore
	sbuf.sem_num = gbl_synch;
	sbuf.sem_flg = (flags & NOWAIT) ? IPC_NOWAIT : 0;
	sbuf.sem_op  = -1;
	if (semop(pointer->gbl_semid, &sbuf, 1) == -1) {
	    if (errno == EINTR)       return (char*) 0;
	    else if (errno == EAGAIN) return (char*) 0;
	    else throw SysError("LSMP_PROD::get_buffer gbl_synch wait failed");
	}
    }

    //-----------------------------------  Try until we succede
    int id   = -1;
    while (id < 0) {

	//-------------------------------  Wait for a free buffer.
        sbuf.sem_num = gbl_empty;
	sbuf.sem_flg = (flags & NOWAIT)? IPC_NOWAIT : 0;
	sbuf.sem_op  = -1;
	if (semop(pointer->gbl_semid, &sbuf, 1) == -1) {
	    if (errno == EINTR)       break;
	    else if (errno == EAGAIN) break;
	    else throw SysError("LSMP_PROD::get_buffer gbl_empty wait failed");
	}

	//------------------------------  Unlink a buffer from the free list
        //                                if one is available
	if (!pointer->free.empty()) {
	    while (!gate(true));
	    id = pointer->free.remove(bufptr);
	    gate(false);


	//------------------------------  Get an unused buffer from the full
	//                                list in SCAVAGE mode
	} else if (testFlag(SCAVAGE)) {
            if (!gate(true)) return (char*) 0;
	    for (id=pointer->full.head ; id >= 0 ; id=bufptr[id].link) {
	        LSMP_buffer* pB = bufptr+id;
		if (!pB->inUse() && (testFlag(RELBUF) || pB->seen())) {
		    pointer->full.remove(bufptr,id);
		    break;
		}
	    }
	    gate(false);
	    if (id >= 0) {
	        sbuf.sem_num = gbl_full;
		sbuf.sem_flg = IPC_NOWAIT;
		sbuf.sem_op  = -1;
		semop(pointer->gbl_semid, &sbuf, 1);
	    }
	} 
    }

    //----------------------------------  Mark the buffer as owned by this
    //                                    process (status = not in queue).
    if (id >= 0) {
        bufptr[id].status    = 0;
	bufptr[id].owner_pid = my_procid();
    }
    ibuf = id;
    return buffer_addr(ibuf);
}

//---------------------------------------------------------------------------
//
//    bool LSMP::release(int length, int mask);
//
//    Distribute a full buffer to the registered consumers.
//
//    Parameters:
//      ibuf     Identifier of buffer to be distributed.
//      length   Length of data in bytes.
//      mask     Data typ bit mask.
//
//    Returns:
//      none
//
//---------------------------------------------------------------------------
void 
LSMP_PROD::release(int length, int mask, int flags) {
    int i, conwd, inx;
    ConFlags sem_mask;
    LSMP_buffer *pbuf;
    LSMP_consbk *pcon;
    struct sembuf sbuf[LSMP_CONSPWD+1];

    //----------------------------------  Get the necessary pointers
    if (ibuf < 0) return;
    if (!valid()) return;
    if ((ibuf < 0) || (ibuf >= pointer->nbuf)) return;
    pbuf = bufptr + ibuf;
    if (pbuf->queued()) {
        std::cout << "LSMP_PROD: Requested release of queued buffer (ID " 
		  << ibuf << "). Producer buffer ID reset." << std::endl;
        ibuf = -1;
	return;}

    //----------------------------------  Set the buffer parameters
    pbuf->link      = -1;
    if (!mask) mask = -1;
    pbuf->trig      = mask;
    pbuf->ldata     = length;
    pbuf->use_count = 0;
    sem_mask.zero();
    pbuf->reserve_mask.zero();
    pbuf->seen_mask.zero();
    pbuf->evt_count++;
    pbuf->fill_time=time(0);

    //----------------------------------  Access the partition control area
    while (!gate(true));

    //----------------------------------  Distribute the buffer to consumers
    pcon = conptr;
    for (i=0 ; i < LSMP_MAXCONS ; i++) {
	if (pointer->conmask.test(i)) {

	    //--------------------------  All consumers get SHOWALL events
	    if (flags & SHOWALL) {
	        pcon->seg_ctr++;
		pbuf->reserve_mask.set(i);
		sem_mask.set(i);
	        pcon->clrWait();

	    //--------------------------  Is event type OK for this consumer
	    } else if ((pcon->trig_mask & mask)) {

	        //----------------------  Give events to READALL consumers
	        if (pcon->isReadAll() || testFlag(EXPOSE)) {
		    pcon->seg_ctr++;
		    pbuf->reserve_mask.set(i);
		    sem_mask.set(i);
		    pcon->clrWait();

		//----------------------  On demand consumers
		} else if (!pcon->mxbuf) {
		    if (pcon->flags & EVWAIT) {
		        sem_mask.set(i);
			pcon->clrWait();
		    }

		//----------------------  Consumers with space available
		} else if (pcon->seg_ctr<pcon->mxbuf && pcon->skip_ctr <= 0) {
		    pcon->seg_ctr++;
		    pbuf->reserve_mask.set(i);
		    sem_mask.set(i);
		    pcon->clrWait();
		    pcon->skip_ctr = pcon->min_sep;
		} else {
		    pcon->skip_ctr--;
		}
	    }
	}
	pcon++;
    }

    //----------------------------------  Put the buffer in the right queue
    int nsem = 0;
    if (testFlag(RELBUF) && !testFlag(SCAVAGE) && !sem_mask) {
        pointer->free.link(bufptr,ibuf);
    } else {
        pointer->full.link(bufptr,ibuf);
	sbuf[nsem].sem_flg = 0;
	sbuf[nsem].sem_op  = 1;
	sbuf[nsem++].sem_num = gbl_full;
    }

    //----------------------------------  Bump empty if the buffer is free or
    //                                    scavageable
    if (testFlag(RELBUF) && !sem_mask) {
	sbuf[nsem].sem_flg = 0;
	sbuf[nsem].sem_op  = 1;
	sbuf[nsem++].sem_num = gbl_empty;
    }

    //-----------------------------------  Release global control area
    gate(false);

    //-----------------------------------  Bump the queue semaphore(s)
    semop(pointer->gbl_semid, sbuf, nsem);

    //-----------------------------------  Tell consumers a buffer awaits
    if (sem_mask.any()) {
        inx = 0;
	for (i=0 ; i<LSMP_MAXCONS ; i++) {
	    if (sem_mask[i]) {
		sbuf[inx].sem_flg = 0;
		sbuf[inx].sem_op  = 1;
		sbuf[inx].sem_num = CON_BIT(i);
		inx++;
	    }
	    conwd = CON_WORD(i);
	    if (inx && (CON_WORD(i+1) != conwd)) {
	        semop(pointer->con_semid[conwd], sbuf, inx);
		inx = 0;}
	}
	if (inx) semop(pointer->con_semid[conwd], sbuf, inx);
    }
    ibuf = -1;
    return;}

//---------------------------------------------------------------------------
//
//    bool LSMP::return_buffer();
//
//    Return a buffer without data
//
//    Parameters:
//      none
//
//    Returns:
//      none
//
//---------------------------------------------------------------------------
void 
LSMP_PROD::return_buffer(void) {
    struct sembuf sbuf;

    //----------------------------------  Get the necessary pointers
    if (ibuf < 0) return;
    if (!valid()) return;
    if (ibuf >= pointer->nbuf) {
        ibuf = -1;
	return;}
    LSMP_buffer *pbuf = bufptr + ibuf;
    if (pbuf->queued()) {
        std::cout << "LSMP_PROD: Requested release of queued buffer (ID " 
		  << ibuf << "). Producer buffer ID reset." << std::endl;
        ibuf = -1;
	return;}

    //----------------------------------  Set the buffer parameters
    pbuf->link      = -1;
    pbuf->trig      = 0;
    pbuf->ldata     = 0;
    pbuf->use_count = 0;
    pbuf->reserve_mask.zero();
    pbuf->seen_mask.zero();

    //----------------------------------  Add buffer to the free queue
    while (!gate(true));
    pointer->free.link(bufptr,ibuf);
    gate(false);

    //-----------------------------------  Bump the free semaphore
    sbuf.sem_flg = 0;
    sbuf.sem_op  = 1;
    sbuf.sem_num = gbl_empty;
    semop(pointer->gbl_semid, &sbuf, 1);

    //--------------------------------------  Mark buffer returned.
    ibuf = -1;
    return;}


//---------------------------------------------------------------------------
//
//    bool LSMP_PROD::SetID();
//
//    Return a buffer without data
//
//    Parameters:
//      none
//
//    Returns:
//      none
//
//---------------------------------------------------------------------------
void 
LSMP_PROD::SetID(int ID) {

    //----------------------------------  Get the necessary pointers
    if (!valid()) return;
    if (ibuf < 0 || ibuf >= pointer->nbuf) {
        ibuf = -1;
	return;
    }

    LSMP_buffer* pbuf = bufptr + ibuf;
    if (pbuf->queued()) {
        ibuf = -1;
	return;
    }

    //----------------------------------  Set the buffer parameters
    pbuf->data_ID = ID;
    return;
}
