/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef GDS_SHMEM_HH
#define GDS_SHMEM_HH

#include <cstddef>
#include <unistd.h>

/**  This class is basically just a system-independent wrapper for the
  *  shm ipc functions provided by *nix operating systems.
  *  \brief Shared memory partition management class.
  *  \author John Zweizig (john.zweizig@LIGO.ORG)
  *  \version $Id$
  */
class gds_shmem {
public:
    /**  Construct an empty shm descriptor.
      *  \brief Empty constructor.
      */
    gds_shmem(void);

    /**  Release a shared memory partition an destroy the descriptor.
      *  \brief destructor.
      */
    virtual ~gds_shmem(void);

    /**  Attach an existing shared memory partition with the specified 
      *  user ID number.
      *  \brief attach a shared memory partition.
      *  \param id User identifier for partition to be attached.
      *  \return True if partition was successfully mapped into memory.
      */
    bool attach(int id);

    /**  Create a shared memory partition with the specified user ID and 
      *  length in bytes.
      *  \brief Create a shared memory partition.
      *  \param id   User identifier for partition to be created.
      *  \param size Partition size.
      *  \return True if partition was successfully mapped into memory.
      */
    bool create(int id, size_t size);

    /**  Test whether a partition with a specified ID and length exists.
      *  This function fails if the partition descriptor has a partition
      *  accessed.
      *  \brief Find/access a partition.
      *  \param id    User identifier.
      *  \param size  Minimum size.
      *  \return True if the partition exists.
      */
    bool exists(int id, size_t size=0);

    /**  Get the system identifier for the current partition.
      *  \brief Partition ID.
      *  \return Partition ID.
      */
    int ident(void) const;

    /**  Test whether partition has been accessed.
      *  \brief Test if partition is attached.
      *  \return true if partition is attached.
      */
    bool is_accessed(void) const;

    /**  Test whether partition is mapped into memory.
      *  \brief Test if partition is attached.
      *  \return true if partition is attached.
      */
    bool is_attached(void) const;

    /**  Test whether partition is locked into memory.
      *  \brief Test if partition is locked.
      *  \return True if partition is locked.
      */
    bool is_locked(void) const;

    /**  Lock or unlock partition into memory. Super-user privileges are 
      *  required on solaris systems.
      *  \brief Lock (unlock) partition into memory.
      *  \param lok Lock if true.
      *  \return True if lock (unlock) operation was successful.
      */
    bool lock(bool lok);

    /**  Get the global pid used by this process for this partition.
      *  \brief  Global pid.
      *  \return Global process identifier.
      */
    pid_t my_pid(void) const;

    /**  Test if the current process is the owner uid of this partition.
      *  \brief Test partition ownership.
      *  \return True if the user is the partition owner.
      */
    bool owner(void) const;

    /**  Get the owner uid for this partition.
      *  \brief get Owner uid.
      *  \return Owner uid.
      */
    int owner_uid(void) const;

    /**  Get a constant pointer to the partition.
      *  \brief get a partition pointer.
      *  \return Constant pointer to the partition.
      */
    const void* ref(void) const;

    /**  Get a pointer to the partition.
      *  \brief Get a partition pointer.
      *  \return Pointer to the partition.
      */
    void* ref(void);

    /**  Unlock, unmap and decrement the use count of the partition.
      *  \brief Release partition.
      *  \param remove Remove the partition.
      */
    void release(bool remove=false);

    /**  The set_mmd function initializes the offset and length words at 
      *  the beginning of the partition for use by the \c new and \c new\[\] 
      *  operators. These operators allow the construction of the buffer 
      *  manager control structures in the partition.
      *  \brief Set memory management data.
      */
    void set_mmd(void);

private:
    /**  Enumerate status bits.
     */
    enum flagbits {
	kAccessed,   // Partition has been created/accessed.
	kAttached,   // Partition is mapped into the process address space.
	kLocked      // Partition is locked into physical memory.
    };

    /**  Clear the specified internal status flag
      *  \note This method is implemented as an in-line function and
      *        is therefore available only to methods implemented in 
      *        the same file.
      *  \brief Clear a flag bit.
      *  \param flg Flag bit to clear.
      */
    void clr_flag(enum flagbits flg);

    /**  Decrement the partition use count.
      *  \brief Deaccess the partition.
      *  \param remove Remove the partition.
      *  \return True if operation was successful.
      */
    bool deaccess(bool remove=false);

    /**  Find and optionally create a partition with a specified ID and 
      *  length. The partition use count is incremented and its system ID
      *  is recorded.
      *  \brief Find/access a partition.
      *  \param id    User identifier.
      *  \param size  Number of byte (for a partition to be created).
      *  \param flags Create flag.
      *  \return True if operation was successful.
      */
    bool find(int id, size_t size, int flags);

    /**  Clear the specified internal status flag
      *  \note This method is implemented as an in-line function and
      *        is therefore available only to methods implemented in 
      *        the same file.
      *  \brief Clear a flag bit.
      *  \param flg Flag bit to clear.
      */
    void set_flag(enum flagbits flg);

    /**  Map partition in to the process address space. The partition must
      *  have been accessed (see find()) before it may be mapped into 
      *  memory.
      *  \brief Map partition into the memory.
      *  \return True if partition was successfully mapped.
      */
    bool map(void);

    /**  Unmap the partition from the process address space.
      *  \brief Unmap partition from memory.
      *  \return True if partition was successfully unmapped.
      */
    bool unmap(void);

private:
    int   _flagwd;
    int   _ID;
    void* _addr;
    long  _size;
    int   _protfl;
    int   _error;
    pid_t _mypid;
};

//======================================  Allocate from a partition
void* operator new   (size_t nb, gds_shmem& part);
void* operator new[] (size_t nb, gds_shmem& part);

//======================================  Test status
inline int
gds_shmem::ident(void) const {
    return _ID;
}

inline bool
gds_shmem::is_accessed(void) const {
    return (_flagwd & (1 << kAccessed)) != 0;
}

inline bool
gds_shmem::is_attached(void) const {
    return (_flagwd & (1 << kAttached)) != 0;
}

inline bool
gds_shmem::is_locked(void) const {
    return (_flagwd & (1 << kLocked)) != 0;
}

inline pid_t 
gds_shmem::my_pid(void) const {
    return _mypid;
}

inline const void*
gds_shmem::ref(void) const {
    return _addr;
}

inline void*
gds_shmem::ref(void) {
    return _addr;
}

#endif // !defined(GDS_SHMEM_HH)
