#ifndef HTML_WRITER_HH
#define HTML_WRITER_HH

#include "html/Attrib.hh"
#include <iosfwd>
#include <string>
#include <vector>

namespace html {

/**  The writer class formats data for HTML output. It provides checks
  *  of the HTML syntax. THe writer is typically constructed and then 
  *  passed to the html::document::write method to receive he HTML
  *  document.
  *  @memo    HTML writer class.
  *  @author  J. Zweizig
  *  @version 1.1; Modified 5-May-2001.
  *  @ingroup IO_html
  */
class writer {
public:
    /**  Construct an XML file writer to format data to an output stream.
      *  @memo HTML writer constructor.
      *  @param ostr Output stream to receive formatted XML.
      */
    writer(std::ostream& ostr);

    /**  Destroy an HTML writer.
      *  @memo Writer destructor.
      */
    ~writer(void);

    /**  Output specified HTML text to the stream.
      *  @memo Write text.
      *  @param data Text to be written to the output stream.
      */
    void text(const std::string& data);

    /**  Flush the current output line to the output stream followed by an 
      *  end-of-line character. THe current line length is reset to zero
      *  @memo End an output line.
      */
    void endLine(void);

    /**  Write an html tag to the output file. A list of attribute names 
      *  and associated values are written inside the tag block. Attribute
      *  names with null value pointers are not written. The tag name is
      *  added to the writer tag list for validation.
      *  @memo Write a tag.
      *  @param tagname Pointer to the tag name.
      *  @param nattr   Number of attributes to be included in the tag.
      *  @param attr    List of pointers to attribute names.
      *  @param value   List of pointers to attribute values.
      */
    void tag(const char* tagname, int nattr=0, const char** attr=0, 
	     const char** value=0);

    /**  Write an html tag to the output file. A list of attributes is 
      *  written inside the tag block. Attribute names with default values 
      *  are not written. The tag name is added to the writer tag list for 
      *  validation.
      *  @memo Write a tag.
      *  @param tagname Pointer to the tag name.
      *  @param attr    List of attributes.
      */
    void tag(const char* tagname, const attList& attr);

     /**  Write a dataless html tag to the output file. A list of attribute 
      *  names and associated values are written inside the tag block. 
      *  Attribute names with null value pointers are not written. The 
      *  "<tag>" syntax is used and no end tag block will be generated.
      *  @memo Write a dataless tag.
      *  @param tagname Pointer to the tag name.
      *  @param nattr   Number of attributes to be included in the tag.
      *  @param attr    List of pointers to attribute names.
      *  @param value   List of pointers to attribute values.
      */
    void tagNData(const char* tagname, int nattr=0, const char** attr=0, 
	     const char** value=0);

    /**  Write an html tag to the output file. A list of attributes is 
      *  written inside the tag block. Attribute names with default values 
      *  are not written. The tag name is added to the writer tag list for 
      *  validation.
      *  @memo Write a tag.
      *  @param tagname Pointer to the tag name.
      *  @param attr    List of attributes.
      */
    void tagNData(const char* tagname, const attList& attr);

    /**  Write HTML meta-data word to the document. The HTML meta-data
      *  have the following form: \<meta http-equiv='name' content='value'\>.
      *  @memo Write HTML meta-data.
      *  @param name HTML meta-data name.
      *  @param value HTML meta-data value.
      */
    void meta(const char* name, const char* value);

    /**  Write HTML meta-data value to the document. The HTML meta-data
      *  have the following form: \<meta http-equiv='name' content='value'\>.
      *  @memo Write HTML meta-data.
      *  @param name HTML meta-data name.
      *  @param value Numberic HTML meta-data value.
      */
    void meta(const char* name, double value);

    /**  Write an integer value to the output stream.
      *  @memo Write an integer.
      *  @param N value to be written.
      */
    void Integer(int N);

    /**  Write the \</name\> end tag to the output stream. The specified
      *  tag name must match that of the most recent unclosed tag.
      *  @memo write a close tag.
      *  @param tagname Name f tag to be closed.
      */
    void endTag(const std::string& tagname);

    /**  Push an attribute list onto the top of the writer's default 
      *  attribute stack. The new attributes will replace any attributes
      *  with the same name.
      *  @memo Push the specified attributes onto the default attribute stack.
      *  @param x attribute list.
      */
    void pushDefault(const attList& x);

    /**  Get a constant reference to the current default attribute list.
      *  @memo Get the current default attribute list.
      *  @return Constant reference to the curent default attribute list.
      */
    const attList& refDefault(void) const;

    /**  Get a reference to the current default attribute list.
      *  @memo Get the current default attribute list.
      *  @return Reference to the curent default attribute list.
      */
    attList& refDefault(void);

    /**  Remove the current attribute list from the default 
      *  attribute stack.
      *  @memo Remove the current list from the default attribute stack.
      */
    void popDefault(void);

    /**  Synchronize the buffer to the output file by flushing buffered data
      *  to the output stream.
      *  @memo Flush output data.
      */
    void sync(void);

private:
    /**  Add blanks to the start of the current line as needed by the 
      *  indentation scheme.
      */
    void indent(void);

private:
    static const int MAXDEPTH=128;
    /**  Pointer to the output stream to receive the XML document.
      *  @memo Output Stream.
      */
    std::ostream*  mStream;

    /**  Current nesting depth of the output stream.
      *  Current nesting depth.
      */
    int       mDepth;

    /**  Number of chracters written on the current line.
      *  @memo current line posiiton.
      */
    int       mCursor;

    /**  List of tags for each nested structure.
      *  @memo Tag list.
      */
    const char* mNest[MAXDEPTH];

    /**  List of tags for each nested structure.
      *  @memo Tag list.
      */
    std::vector<attList> mFont;

    /**  Document type text
      */
    std::string mDocType;

};     // xsil::Xwriter

}      // namespace html

inline html::attList&
html::writer::refDefault(void) {
    return mFont.back();
}

const inline html::attList&
html::writer::refDefault(void) const {
    return mFont.back();
}


#endif  //   HTML_WRITER_HH
