/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef HTML_STYLE_HH
#define HTML_STYLE_HH

#include <string>
#include "html/Attrib.hh"

namespace html {
    class color;
    class font;
    class size;
    class writer;

    /**  Class style contains text style attributes.
      *  @memo Text style attribute.
      *  @author John Zweizig
      *  @version 1.1; Last modified March 3, 2008
      *  @ingroup IO_html
      */
    class StyleClass : public attList {
    public:
	/**  Construct a style specifier.
	  *  @brief Constructor
	  */
	StyleClass(void);

	/**  Destroy a style specifier.
	  *  @brief Destructor.
	  */
	~StyleClass(void);
	
	/**  Write an attribute to the a document.
	  *  @brief Write an attribute.
	  *  @param w Writer to whic the attribute will be written.
	  */
	void putAttr(writer& w) const;

	/**  Set the style color attribute.
	  *  @brief Set the color attribute.
	  *  @param c Color attribute.
	  *  @return Reference to this StyleClass object.
	  */
	StyleClass& setColor(const color& c);

	/**  Set the style font face attribute.
	  *  @brief Set the font attribute.
	  *  @param c Font attribute.
	  *  @return Rference to this StyleClass object.
	  */
	StyleClass& setFace(const font& c);

	/**  Set the style size attribute.
	  *  @brief Set the size attribute.
	  *  @param c Size attribute.
	  *  @return Reference to this StyleClass object.
	  */
	StyleClass& setSize(const size& c);

	/**  Set the style weight attribute.
	  *  @brief Set the weight attribute.
	  *  @param c Weight attribute.
	  *  @return Reference to this StyleClass object.
	  */
	StyleClass& setWeight(const std::string& c);
    };

    /**  Class style contains a dictionary of text style attributes.
      *  @memo Text style attribute.
      *  @author J. Zweizig
      *  @version 1.1; Last modified March 3, 2008
      *  @ingroup IO_html
      */
    class style {
    public:
	/**  Construct an empty style dictionary.
	  *  @brief Defult constructor
	  */
	style(void);

	/**  Destroy a style dictionary.
	  *  @brief Destructor.
	  */
	~style(void);

	/**  Add a style class to the dictionary.
	  *  @brief Add a style class.
	  *  @param name Style class name.
	  *  @param c    Named style class.
	  */
	void addClass(const std::string& name, const StyleClass& c);

	/**  Write the style dictionary to the document.
	  *  @brief Write out the style dictionary.
	  *  @param w Html writer.
	  */
	void write(writer& w) const;

	/**  Test whether the style dictionary is empty.
	  *  @brief Test if empty.
	  *  @return True if no styles are defined.
	  */
	bool empty(void) const;

    private:
	typedef std::map<std::string, StyleClass> StyleDict;
	typedef StyleDict::iterator style_iterator;
	typedef StyleDict::const_iterator const_style_iterator;

	StyleDict mDict;
    };

}

inline bool
html::style::empty(void) const {
    return mDict.empty();
}

#endif //  HTML_STYLE_HH
