/* -*- mode: c++; c-basic-offset: 4; -*- */
//
#ifndef __EXTENSIONS__
#define __EXTENSIONS__
#endif
#ifndef _BSD_SOURCE
#define _BSD_SOURCE
#endif
#include "PConfig.h"
#include "NDS2Socket.hh"
#include "sockutil.h"
#include "Time.hh"
#include "EggTimer.hh"
#include "Interval.hh"
#include "lcl_array.hh"
#include <fcntl.h>
#include <cerrno>
#include <unistd.h>
#include <iostream>
#include <sstream>
#include <cstring>
#include <cstdio>

#ifdef HAVE_SASL
#include <sasl/sasl.h>
#include <sasl/saslutil.h>
#endif

#define _TIMEOUT 30000000	/* 30 sec */
#ifdef P__WIN32
#define MSG_DONTWAIT 0
#endif

#define DAQD_ERROR 0x0001
#define DAQD_SASL  0x0018

//////////////////////////////////////////////////////////////////////////
//                                                                      //
// Forwards							        //
//                                                                      //
//////////////////////////////////////////////////////////////////////////

using namespace sends;
using namespace std;
using namespace thread;

static const DAQC_api::wait_time tSample = 0.1;
bool NDS2Socket::sasl_init = false;

//======================================  Constructors
NDS2Socket::NDS2Socket(void)
    : mSocket(-1), mOffline(0), mAuthContxt(0)
{
}

NDS2Socket::NDS2Socket(const string& ipaddr, int ipport, long RcvBufferLen)
    : mSocket(-1), mOffline(0), mAuthContxt(0)
{
    //----------------------------------  Open the socket
    open(ipaddr, ipport, RcvBufferLen);
}

//======================================  Destructor
NDS2Socket::~NDS2Socket() {
    if (isOpen()) close();
}

//======================================  Perform sasl based authentication
int 
NDS2Socket::authenticate(const char* server) {
    int rc = 0;
#ifdef HAVE_SASL
    //----------------------------------  Global sasl client initialization.
    if (!sasl_init) {
	rc = sasl_client_init(0);
	if (rc) return rc;
	sasl_init = true;
    }

    //----------------------------------  Allocate a temporary buffer.
    int buflen = 2048;
    lcl_array<char> buf(buflen);

    //----------------------------------  Get mechanism list
    int nch = gets(buf, buflen);
    if (nch < 0) return DAQD_ERROR;

    //----------------------------------  Client new connection
    sasl_conn_t* conn = 0;
    rc = sasl_client_new("nds2",     // The service we are using
			 server,     // The fully qualified domain name of
			             // the server we're connecting to
			 NULL, NULL, // Local and remote IP address strings
				     // (NULL disables mechanisms
				     // which require this info)*/
			 NULL,       // connection-specific callbacks
			 0,          // security flags
			 &conn       // alloc on success
			 );
    if (rc != SASL_OK) {
	cerr << "sasl_client_new failed, rc = " << rc << endl;
	return rc;
    }
    mAuthContxt = conn;

    //-----------------------------------  Start authentication interchange.
    //                                     decide on a mechanism
    const char *out, *mechusing;
    unsigned int outlen;
    rc = sasl_client_start(conn,      // the context from above
			   buf,       // list of mechanisms from the server
			   NULL,      // filled if an interaction is needed
			   &out,      // filled in on success
			   &outlen,   // filled in on success
			   &mechusing
			   );
    if (rc != SASL_OK && rc != SASL_CONTINUE) {
	cerr << "error detail: " << sasl_errdetail(conn) << endl;
	sasl_dispose(&conn);
	mAuthContxt = 0;
	return DAQD_ERROR;
    }

    //-----------------------------------  Send the mechanism
    int lmech = strlen(mechusing);
    puts(mechusing, lmech);

    ///----------------------------------  interact
    while (rc == SASL_CONTINUE) {
	//-------------------------------  Send the line formatted by SASL
	nch = puts(out, outlen);
	if (nch < 0) {
	    rc = DAQD_ERROR;
	    cerr << "nds2_authenticate: Error sending client string: "
		 << nch << endl;
	    break;
	}

	//-------------------  Get a response from the server
	nch = gets(buf, buflen);
	if (nch < 0) {
	    rc = DAQD_ERROR;
	    cerr << "nds2_authenticate: Error reading server string: " 
		 << nch << endl;
	    break;
	}

	//-------------------  Turn the crank
	rc = sasl_client_step(conn,   // our context
			      buf,    // the data from the server
			      nch,    // it's length
			      NULL,   // Client interactions?
			      &out,   // filled in on success
			      &outlen // filled in on success
			      );
    }

    //-----------------------  See how it all went
    if (rc != SASL_OK) {
	cerr << "nds2_authenticate: Error stepping client: " << rc << endl;
	sasl_dispose(&conn);
	mAuthContxt = 0;
	return DAQD_ERROR;
    }

    //-----------------------  Send the last output line
    puts(out, outlen);
#else
    rc = 1;
#endif
    return rc;
}

//======================================  Put a line to sasl
int
NDS2Socket::gets(char* buf, int buflen) {
    int l = 0;
#ifdef HAVE_SASL
    for (l=0; l<buflen; l++) {
	int rc = read(mSocket, buf+l, 1);
	if (rc == 0) {
	    printf("nds2_gets: Unexpected EOF\n");
	    break;
	} else if (rc < 0) {
	    perror("nds2_gets error");
	    l = -1;
	    break;
	}
	if (!buf[l] || buf[l] == '\n') {
	    buf[l] = 0;
	    break;
	}
    }
    if (l > 0) {
	unsigned int lv = 0;
	sasl_decode64(buf, l, buf, buflen, &lv);
	l = lv;
    } else {
	cerr << "NDS2Socket::gets: Error receiving string: " << l << endl;
    }
#endif
    return l;
}

//======================================  Put a line to sasl
int
NDS2Socket::puts(const char* buf, int len) {
    int rc = -1;
#ifdef HAVE_SASL
    int   ltmp = ((len / 3) + 1) * 4 + 2;
    lcl_array<char> tbuf(ltmp);
    unsigned int ldat = 0;
    sasl_encode64(buf, len, tbuf, ltmp, &ldat);
    tbuf[ldat++] = '\n';
    rc = send(mSocket, tbuf, ldat, 0);
#endif
    return rc;
}


//======================================  Open a socket
int 
NDS2Socket::open(const string& ipaddr, int ipport, long RcvBufferLen) 
{
    semlock lockit (mux);
    struct sockaddr_in socknam;
    //char version[4];
    long size;
   
    //----------------------------------  Make sure the socket isn't open
    if (isOpen()) {
	if (mDebug) cerr << "Socket is open" << endl;
	return -1;
    }

    //----------------------------------  Get a socket
    mSocket = socket(PF_INET,SOCK_STREAM,0);
    if (mSocket < 0) {
	if (mDebug) perror("Error in socket");
	return -1;
    }
      
    //----------------------------------  Bind the socket to a port.
    socknam.sin_family      = AF_INET;
    socknam.sin_port        = 0;
    socknam.sin_addr.s_addr = 0;
    int len                 = sizeof(socknam);
    if (::bind(mSocket, (struct sockaddr *)&socknam, len) < 0) {
	if (mDebug) perror("Error in bind");
	return -1;
    }

    //----------------------------------  Connect to the server.
    socknam.sin_family      = AF_INET;
    socknam.sin_port        = htons (ipport);
    if (nslookup (ipaddr.c_str(), &socknam.sin_addr) < 0) {
	if (mDebug) {
	    cerr << "Address lookup failed for host: " << ipaddr << endl;
	    perror("System error");

	}
	return -1;
    }

    wait_time timeout = _TIMEOUT / 1000000;
    if (connectWithTimeout (mSocket, (struct sockaddr *)&socknam, 
			    sizeof (socknam), timeout) < 0) {
	if (mDebug) perror("connectWithTimeout failed");
	return -1;
    }

    mOpened = true;
    setRcvBufLen(mSocket, RcvBufferLen);

    //----------------------------------  Perform authentication handshake
    int rc = SendRequest("authorize\n", 0, 0, &size, timeout );
    if (rc == DAQD_SASL) {
#ifdef HAVE_SASL
	rc = authenticate(ipaddr.c_str());
#endif
    }
    if (rc) {
	if (mDebug) cerr << "Authentication failed" << endl;
    	::close(mSocket);
    	mOpened = false;
    	return rc ? rc : -1;
    }
   
    //----------------------------------  Get the server version number
    mVersion = 0;
    mRevision = 0;
    //rc = SendRequest("version;\n", version, sizeof(version), &size, 
    //		       timeout );
    //if (rc || (size != 4)) {
    //	::close(mSocket);
    //	mOpened = false;
    //	return rc ? rc : -1;
    //}

    //mVersion = CVHex(version, 4);
    //rc = SendRequest("revision;\n", version, 4, &size, timeout);
    //if (rc || (size != 4)) {
    //	::close(mSocket); 
    //	mOpened = false;
    //	return rc ? rc : -1;
    //}

    //mRevision = CVHex(version, 4);
    if (mDebug) cerr << "Connected to server version " << Version() << endl;
   
    //----------------------------------  Done, Return.
    return rc;
}

//======================================  Close a socket
void NDS2Socket::close(void) 
{
    semlock lockit (mux);
#ifdef HAVE_SASL
    if (mAuthContxt) {
	sasl_dispose(reinterpret_cast<sasl_conn_t**>(&mAuthContxt));
	mAuthContxt = 0;
    }
#endif
    if (mOpened) {
	StopWriter();
	SendRequest("quit;\n");
	::close(mSocket);
	mOpened = false;
    }
    mRequest_List.clear();
    mWriterType = NoWriter;
}

//======================================  flush data from socket
void NDS2Socket::flush () {
    semlock lockit (mux);
    const int buflen = 16 * 1024;
    char text[buflen];
   
    int i = 0;
    int rc;
    do {
	rc = recv (mSocket, text, buflen, MSG_DONTWAIT);
    } while ((rc >= buflen) && (++i < 100));
}

//======================================  Stop data transfer
int NDS2Socket::StopWriter() {
    if (mAbort) *mAbort = true;
    return 0;
}

//======================================  Build NDS2 channel list.
ostream&
list_chans(ostream& str, const DAQC_api::chan_list& list) {
    str << "{";
    for (DAQC_api::const_channel_iter i=list.begin(); i!=list.end(); ++i) {
	if (i != list.begin()) str << " ";
	str << i->mName;
    }
    return str << "}";
}

//======================================  Request a stream of channel data
int 
NDS2Socket::RequestOnlineData (double stride, wait_time timeout) 
{
    semlock lockit (mux);

    //----------------------------------  Build the request
    ostringstream request;
    request << "get-online-data 0 " << stride << " ";
    list_chans(request, mRequest_List);
    request << ";\n";

    //----------------------------------  Send the request
    char transid[8];
    int rc = SendRequest(request.str(), transid, sizeof(transid), 0, timeout);
    if (rc) {
	cerr << "Failure in reading transaction id. error: " << rc << endl;
	return rc;
    }

    //----------------------------------  Read in the online word
    int ldata = RecvRec((char*) &mOffline, sizeof(mOffline), false, timeout);
    if (ldata != sizeof(mOffline)) {
	cerr << "Incorrect Offline flag length: " << ldata << endl;
	return ldata;
    }
    if (mDebug) cout << "Offline flag: " << mOffline << endl;

    //----------------------------------  Done, return
    return rc;
}

//======================================  Request a stream of channel data
int NDS2Socket::RequestData (unsigned long start, unsigned long duration,
			     wait_time timeout)
{
    semlock lockit (mux);

    //----------------------------------  Build the request
    ostringstream request;
    request << "get-data " << start << " " << start+duration << " ";
    // if (duration > 32) request << "32 ";
    list_chans(request, mRequest_List);
    request << ";\n";

    //----------------------------------  Send the request
    char transid[8];
    int rc = SendRequest(request.str(), transid, sizeof(transid),
			 0, timeout);
    if (rc) {
	if (mDebug) cerr << "RequestData: Error reading transaction ID, rc = "
			 << rc << endl;
    } else {
	if (mDebug) cerr << string(transid,8) << " = " << CVHex(transid, 8) 
			 << endl;

	//------------------------------  Read in the online flag
	int ldata = RecvRec((char*)&mOffline, sizeof(mOffline), true, timeout);
	if (mDebug) cerr << "Offline tag: " << mOffline << endl;
	if (ldata != sizeof(mOffline)) return ldata;
    }

    //----------------------------------  Done, return
    return rc;
}

//======================================  Get a channel data
int
NDS2Socket::GetData(wait_time timeout) {
    semlock lockit(mux);
    int rc;
    do {
	rc = RecvData(timeout);
	if (mDebug > 2) cout << "GetData: RecvData rc=" << rc << endl;
	if (rc == -2) {
	    long block_len = mRecvBuf.ref_header().Blen 
		           - (sizeof(DAQDRecHdr) - sizeof(int));
	    rc = RecvReconfig(block_len, timeout);
	}
    } while (rc == -2);
    return rc;
}

//======================================  Receive proprietary data into buffer
int 
NDS2Socket::RecvData(wait_time timeout) 
{
    //----------------------------------  Read in the data header. Calculate
    //                                    the record length.
    int rc = RecvRec(reinterpret_cast<char*>(&mRecvBuf.ref_header()), 
		     sizeof(DAQDRecHdr), true, timeout);
    if (rc != sizeof(DAQDRecHdr)) {
	return -1;
    }
    SwapHeader();

    //----------------------------------  Optionally dump the header.
    DAQDRecHdr& h = mRecvBuf.ref_header();
    if (mDebug) {
	cerr << "Record Header: BLen=" << h.Blen << " Secs=" << h.Secs 
	     << " GPS=" << h.GPS << " NSec=" << h.NSec << " SeqNum=" 
	     << h.SeqNum << endl;
    }
    if (h.Secs == -1) return -2;

    //----------------------------------  Calculate the remaining data
    int ndata = h.Blen - (int)(sizeof(DAQDRecHdr) - sizeof(int));
    if (ndata < 0) {
	ndata = 0;
    }

    //----------------------------------  Read in the data body
    mRecvBuf.reserve(ndata);
    rc = RecvRec (mRecvBuf.ref_data(), ndata, true, timeout);
    if (rc <= 0) return rc;

    //----------------------------------  Calculate offsets and swap data
    SwapData();
    return rc;
}

//======================================  Get a list of available channels
int 
NDS2Socket::addAvailable (chantype chanty, long gps, vector<DAQDChannel>& list, 
		       wait_time timeout) {
    semlock lockit (mux);

    //----------------------------------  Request the channel list.
    ostringstream request;
    request << "get-channels " << gps << " " 
	    << DAQDChannel::cvt_chantype_str(chanty) << ";\n";

    int rc = SendRequest(request.str(), 0, 0);
    if (rc) return rc;

    //----------------------------------  Get the (incremental) channel count
    int nChan = 0;
    rc = RecvInt(nChan, timeout);    
    if (rc != sizeof(int)) return rc;

    if (nChan <= 0) {
	cerr << "NDS2Socket::Available: Can't detemine number of channels" 
	     << endl;
	return -1;
    }
    if (mDebug) cout << "Available: Number of channels =" << nChan << endl;
    list.reserve(list.size() + nChan);

    /*----------------------------------  Loop over channels              */
    rc = nChan;
    string instr, typestr, datstr;
    for (int i=0; i < nChan; i++) {
        int len = RecvStr(instr, timeout);
	if (len < 0) {
	    rc = -1;
	    break;
	}
	if (!instr[len-1]) instr.erase(len-1, 1);
	if (mDebug) cout << "Available: channel string " << instr << endl;
	istringstream inp(instr);
	DAQDChannel chan;	
	inp >> chan.mName >> typestr >> chan.mRate >> datstr;
	if (mDebug) cout << "Available: Parsed string: Channel=" 
			 << chan.mName << " chan-type=" << typestr 
			 << " rate=" << chan.mRate << " data-type='" 
			 << datstr << "' (len: " << datstr.size() << ")" 
			 << endl; 
	chan.mChanType = DAQDChannel::cvt_str_chantype(typestr);
	chan.mDatatype = DAQDChannel::cvt_str_datatype(datstr);
	list.push_back(chan);
    }
    return rc;
}

//======================================  Get a list of available channels
int 
NDS2Socket::Times (chantype ty, unsigned long& start, unsigned long& duration, 
		   wait_time timeout) 
{
    semlock lockit (mux);

#if 0
    //----------------------------------  Build the request
    string request = "get-times;\n";
   
    //----------------------------------  Send the request
    char trans[8];
    int rc = SendRequest(request.c_str(), trans, sizeof(trans), 0, timeout);
    if (mDebug) cerr << trans << " = " << CVHex (trans, 8) << endl;
    if (rc) {
	return rc;
    }
   
    //----------------------------------  Read in the header
    int ldata = RecvRec((char*) &mOffline, sizeof(mOffline), true, timeout);
    if (mDebug) cerr << mOffline << endl;
    if (ldata != sizeof(mOffline)) 
	return ldata;
   
    //----------------------------------  Get an empty data block
    rc = RecvData(timeout);
    if (rc) {
	if (mDebug) cerr << "times failed" << rc << endl;
	return rc;
    }
    start    = mRecvBuf.ref_header().GPS;
    duration = mRecvBuf.ref_header().Secs;
#else
    int   rc = 0;
    start    = 0;
    duration = 900000000;
#endif

    //----------------------------------  Done, return
    return rc;
}

//======================================  Receive data into a buffer
struct nds2_chan_status {
    int status;
    int offset;
    int data_type;
    float rate;
    float signal_offset;
    float signal_slope;
};

//======================================  Receive data into a buffer
int
NDS2Socket::RecvReconfig(count_type block_len, wait_time maxwait){
    count_type nchannels = block_len / sizeof(nds2_chan_status);
    if (nchannels * sizeof(nds2_chan_status) != block_len) {
	cerr << "Channel reconfigure block length has bad length "
	     << block_len << endl;
	return -1;
    }
    if (nchannels != mRequest_List.size()) return -1;

    if (mDebug) cout << "receiving reconfigure for " << nchannels 
		     << " channels" << endl;

    for (count_type i=0; i < nchannels; i++) {
	DAQDChannel* lp = &mRequest_List[i];
	float rate;
	int dtype;
	if (RecvInt(lp->mStatus)   != sizeof(int)) return -1;
	if (RecvInt(lp->mBOffset)  != sizeof(int)) return -1;
	if (RecvInt(dtype)         != sizeof(int)) return -1;
	lp->mDatatype = static_cast<datatype>(dtype & 0xffff);
	lp->mChanType = static_cast<chantype>(dtype >> 16);
	if (RecvFloat(rate)        != sizeof(int)) return -1;
	lp->mRate = rate;
	if (RecvFloat(lp->mOffset) != sizeof(int)) return -1;
	if (RecvFloat(lp->mSlope)  != sizeof(int)) return -1;
    }
    return -2;
}

//======================================  Receive data into a buffer
int 
NDS2Socket::RecvRec(char *buffer, long length, bool readall, wait_time maxwait){
    EggTimer egg_timer;
    int fileflags = fcntl (mSocket, F_GETFL, 0);
    if (fileflags == -1) {
	if (mDebug) perror("RecvRec: fcntl(GETFL) error");
	return -1;
    }

    if (fcntl (mSocket, F_SETFL, fileflags & ~O_NONBLOCK) == -1) {
	if (mDebug) perror("RecvRec: fcntl(SETFL) error");
	return -1;
    }

    char* point = buffer;
    long  nRead = 0;
    do {
	wait_time tWait = maxwait;
	if (maxwait >= 0) {
	    tWait -= egg_timer.elapsed();
	    if (tWait < 0) tWait = 0;
	}
	if (mAbort && (tWait > tSample || maxwait < 0)) tWait = tSample;
	if (mDebug > 2) cout << "RecvRec: Waiting " << tWait << "/" 
			     << maxwait << " sec" << endl;
	int nset = ::socketWait(mSocket, tWait, wm_read);
	if (mDebug > 2) cerr << "RecvRec: socketWait return code: " 
			     << nset << endl;
	if (nset < 0) {
	    if (errno == EINTR) {
		cerr << "NDS2Socket::RecvRec Signal received in select "
		     << string().insert((unsigned int)0,44,'+') << endl;
		continue;
	    } else {
		perror("NDS2Socket: Error in select()");
		nRead = -12;
		break;
	    }
	} else if (nset== 0) {
	    if (tWait > 0) continue; // not the last chance
	    if (readall && !nRead) nRead = -13;
	    break;
	}

	/* set socket to non blocking */
	if (fcntl (mSocket, F_SETFL, fileflags | O_NONBLOCK) == -1) {
	    if (mDebug) perror("RecvRec: fcntl(SETFL) error");
	    return -1;
	}

	int nB = ::recv(mSocket, point, length - nRead, 0);
	if (nB < 0) {
	    if (errno == EAGAIN || errno == EINTR) {
		nB = 0; // random interrupt - treat as a failed read.
	    } else {
		perror("NDS2Socket::RecvRec failed");
		nRead = -10;
		break;
	    }
	} else if (nB == 0) {
	    if (mDebug) cerr << "NDS2Socket::RecvRec: Unexpected EOF" << endl;
	    nRead = -10;
	    break;
	}

	//------------------------------  set socket to non blocking
	if (fcntl (mSocket, F_SETFL, fileflags & ~O_NONBLOCK) == -1) {
	    if (mDebug) perror("RecvRec: fcntl(SETFL) error");
	    return -1;
	}

	point += nB;
	nRead += nB;
	if (tWait == 0 || (mAbort && *mAbort)) {
	    nRead = -13;
	    break;
	}
    } while (readall && nRead < length);
    if (fcntl (mSocket, F_SETFL, fileflags) == -1) {
	if (mDebug) perror("RecvRec: fcntl(SETFL) error");
	return -1;
    }
    if (mDebug > 2) cerr << "RecvRec read " << nRead << "/" << length << endl;
    return nRead;
}

//======================================  Send data from a buffer
int 
NDS2Socket::SendRec(const char *buffer, long length, wait_time maxwait) {
    EggTimer egg_timer;

    /* set socket to non blocking */
    int fileflags = fcntl (mSocket, F_GETFL, 0);
    if (fileflags == -1) {
	if (mDebug) perror("SendRec: fcntl(GETFL) error");
	return -1;
    }

    Time endTime = Now();
    if (maxwait >= 0  || mAbort) {
	endTime += maxwait;
	if (fcntl (mSocket, F_SETFL, fileflags | O_NONBLOCK) == -1) {
	    if (mDebug) perror("SendRec: fcntl(SETFL) error");
	    return -1;
	}
    }

    long nWrite = 0;
    const char* point = buffer;
    while (nWrite < length) {
	//------------------------------ Calculate wait step.
	wait_time wstep = maxwait;
	if (maxwait >= 0) {
	    wstep -= egg_timer.elapsed();
	    if (wstep < 0) wstep = 0;
	}
	if (mAbort && (wstep > tSample || maxwait < 0)) wstep = tSample;
	if (mDebug) cerr << "SendRec: Waiting " << wstep << "/" 
			 << maxwait << " sec" << endl;

	//------------------------------ Use select to test for timeout
	int nset = socketWait(mSocket, wstep, wm_write);
	if (nset < 0) {
	    perror("NDS2Socket::SendRec Error in select(write)");
	    nWrite = -12;
	    break;
	} else if (nset == 0 && !mAbort) {
	    nWrite = -13;
	    break;
	}

	int nB = send (mSocket, point, length - nWrite, 0);
	if (nB == -1) {
	    if (mDebug) perror("SendRec: send failed with error");
	    nWrite = -1;
	    break;
	}
	point += nB;
	nWrite += nB;
	if (wstep == 0.0 || (mAbort && *mAbort)) break;
    }
    if (mDebug > 1) cerr << "SendRec write " << nWrite << "/" << length << endl;
    if (maxwait >= 0 || mAbort) {
	fcntl (mSocket, F_SETFL, fileflags);
    }
    return nWrite;
}

//======================================  Send a request, wait for response.
int
NDS2Socket::SendRequest(const string& text, char *reply, long length, 
			long *Size, wait_time timeout) {
    char status[5];
   
    //----------------------------------  Send the request
    if (mDebug) cerr << "NDS2 request: " << text << endl;
    int rc = SendRec (text.c_str(), text.size(), timeout);
    if (rc <= 0) return rc;
   
    //----------------------------------  Read the reply status.
    rc = RecvRec (status, 4, true, timeout);
    if (rc != 4) {
	if (mDebug) cerr << "SendRequest: response length = " << rc << endl; 
	return -1;
    }

    status[4] = 0;
    if (mDebug) cerr << "SendRequest: response = " << status << endl;
    rc = CVHex(status, 4);

    //----------------------------------  Read the reply text.
    if (!rc && length != 0 && reply != 0) {
	int len = RecvRec(reply, length, true, timeout);
	if (len < 0) {
	    if (mDebug) cerr << "SendRequest: reply length = " << rc << endl; 
	    rc = len;
	} else {
	    if (len < length) reply[len] = 0;
	    if (mDebug) cerr << "SendRequest: reply text = " 
			     << string(reply, len) << endl;
	    if (Size)       *Size = len;
	}
    }
    return rc;
}

//======================================  Set an epoch of interest.
int
NDS2Socket::SetEpoch(const std::string& epoch) {
    string cmd = "set-epoch ";
    cmd += epoch;
    cmd += ";";
    long replysize;
    return SendRequest(cmd, 0, 0, &replysize, 10.0);
}

//======================================  Wait for a message to arrive
//                                        WaitforData() <= 0 means no data.
int 
NDS2Socket::WaitforData (bool poll) {
    int nset = 0;
    if (poll) nset = socketWait(mSocket,  0.0, wm_read);
    else      nset = socketWait(mSocket, -1.0, wm_read);
    return nset;
}
