// Yo, Emacs! This is -*- C++ -*-
//
// $Id: StdRange.cc 6619 2012-03-12 23:58:12Z john.zweizig@LIGO.ORG $
//
// Author: David Chin <dwchin@umich.edu>
//
//         2477 Randall Lab, Univ. of Michigan
//         500 E. University Ave.
//         Ann Arbor, MI 48109
//         +1-734-764-5146
//         +1-734-730-1274
//
// Written based on idea by K. Riles <kriles@umich.edu>

#include "osc/StdRange.hh"

using namespace std;

osc::StdRange::StdRange(DaccAPI      *access,
                        ostream    *outstream,
                        string      chanName,
                        osc::IntParMap &intparams,
                        osc::DblParMap &dblparams,
                        int debug)
    : OperStateCond(access, outstream, chanName, "stdrange",
                    true, intparams, dblparams, debug)
{
    return;
}


bool osc::StdRange::satisfied()
{
    if (mDebug > 0) {
        cout << "osc::" << mName << ": entered satisfied()" << endl;
    }

    // Read in time-series data
    TSeries *ptx = dataSeries();

    // If the time series is empty, return False
    if (ptx->getNSample() <= 0 || ptx->isEmpty() == true)
        return false;

    // If this condition has already been checked for in the current Frame,
    // return the buffered result
    Time t0 = ptx->getStartTime();
    if (t0 == mCurResults.mT0) {
        return mCurResults.mSatisfiedP;
    } else {

        updateResultsBuffer();
        
        //
        // Current stuff
        //
        mCurResults.mT0 = t0;

        try {
            readParamsFromChanMaybe();
        }

        catch (DataRead_err &err) {
            err.printErrMsg();
            return false;
        }

                
        // compute Standard deviation of time-series
        double avesqr = (*ptx)*(*ptx) / ptx->getNSample();
	double sqrave = pow(ptx->getAverage(),2.);
	double rmsdev = avesqr-sqrave;
	if (rmsdev>=0.0) {
	  mCurResults.mDoubleVal = sqrt(rmsdev);
	} else if (rmsdev>=-1.e-10*avesqr) {
	  mCurResults.mDoubleVal = 0.;
	} else {
	  cerr << "Warning from StdRange: avesqr=" << avesqr << " sqrave=" << sqrave << " --> Imaginary standard deviation" << endl;
	  mCurResults.mDoubleVal = 0.;
	}

        if (mDebug > 0) {
            printf("osc::StdRange on '%s': STD = %6.4f\n",
                   mChannel.c_str(), mCurResults.mDoubleVal);
        }

        if (mCurResults.mDoubleVal > mDblParams["lo"].value() &&
            mCurResults.mDoubleVal < mDblParams["hi"].value())
            mCurResults.mSatisfiedP = true;
        else
            mCurResults.mSatisfiedP = false;

        return mCurResults.mSatisfiedP;
    }
}
