//
//   macro to test the fft normalization
//
inline double 
RMS(const TSeries& ts) {
    return (ts*ts)/double(ts.getNSample());
}

inline void 
Test(const char* Name, double actual, const char* xTitle, 
     double expected, double error) {
    cout << "The " << Name << " is " << actual;
    if (xTitle) {
        cout << ", it should equal the " << xTitle << " ("
	     << expected << "+/-" << error << ")" << endl;
    } else {
         cout << ", it should be " << expected << "+/-" << error << endl;
    }
    if (actual < expected-2*error || actual > expected+2*error) {
        cout << endl;
	cout << "*************************" << endl;
	cout << "*      Error  Found     *" << endl;
	cout << "*************************" << endl;
    }
}

void 
testfft(void)
{
    //---------    Real gaussian series
    cout << "Create a gaussian noise time series, 1 sec, 1024 samples/s..."; 
    TSeries t1(Time(0), Interval(1./1024.0), 1024, GaussNoise(0.0));
    cout << " done!" << endl;
    Test("RMS", RMS(t1), 0, 1.0, 1.0/sqrt(1024.0));
    cout << "Now create an FSeries from the TSeries...";
    FSeries f1(t1);
    cout << " done!" << endl;
    Test("total power from 0-Fnyquist", f1.Power(0,512), 
	 "RMS", 1, sqrt(1.0/1024.0));

    //---------    Biassed gaussian series
    cout << "Bias the gaussian series,by 1 unit..."; 
    TSeries t1b(t1);
    t1b += 1.0;
    cout << " done!" << endl;
    Test("RMS", RMS(t1b), 0, 2.0, 1.0/sqrt(1024.0));
    cout << "Now create an FSeries from the TSeries...";
    FSeries f1b(t1b);
    cout << " done!" << endl;
    Test("total power from 0-Fnyquist", f1b.Power(0,512), 
	 "RMS", 2.0, sqrt(1.0/1024.0));
    
    //--------  Biassed complex gaussian series
    cout << "Convert the series above to complex..."; 
    TSeries t1c(t1b);
    t1c.Convert(DVector::t_complex);
    cout << " done!" << endl;
    Test("RMS", RMS(t1c), 0, 2.0, 1.0/sqrt(1024.0));
    cout << "Now create an FSeries from the TSeries...";
    FSeries f1c(t1c);
    cout << " done!" << endl;
    Test("total power from 0-Fnyquist", f1c.Power(0,512), 
	 "RMS", 2.0, sqrt(1.0/1024.0));

    //---------    Real Sine series
    cout << endl;
    cout << "Create a 64Hz Sinusoidal time series, 1 sec, 1024 samples/s..."; 
    TSeries t2(Time(0), Interval(1./1024.0), 1024, Sine(64.0));
    cout << " done!" << endl;
    Test("RMS", RMS(t2), 0, 0.5, 0.5/sqrt(1024.0));
    cout << "Now create an FSeries from the TSeries...";
    FSeries f2(t2);
    cout << " done!" << endl;

    Test("total power from 0-Fnyquist", f2.Power(0.0, 512.0),
	 "RMS", 0.5, 0.5/sqrt(1024.));
    Test("amplitude at 64Hz", f2(64.0).Mag(),
	 "1/sqrt(2)", 0.707107, sqrt(2.0/1024.0));

    //---------    Complex Sine series
    cout << endl;
    cout << "Create a 64Hz complex series by heterodyning the 64Hz series"; 
    TSeries t2c = t2.fShift(0.0);
    cout << " done!" << endl;
    Test("RMS", RMS(t2c), 0, 0.5, 0.5/sqrt(1024.));
    cout << "Now create an FSeries from the TSeries...";
    FSeries f2c(t2c);
    cout << " Done!" << endl;
    Test("Total power from 0 to Fnyquist", f2c.Power(0.0, 512.0), 
	 "RMS", 0.5, 0.5/sqrt(1024.0)); 
    Test("amplitude at 64Hz", f2c(64.0).Mag(), 0, 1.0/sqrt(2.0), 
	 1.0/sqrt(2*1024.0));

    //--------     Decimated series
    cout << endl;
    cout << "Down-convert the 64Hz series by 56Hz, filter and decimate it...";
    Filter f(40,1024.0);
    f.dFirW(40,"Hanning","LowPass", 30.0);
    double df = -56.0;
    TSeries t2d = t2.fShift(df).decimate(32, f);
    cout << " Done!" << endl;
    cout << "Now make it into an FSeries...";
    FSeries f2d(t2d);
    cout << " Done!" << endl;
    FSeries Fx = f.Xfer();
    double eFx = Fx(8.0).Mag();
    Test("amplitude at 64Hz", f2d(64.0).Mag(), "<filter-eff>/sqrt(2)",
	 eFx/sqrt(2.0), eFx/sqrt(2.0*1024.0));
    
    //---------    Real Sine series
    cout << endl;
    cout << "Create a 64Hz real series with 2048 samples at a 1024Hz rate.."; 
    TSeries t3(Time(0), Interval(1/1024.), 2048, Sine(64.0));
    cout << " done!" << endl;
    Test("RMS", RMS(t3), 0, 0.5, 0.5/sqrt(2048.0));
    cout << "Now create a Fseries from the TSeries...";
    FSeries f3(t3);
    cout << " done!" << endl;
    Test("Total power from 0-Fnyquist", f3.Power(0, 512), 
	 "RMS", 0.5, .5/sqrt(2048.));
    Test("amplitude at 64Hz", f3(64).Mag(), "1/sqrt(2*bw)", 1, 1/sqrt(2048.));
	 
    //---------    Inverse transformation.
    cout << endl;
    cout << "Perform an inverse transformation on the previous data..."; 
    TSeries t3a(f3);
    cout << " Done!" << endl;
    Test("RMS", RMS(t3a), 0, 0.5, 0.5/sqrt(2048.0));
    Test("RMS difference from the original", RMS(t3a-t3), 0, 0, 
	 1.0/sqrt(2048.0));

    cout << endl;
    cout << "Once again with the double ended FSeries"; 
    TSeries t3c(f2c);
    cout << " Done!" << endl;
    Test("RMS", RMS(t3c), 0, 0.5, 0.5/sqrt(1024.0));
    Test("RMS difference from the original", RMS(t3c-t2c), 0, 0, 
	 1.0/sqrt(1024.0));
}
