//
//    Chart produces a constantly updating Strip-Chart. Each channel 
//    displays the power produced on a single channel in a limited band
//    width. By default, the scale is logarithmic with a factor of 10 
//    corresponding to half the pen spacing. The Channel configuration 
//    for the strip chart is specified by an ascii file passed to Chart()
//    as an argument. The format of the file is as follows:
//
//    Line 1: <nPens> <tAccum> <write> <nSteps>
//    Line 2: <Title>
//    Line 3: <output-file>
//    ...   : <channel> <fLow> <fHigh>
//
//    The fields on the first 3 lines have the following meaning.
//
//    <nPens>        The number of pens to be displayed.
//    <tAccum>       The accumulation period in seconds. Data from each
//                   specified channel will be accumulated for the period 
//                   specified by <tAccum>
//    <write>        Specifies whether data are to be written to a log file.
//                   <write=0 indicates that no log file is to be written.
//    <nSteps>       The number of time intervals to be displayed on the strip 
//                   chart.
//    <Title>        A title for the chart. The title is displayed on the 
//                   strip chart window.
//    <output-file>  a fully qualified file name for the strip chart log. The
//                   file name may contain TimeStr format codes.
//
//    Following these lines comes a description of the contents of each pen.
//    One line must be specified for each pen containing:
//
//    <channel>      The name of the channel from which the data are derived.
//    <fLow>         the minimum fraquency.
//    <fHigh>        The maximum frequency.
//
//--------------------------------------  Strip descriptor class
class Strip {
public:
  Strip() {}
  ~Strip() {}
  float mFlow;
  float mFhi;
  int mChanID;
};

//--------------------------------------  Channel descriptor class
class Chan {
public:
  Chan() {}
  ~Chan () {}
  char mName[40];
  TSeries* mTS;
  FSeries* mFS;
};

void
Chart(const char* config) {
    int nPens = 0;
    int nStep = 0;
    float dTime = 1;
    bool writeData = false;
    float sum[16];
    char line[128], Title[128];

    //---------------------------------  Check file name
    if (!config || !*config) {
        cerr<< "Chart: configuration file not specified" << endl;
	return;
    }

    //---------------------------------  Get parameters.
    ifstream ifs(config, ios::in);
    if (!ifs) {
        cerr<< "Chart: Unable to open file " << config << endl;
	return;
    }

    //---------------------------------  Get the global parameters
    ifs.getline(line, sizeof(line));
    sscanf(line,"%i %f %i %i", &nPens, &dTime, &writeData, &nStep);
    ifs.getline(Title, sizeof(Title));
    if (nPens <= 0) {
        cerr << "Chart: Number of pens must be > 0" << endl;
	return;
    } else if (nPens > 16) {
        nPens = 16;
    }
    if (dTime <= 0) dTime = 1.0;
    if (nStep <= 0) nStep = 1000;

    //---------------------------------  Get the output file name
    ofstream ofs;
    char ofile[128];
    ifs.getline(line, sizeof(line));
    if (writeData) {
       TimeStr(Now(), ofile, line);
       ofs.open(ofile, ios::out);
    }

    //---------------------------------  Read in the pen descriptors.
    char chname[40];
    int nChan = 0;
    Chan  chanv[16];
    Strip stripv[16];
    for (int i=0 ; i<nPens ; i++) {
	ifs >> chname >> stripv[i].mFlow >> stripv[i].mFhi;
	int id;
	for (id=0 ; id<nChan ; id++) {
	    if (!strcmp(chname, chanv[id].mName)) break;
	}
	if (id >= nChan) {
	    id = nChan++;
	    strcpy(chanv[id].mName, chname);
	    chanv[id].mTS = new TSeries(Time(0), Interval(0.0), 4096);
	    chanv[id].mFS = new FSeries;
	    In.addChannel(chname, 0, &(chanv[id].mTS));
	}
	stripv[i].mChanID = id;
    }
    ifs.close();

    //---------------------------------  Print the configuration
    cout << "# Chart data from config file: " << config << endl;
    cout << "# Title: " << Title << endl;
    cout << "# NPens: " << nPens << " Sample interval: " << dTime 
	 << " Step Count: " << nStep << endl;
    if (writeData) {
        ofs << "# Chart data from config file: " << config << endl;
	ofs << "# Title: " << Title << endl;
	ofs << "# NPens: " << nPens << " Sample interval: " << dTime << endl;
	cout << "# Output written to: " << ofile << endl;
    } else {
        cout << "# No output file" << endl;
    }
    for (int i=0 ; i<nPens ; i++) {
        int id = stripv[i].mChanID;
        cout << "# Pen: " << i << " Channel: '" << chanv[id].mName
	     << "' Low Freq: " << stripv[i].mFlow << " High Freq: " 
	     << stripv[i].mFhi << endl;
	if (writeData) {
	    ofs << "# Pen: " << i << " Channel: " << chanv[id].mName
		<< " Low Freq: " << stripv[i].mFlow << " High Freq: " 
		<< stripv[i].mFhi << endl;
	}
    }

    //---------------------------------  define some constants.
    Interval FillTime(dTime);
    double tsecs = dTime;

    //---------------------------------  Create the strip chart.
    StripChart sc(nPens, Title);
    sc.setN(nStep);
    sc.setLogY(true);
    for (int p=0 ; p<nPens ; p++) {
        if (stripv[p]->mFlow >= 1.0) {
	    sprintf(line, "%s\n%.0f-%.0f Hz", chanv[stripv[p].mChanID].mName,
		    stripv[p].mFlow, stripv[p].mFhi);
	} else {
	    sprintf(line, "%s\n%.2f-%.2f Hz", chanv[stripv[p].mChanID].mName,
		    stripv[p].mFlow, stripv[p].mFhi);
	}
	sc.setTitle(p, line);
    }
    sc.setAxisType(StripChart::kLocalTime);

    //---------------------------------  Loop over sampling intervals
    gROOT->SetInterrupt(kTRUE);
    // In.setBuffer(2);
    bool first = true;
    TSeries* ts;
    FSeries* fs;
    Time t;
    double tbin;
    Strip* si;
    while (1) {
        gSystem->ProcessEvents();            // check for interrupts.
	if (gROOT->IsInterrupted()) break;

        int rc = In.fillData(FillTime);     // get a contiguous data sequence
	if (rc == -1) continue;             // make sure it worked.
	else if (rc)  break;

	for (int i=0 ; i<nChan ; i++) {
	    ts = chanv[i].mTS;
	    *ts += 0.0;        // subtract out DC off
	    *ts *= 1.0;        // Default (non-)calibration

	    fs = chanv[i].mFS;
	    fs->setData(*ts);   // FFT velocity to an FSeries
	    fs->tIntegral();    // Integral of velocity gives position
	}

	//------------------------------  Get RMS values from the FSeries.
	si = stripv;
	for (int i=0 ; i<nPens ; i++) {
	    fs = chanv[si->mChanID].mFS;
	    sum[i] = fs->Power(si->mFlow, si->mFhi);
	    if (sum[i] == 0.0) sum[i] = 1.0;
	    si++;
	}

	//------------------------------  Get the local time.
	t = ts.getStartTime();
	tbin = getUTC(t);

	//------------------------------  Set the scales on the first pass
	if (first) {
	    for (int i=0 ; i<nPens ; i++) sc.setStripe(i, sum[i], 10.0);
	    sc.setMaxX(tbin + nStep*dTime);
	    sc.setMinX(tbin);
	    first = false;
	}

	//------------------------------  Optional data logging
	if (writeData) {
	    ofs << t.getS() << " " << sum[0];
	    for (int i=1 ; i<nPens ; i++) ofs << " " << sum[i];
	    ofs << " " << endl; // Extra space to avoid CINT bug
	}

	//------------------------------  Update the plot.
	sc.plot(tbin, sum);
    }

    //---------------------------------  Clean up at the end
    for (int i=0 ; i<nChan ; i++) {
        In.rmChannel(chanv[i].mName);
	delete chanv[i].mTS;
	delete chanv[i].mFS;
    }
    if (writeData) ofs.close();
    cout << "Chart finished" << endl;
}


