/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef GENERATOR_HH
#define GENERATOR_HH

#include "DataSource.hh"
#include "GenDet.hh"
#include "GenChan.hh"
#include <vector>
#include <string>
class Dacc;
class Pipe;

/**  \section gener_intro DMT Data Generation Namespace
  *  The generator namespace contains the data classes used to simulate
  *  data for testing aspects Ligo data analysis and data monitoring 
  *  functions. Data are written to frames and may be processed by  
  *  unmodified analysis pipelines.
  *  The data generation classes include data source classes
  *  auxiliary function classes and a driver program class - DMTGen - that 
  *  allows the user to define and run a generation job.
  *
  *  \section gener_dsrc Data Sources
  *  Data source classes are all based on the generator::DataSource class.
  *  Each data source generates a specific waveform or fetches data from 
  *  a configurable source. Data source classes include:
  *  - DampedSine - Exponentially damped sinusoid
  *  - FrameData - %Channel data read from a specified frame set.
  *  - GaussBurst - Gaussian noise in a Gaussian envelope.
  *  - Sinc - %Sinc function
  *  - SineData - Sinusoid
  *  - SinGauss - %Sine Gaussian
  *  - WhiteNoise - Gaussian noise.
  *
  *  Utility classes include the following;
  *  - DaqErrFilter - Simulate random DAQ errors in the data.
  *  - DataSource   - Data source base class
  *  - GenChan      - %Channel generation class.
  *  - GenDet       - Detector data class.
  *  - Generator    - Generation driver class.
  *  - GenParam     - %Parameter management and randomization.
  *
  *  @memo DMT data simulation package.
  *  @author John Zweizig (john.zweizig@ligo.org)
  */
namespace generator {

  /**  The %Generator class generates simulated data for Monte Carlo studies 
    *  of LIGO analysis pipelines. The signals are optionally passed through 
    *  a response filter and then added to the noise. One or more signals 
    *  may be injected at fixed or random times. 
    *  Typical %Generator usage is as follows:
    *   - Construct an empty %Generator.
    *   - Configure the simulation with addSource(), addDetector() and 
    *     addChannel() methods.
    *   - Specify the generation time stride with setGenerStride()
    *   - Generate a stride at a time using the generate() method.
    *   - Reference data ad needed with refChannel() and refSource()
    *   - after processing/writing results, release data with 
    *     releaseChannelData() and releaseSourceData().
    *   - Iterate over data strides.
    *
    *  @memo DMT data generation class.
    *  @version 1.0 ; Modified December 27, 2004
    *  @author John G. Zweizig
    */
  class Generator {
  public:
    /**  Constructor a generator class with empty configuration.
      *  @memo Default constructor.
      */
    Generator(void);

    /**  Destroy a generator. Destroy and release all configured data 
      *  sources.
      *  @memo Destructor.
      */
    ~Generator(void);

    /**  Add a channel to the list of channels to be simulated. The channel
      *  definition includes a name, a detector identifier.
      *  @memo Add an entry to the list of channels to be generated.
      *  @param chan  %Channel name
      *  @param detID Detector index
      *  @param daq   DAQ simulation filter pipe.
      *  @return New channel index
      */
    int addChannel(const char* chan, int detID=0, const Pipe* daq=0);

    /**  Add default detector information for the specified ifo
      *  @memo Add specified ifo to the detector list.
      *  @param ifo Two digit standard IFO name (in the form "Xn")
      *  @return Index of the added detector.
      */
    int addDetector(const char* ifo);

    /**  Add a specified detector.
      *  @memo Add an IFO to the %Generator list.
      *  @param det Detector specifier.
      *  @return Index of the added detector.
      */
    int addDetector(const GenDet& det);

    /**  Create a copy of a data source and add the copy to the data source 
      *  list.
      *  @memo Add a named data source.
      *  @param name Name of the data source.
      *  @param data Data source to be cloned
      *  @return Source index.
      */
    int addSource(const std::string& name, const DataSource& data);

    /**  Add a data source object to the %Generator source list. The specified
      *  object is added to the list. Ownership of the %DataSource is passed
      *  to the %Generator class object.
      *  @memo Add an existing %DataSource to the %Generator source list.
      *  @param name Name of the new data source.
      *  @param data %DataSource object.
      *  @return Source index
      */
    int addSource(const std::string& name, DataSource* data=0);

    /**  Generate data for all sources in the specified time segment. The
      *  source data are then combined as specified to create the output
      *  channels.
      *  @memo Generate data for the specified time range.
      *  @param t0 Segment start time
      *  @param dT Length of time segment
      */
    void generate(const Time& t0, Interval dT);

    /**  Release data from all data sources up to the specified time.
      *  @memo Release source data
      *  @param t0 start of data to be retained.
      */
    void releaseSourceData(const Time& t0);

    /**  Release data from all channels up to the specified time.
      *  @memo Release channel data
      *  @param t0 start of data to be retained.
      */
    void releaseChannelData(const Time& t0);

    /**  Get the basic generation time interval.
      *  @memo Get the generation stride.
      *  @return Generation time stride.
      */
    Interval getGenerStride(void) const;

    /**  Get the latest time for which the defined channel data are complete.
      *  @memo Get the last generation time.
      *  @return Latest time for which channel data are complete.
      */
    Time getLatest(void) const;

    /**  Get the number of channels defined.
      *  @memo Get the number of channels.
      *  @return Get number of channels defined.
      */
    int getNChannel(void) const;

    /**  Get the number of detectors currently defined in the detector list.
      *  @memo Get the number of detectors.
      *  @return Number of defined detectors.
      */
    int getNDetector(void) const;

    /**  Get the number of data sources defined in the generator data 
      *  source list.
      *  @memo Get the number of data sources 
      *  @return Number of data sources defined in the source list.
      */
    int getNSource(void) const;

    /**  Print a statistical summary to the specified output stream.
      *  @memo Print statistics
      *  @param out Output stream
      *  @returns reference to the output stream
      */
    std::ostream& print_stats(std::ostream& out) const;
 
    /**  Get a constant reference to the specified channel ID.
      *  @memo Reference constant channel entry by ID.
      *  @param id channel index
      *  @return Constant reference to a generator channel class object
      */
    const GenChan& refChannel(int id) const;
 
    /**  Get a reference to the specified channel ID.
      *  @memo Reference channel entry by ID.
      *  @param id Channel index
      *  @return Reference to a generator channel class object
      */
    GenChan& refChannel(int id);

    /**  Get a constant reference to a Detector definition.
      *  @memo Reference constant detector entry by ID.
      *  @param id Detector index
      *  @return Constant reference to the specified generator detector object.
      */
    const GenDet& refDetector(int id) const;

    /**  Get a constant reference to a Detector definition.
      *  @memo Reference detector entry by ID.
      *  @param id Detector index
      *  @return Reference to the specified generator detector object.
      */
    GenDet& refDetector(int id);

    /**  Get a constant reference to the data source object with the specified
      *  index.
      *  @memo Reference constant source descriptor by ID.
      *  @param id Data source object index.
      *  @return Constant reference to the specified data source.
      */
    const DataSource& refSource(int id) const;

    /**  Get a reference to the data source object with the specified
      *  index.
      *  @memo Reference source descriptor by ID.
      *  @param id Data source object index.
      *  @return Reference to the specified data source.
      */
    DataSource& refSource(int id);

    /**  Get the data source index for the source with the specified name.
      *  @memo Get named source index.
      *  @param name %Data source name.
      *  @return Index of named DataSource.
      */
    int getSourceID(const std::string& name) const;

    /**  Set the debug verbosity level.
      *  @memo Set the debug level.
      *  @param lvl Level of debug information to be produced.
      */
    void setDebug(int lvl);

    /**  Get the basic generation time interval.
      *  @memo Get the generation stride.
      *  @param  dt Get generation time stride.
      */
    void setGenerStride(Interval dt);

  private:
    typedef std::vector<GenChan>    ChanList;
    typedef ChanList::iterator chan_iter;
    
    typedef std::vector<GenDet>     DetList;
    typedef DetList::iterator det_iter;
    
    typedef std::vector<DataSource*>  SourceList;
    typedef SourceList::iterator src_iter;
    
  private:
    Interval   mGenerStride;
    SourceList mSource;
    ChanList   mChannel;
    DetList    mDetect;
    int        mDebug;
  };
  
  //------------------------------------  Inline methods.
  inline Interval
  Generator::getGenerStride(void) const {
      return mGenerStride;
  }

  inline int 
  Generator::getNChannel(void) const {
      return mChannel.size(); 
  }

  inline int     
  Generator::getNDetector(void) const {
      return mDetect.size();
  }

  inline int     
  Generator::getNSource(void) const {
      return mSource.size();
  }

  inline const GenChan&
  Generator::refChannel(int id) const {
      return mChannel[id];
  }

  inline const GenDet&
  Generator::refDetector(int id) const {
      return mDetect[id];
  }

  inline const DataSource& 
  Generator::refSource(int id) const {
      return *mSource[id];
  }

  inline GenChan&
  Generator::refChannel(int id) {
      return mChannel[id];
  }

  inline GenDet&
  Generator::refDetector(int id) {
      return mDetect[id];
  }

  inline DataSource& 
  Generator::refSource(int id) {
      return *mSource[id];
  }
}

#endif
