#ifndef ENGINE_HH
#define ENGINE_HH

#include "eval_stack.hh"
#include <string>
#include <vector>
#include <iosfwd>

/**  The %Engine class parses a symbolic expression and stores the parsed 
  *  expression as a reverse polish list. The parsed expression can then 
  *  be evaluated an arbitrary number of times. The result(s) are left
  *  on the evaluation stack. Symbol replacement, function evaluation, and
  *  operator evaluaton may be specified by replacement of virtual functions.
  *  The operations implemented in the default version are specified by the
  *  OpsEnum enumerator. The operator symbols, values and the evaluation order
  *  are listed below.
  *  <table>
  *  <tr><td>\c + </td><td>kAdd</td><td>2</td>
  *      <td>\c - </td><td>kSub</td><td>2</td>
  *      <td>\c * </td><td>kMpy</td><td>1</td></tr>
  *  <tr><td>\c / </td><td>kDiv</td><td>1</td>
  *      <td>\c | </td><td>kOr</td><td>2</td>
  *      <td>\c \& </td><td>kAnd</td><td>2</td></tr>
  *  <tr><td>\c \&\& </td><td>kLogAnd</td><td>4</td>
  *      <td>\c || </td><td>kLogOr</td><td>4</td>
  *      <td>\c == </td><td>kCmpEq</td><td>3</td></tr>
  *  <tr><td>\c != </td><td>kCmpNE</td><td>3</td>
  *      <td>\c \<  </td><td>kCmpLt</td><td>3</td>
  *      <td>\c \<= </td><td>kCmpLE</td><td>3</td></tr>
  *  <tr><td>\c \> </td><td>kCmpGt</td><td>3</td>
  *      <td>\c \>= </td><td>kCmpGE</td><td>3</td>
  *      <td>\c \%  </td><td>kModulo</td><td>1</td></tr>
  *  </table>
  *  \brief Symbolic arithmetic evaluation engine. 
  *  \author J. Zweizig
  *  \version 1.0; Last Modified July 6, 2004
  */
class Engine {
public:
  /** Construct an evaluation engine.
    *  \brief Construct an engine
    * \param stk Evaluation stack.
    */
  Engine(eval_stack& stk);

  /**  Destroy the evaluation engine.
    *  \brief Destructor.
    */
  virtual ~Engine(void);

  /**  Parse the equation from the specified input stream.
    *  \brief Parse a stream
    *  \param in Input stream.
    */
  virtual void Parse(std::istream& in);

  /**  Evaluate an already parsed string.
    *  \brief Turn the crank
    */
  virtual void crank(void);

  /**  Push the data from the names symbol
    *  \brief Push the symbol content. 
    *  \param sym Symbol for which the data is to be pushed onto the spack.
    */
  virtual void push_symbol(const std::string& sym);

  /**  Push the specified constant value onto the evaluation stack.
    *  \brief Push the numeric data
    *  \param lit numeric literal string.
    */
  virtual void push_numeric(const std::string& lit);

  /**  Push the specified constant value onto the evaluation stack.
    *  \brief Push the literal data
    *  \param lit numeric literal string.
    */
  virtual void push_literal(const std::string& lit);

  /**  Evaluate a specified function.
    *  \brief Evaluate a function.
    *  \param func Function name to be evaluated.
    *  \param nArg Number of function arguments pushed onto the stack.
    */
  virtual void eval_funct(const std::string& func, int nArg);

  /**  Dumped the statement to be evaluated as parsed and stored in 
    *  reverse polish notation.
    *  \brief Dump the function to be evaluated.
    *  \param out Output stream to which the parsed statement is to be dumped.
    *  \return Stream reference.
    */
  std::ostream& dump_rpn(std::ostream& out) const;

  /**  Get the debug printout verbosity level.
    *  \brief Get the debug level.
    *  \return Debug level.
    */
  int getDebug(void)const;

  /**  Set a debug printout level. Debugging information is printed to 
    *  the cerr stream depending on the requested level of detail.
    *  \brief Set the debug level
    *  \param lvl Debug level.
    */
  void setDebug(int lvl);

  /**  Return a  constant reference to the evaluation stack.
    *  \brief Return a constant stack reference.
    *  \return Constant stack reference.
    */
  const eval_stack& refStack(void) const;

  /**  Return a reference to the evaluation stack.
    *  \brief Return a stack reference.
    *  \return Stack reference.
    */
  eval_stack& refStack(void);

  /**  Get a constant reference to a token from the top of the stack
    *  \brief Get a constant token reference.
    *  \param i Token position.
    *  \return Constan reference to the specified token
    */
  const eval_token& peek(int i=1) const;

private:
  /**  Internal recursive parsing method.
   */
   int recursive_parse(std::istream& in);

private:
  typedef std::vector<eval_stack::OpsEnum> op_stack;
  typedef std::vector<std::string> arg_stack;

private:
  int mLevel;
  int mDebug;
  eval_stack& mEvalStack;
  op_stack mOpList;
  arg_stack mArgList;
};

inline int
Engine::getDebug(void) const {
    return mDebug;
}

inline const eval_token& 
Engine::peek(int i) const {
    return mEvalStack.peek(i);
}

inline eval_stack& 
Engine::refStack(void) {
    return mEvalStack;
}

inline const eval_stack& 
Engine::refStack(void) const {
    return mEvalStack;
}

#endif // !defined(ENGINE_HH)
