#ifndef DatTEMPLATEMC_HH
#define DatTEMPLATEMC_HH

#include "DatEnv.hh"
#include "MonServer.hh"
#include "FixedLenTS.hh"
#include "Trend.hh"
#include "autopipe.hh"
#include <vector>
#include <string>

//
//    Data Monitor template class.

/**  @name DatTemplateMC
  *  @memo Template of a data environment (DatEnv) based monitor.
  *  The DatTemplate monitor is a template for a data monitor program.
  *  DatTemplate makes a simple calculation of the standard deviation of
  *  a specified channel (by default H0:PEM-LVEA_SEISX), serves the result 
  *  for display by the DMT viewer and stores the result as a minute trend.
  *
  *  {\bf {\Large Running DatTemplate}}
  *
  *  The DatTemplate command line syntax is:
  *  \begin{verbatim}
        DatTemplate [-channel <channel>] [-frames <number>] [-stride <time>] \
                    [-filter <filter>] [-settle <s-time>] [-dvIndex]
     \end{verbatim}
  *  The #<number># argument specifies the number of data strides to process
  *  before terminating the program. The #<time># argument specifies the data
  *  stride. If "-dvIndex" is specified, a trend index file will be written
  *  to '\$DMTOUTPUT/channel.cnf'.
  *
  *  {\bf {\Large Configuration File}}
  *
  *  DatTemplate uses a fixed configuration. There is therfore no 
  *  configuration file
  *
  *  {\bf {\Large DatTemplate Output}}
  *
  *  {\bf Trend channels}
  *  DatTemplate produces a minute trend of the sigma value called 
  *  #<channel>_sigma#
  *
  *  {\bf Monitor data channels}
  *  DatTemplate publishes a time series history of the sigma values 
  *  with the name #<channel>_sigma#
  *
  *  @author John Zweizig; Caltech/LIGO
  *  @version 1.1; Last modified August 12, 2004
  */
//{
//}

class Pipe;

class chan_data {
public:
    chan_data(const std::string& chan);
    chan_data(const chan_data& c);
    ~chan_data(void);
    float computeSigma(const TSeries& ts);
    const char* getChannel(void) const;
    const char* getSigmaName(void) const;
    const TSeries& refHistory(void) const;
    void setFilter(const std::string& f);
    void setSettle(Interval dt);
private:
    std::string mChannel;       //  Channel name
    std::string mFilter;        //  Filter description
    Interval    mSettle;        //  Settling time.
    std::string mSigmaName;     //  Channel data and trend name
    auto_pipe   mPipe;          // Pipe pointer
    FixedLenTS  mHistory;       //  X sigma history.
};

class DatTemplateMC : public DatEnv, MonServer {
public:
    DatTemplateMC(int argc, const char *argv[]); // Constructor
    ~DatTemplateMC(void);                        // Destructor
    void ProcessData(void);                      // Process 1 data stride
    void configure(const char* file);            // Configure channels
    void Attention(void);                        // Handle Message signal.
public:
    typedef std::vector<chan_data>    chan_list;
    typedef chan_list::iterator       chan_iter;
    typedef chan_list::const_iterator const_chan_iter;
private:
    int         maxFrame;       //  Number of frames to process
    Interval    mStep;          //  Time stride.
    Trend       mTrend;         //  Trend writer
    chan_list   mList;          //  Channel list
};

//======================================  Inline methods
inline const char*
chan_data::getChannel(void) const {
    return mChannel.c_str();
}

inline const char*
chan_data::getSigmaName(void) const {
    return mSigmaName.c_str();
}

inline const TSeries&
chan_data::refHistory(void) const {
    return mHistory;
}


#endif     //  DatTEMPLATEMC_HH
