/* -*- mode: c++; c-basic-offset: 4; -*- */
#ifndef FILTERDB_HH
#define FILTERDB_HH

#include "FilterModule.hh"
#include <map>
#include <string>

/**  Get a filter definition from the online filter database.
 */
class FilterDB {
public:
  /**  Construct an empty database.
    *  \brief Default constructor.
    */
  FilterDB(void);

  /**  Construct a database and initialize it with a filter definition file.
    *  \brief File constructor.
    *  \param file Filter module definition file path.
    */
  explicit FilterDB(const char* file);

  /**  Return a reference to the specified filter module.
    *  \brief Filter module accessor
    *  \param f Filter module anems
    *  \return reference to filter module definition.
    */
  const FilterModule& operator[](const std::string& f) const;

  /** Reset the filter database.
    */
  void reset(void);

  /**  Initialize a filter dataabase from the filter module description file.
    *  \param file File path.
    */
  void init(const char* file);

  /**  Insert a filter module into the database.
    *  \param name Filter module name
    *  \param filter Filter module to be inserted.
    */
  void insert(const std::string& name, const FilterModule& filter);

  /**  Test if filter database is empty.
    *  \return True if no filter modules have been defined.
    */
  bool empty(void) const;

  /**  Test if the specified filter name is defined.
    *  \param name Filter module name to test.
    *  \return True if the named filter module has been defined.
    */
  bool exists(const std::string& name) const;

private:
  typedef std::map<std::string, FilterModule> FilterMap;
  typedef FilterMap::const_iterator const_map_iter;
  typedef FilterMap::iterator       map_iter;
  FilterMap  mMap;
};

//======================================  inline methods
inline bool
FilterDB::empty(void) const {
    return mMap.empty();
}

inline bool
FilterDB::exists(const std::string& name) const {
    return mMap.find(name) != mMap.end();
}

#endif  // FILTERDB_HH
