#include "Histogram1.hh"
#include "Time.hh"
#include <cmath>
#include <cstring>
#include <iostream>
#include <iomanip>
#include <iosfwd>

   using namespace std;

//-------------------------------------- Data Constructor
   Histogram1::Histogram1(const char* name, int nbinx, xbin_t xmin, xbin_t xmax,
                     const char* xlabel, const char* nlabel):fArray(0),fSumw2(0),fXbins(0)
   {
      Reset();
   
      if (name)   fTitle  = name;
      if (xlabel) fXLabel = xlabel;
      if (nlabel) fNLabel = nlabel;
   
      SetBinLowEdges(nbinx,xmin,xmax);
   
   }

//-------------------------------------- Data Constructor
   Histogram1::Histogram1(const char* name, int nbinx, const xbin_t* xbins,
                     const char* xlabel, const char* nlabel):fArray(0),fSumw2(0),fXbins(0)
   {
      Reset();
   
      if (name)   fTitle  = name;
      if (xlabel) fXLabel = xlabel;
      if (nlabel) fNLabel = nlabel;
   
      SetBinLowEdges(nbinx,xbins);
   
   }

//-------------------------------------- Destructor
   Histogram1::~Histogram1()
   {
      if(fArray) delete[] fArray;
      if(fSumw2) delete[] fSumw2;
      if(fXbins) delete[] fXbins;
   }


//-------------------------------------- Find bin number (internal use only)
   int Histogram1::SearchBin(int min, int max, xbin_t x) const
   {
      if ( x >= fXbins[min] && x < fXbins[min+1] ) 
         return min;
      else if ( x >= fXbins[min] && x < fXbins[(max+min)/2+1] ){
         return SearchBin(min , (max+min)/2, x);
      }
      return  SearchBin((max+min)/2+1 , max, x);
   }

//-------------------------------------- Allocate memory (internal use only)
   void Histogram1::Allocate(int nbin)
   {
      if(fArray){
         delete[] fArray;
         delete[] fXbins;
      }
      if(fSumw2) delete[] fSumw2;
   
      fArray = 0;
      fSumw2 = 0;
      fXbins = 0;
      fNbinx = nbin;
   
      if(nbin){
         fArray = new histdata_t[nbin+2];
         fXbins = new xbin_t[nbin+1];
         memset (fArray, 0, (nbin+2) * sizeof (histdata_t));
         memset (fXbins, 0, (nbin+1) * sizeof (xbin_t));
      }
   }

//======================================

   void Histogram1::Clear()
   {
      if (fNbinx) {
         memset (fArray, 0, (fNbinx+2) * sizeof (histdata_t));
         if (fBinErrorFlag) {
            memset (fSumw2, 0, (fNbinx+2) * sizeof (stat_t));
         }
      }
   
      fTsumw = fTsumw2 = fTsumwx = fTsumwx2 = 0;
   
      fNEntries = 0;
      fTime = Time(0);
   }

   ostream&
   Histogram1::Dump(ostream& out) const
   {
      out<<"Title                 : "<<fTitle<<endl
         <<"XLabel                : "<<fXLabel<<endl
         <<"NLabel                : "<<fNLabel<<endl
         <<"GPS Time              : "<<fTime.totalS()<<endl
         <<"# of Entries          : "<<fNEntries<<endl
         <<"# of Bins             : "<<fNbinx<<endl
         <<"Bin Type              : ";
      switch (fBinType) {
         case kUndefinedBin:
            out<<"Undefined"<<endl;
            break;
         case kFixedBin:
            out<<"Fixed"<<endl;
            break;
         default:
            out<<"Variable"<<endl;
      }
      out<<"Sum of Weights        : "<<fTsumw<<endl
         <<"Sum of Weights^2      : "<<fTsumw2<<endl
         <<"Sum of Weights*data   : "<<fTsumwx<<endl
         <<"Sum of Weights*data^2 : "<<fTsumwx2<<endl;
      if (fBinErrorFlag) out<<"Bin Error ON"<<endl;
      else out<<"Bin Error OFF"<<endl;
      out<<"-----------+--------+---------"<<endl
         <<setw(11)<<"Low Edge"
         <<"|"
         <<setw(8)<<"Content"
         <<"|"
         <<setw(8)<<"Error^2"<<endl
         <<"-----------+--------+---------"<<endl;
      if (fNbinx){  
         if (fBinErrorFlag) {
            for(int i=0 ; i<fNbinx+2; i++){
               if(!i) {
                  out<<setw(11)<<"Underflow"
                     <<"|"
                     <<setw(8)<<fArray[i]
                     <<"|"
                     <<setw(8)<<fSumw2[i]
                     <<endl;
               }
               else {
                  out<<setw(11)<<fXbins[i-1]
                     <<"|"
                     <<setw(8)<<fArray[i]
                     <<"|"
                     <<setw(8)<<fSumw2[i]
                     <<endl;
               }
            }
         }
         else {
            for(int i=0 ; i<fNbinx+2; i++){
               if(!i) {
                  out<<setw(11)<<"Underflow"
                     <<"|"
                     <<setw(8)<<fArray[i]
                     <<"|"
                     <<endl;
               }
               else {
                  out<<setw(11)<<fXbins[i-1]
                     <<"|"
                     <<setw(8)<<fArray[i]
                     <<"|"
                     <<endl;
               }
            }
         }
      }
      else out<<"No Data"<<endl;
   
      return out;
   }

   void Histogram1::Fill(xbin_t x, double w)
   {
   
      if (fBinType==kFixedBin){  // fixed bin
         if ( x >= fXbins[fNbinx] ) {  // overflow
            fArray[fNbinx+1] += w;
            if (fBinErrorFlag) fSumw2[fNbinx+1] += w*w;
         }
         else if ( x < fXbins[0] ) {  // underflow
            fArray[0] += w; 
            if (fBinErrorFlag) fSumw2[0] += w*w;
         }
         else {
            int index = 1 + (int)(( x - fXbins[0] ) * fNbinx / ( fXbins[fNbinx] - fXbins[0] ));
            fArray[index] += w;
            if (fBinErrorFlag) fSumw2[index] += w*w;
            fTsumw += w;
            fTsumw2 += w*w;
            fTsumwx += w*x;
            fTsumwx2 += w*x*x;
         }
         fNEntries++;
      }
      else if (fBinType==kVariableBin){  // variable bin
         int index = GetBinNumber(x);
         fArray[index] += w;
         if (fBinErrorFlag) fSumw2[index] += w*w;
      
         if (index>0 && index<fNbinx+1){
            fTsumw += w;
            fTsumw2 += w*w;
            fTsumwx += w*x;
            fTsumwx2 += w*x*x;
         }
         fNEntries++;
      }
   }

   void Histogram1::FillN(int ntimes, const xbin_t* xp)
   {
   
      if (fBinType==kFixedBin){  // fixed bin
	  xbin_t x0   = fXbins[0];
	  xbin_t xEnd = fXbins[fNbinx];
	  xbin_t xBin = xbin_t(fNbinx)/(xEnd - x0);
	  long   nBinInc = 0;
	  for ( int i = 0; i < ntimes; i++) {
	      register xbin_t x = xp[i];
	      if ( x >= xEnd ) {  // overflow
		  fArray[fNbinx+1] += 1.0;
		  if (fBinErrorFlag) fSumw2[fNbinx+1] += 1.0;
	      }	else if ( x < x0 ) {  // underflow
		  fArray[0] += 1.0;
		  if (fBinErrorFlag) fSumw2[0] += 1.0;
	      }	else {
		  int index = int(( x - x0 ) * xBin);
		  nBinInc++;
		  fTsumwx  += x;
		  fTsumwx2 += x*x;
		  fArray[index+1] += 1.0;
		  if (fBinErrorFlag) fSumw2[index+1] += 1.0;
	      }
	  }
	  fNEntries += ntimes;
	  fTsumw    += xbin_t(nBinInc);
	  fTsumw2   += xbin_t(nBinInc);
      }
      else if (fBinType==kVariableBin){  // variable bin
	 for ( int i = 0; i < ntimes; i++) {
	     register xbin_t x = xp[i];
	     int index = GetBinNumber(x);
	     fArray[index] += 1.0;
	     if (fBinErrorFlag) fSumw2[index] += 1.0;
      
	     if (index>0 && index<fNbinx+1){
	         fTsumw   += 1.0;
		 fTsumw2  += 1.0;
		 fTsumwx  += x;
		 fTsumwx2 += x*x;
	     }
	 }
	 fNEntries += ntimes;
      }
   }

   void Histogram1::FillN(int ntimes, const xbin_t* x, const double* w)
   {
      for ( int i = 0; i < ntimes; i++) { Fill(x[i],w[i]); }
   }

   void Histogram1::GetBinContents(histdata_t* data) const
   {
      // for( int i = 0 ; i < fNbinx+2 ; i++) { data[i] = fArray[i]; }
      memcpy (data, fArray, (fNbinx+2) * sizeof(histdata_t));
   }

   void Histogram1::GetBinLowEdges(xbin_t* data) const
   {
      // for( int i = 0 ; i < fNbinx+1 ; i++) { data[i] = fXbins[i]; }
      memcpy (data, fXbins, (fNbinx+1) * sizeof(xbin_t));
   }

   bool Histogram1::GetBinErrors(stat_t* err) const
   {
      if (fBinErrorFlag) {
         for( int i = 0 ; i < fNbinx+2 ; i++) { err[i] = sqrt(fSumw2[i]); }
      }
      return fBinErrorFlag;
   }

   Histogram1::xbin_t 
   Histogram1::GetBinLowEdge(int n) const
   {
      if(n < 1) n = 1;
      else if (n >= fNbinx+2) n = fNbinx+1; 
   
      return fXbins[n-1] ;
   }

   Histogram1::histdata_t 
   Histogram1::GetBinContent(int n) const
   { 
      if(n < 0) n = 0;
      else if (n >= fNbinx+2) n = fNbinx+1;
      return fArray[n];
   }

   Histogram1::xbin_t 
   Histogram1::GetBinCenter(int n) const
   {
      if(n < 1) n = 1;
      else if (n >= fNbinx+1) n = fNbinx;  
      return (fXbins[n-1] + fXbins[n])/2;
   }


   Histogram1::stat_t 
   Histogram1::GetBinError(int n) const
   {
      if (!fBinErrorFlag) 
         return -1;
      if(n < 0) n = 0;
      else if (n >= fNbinx+2) n = fNbinx+1;
      return sqrt(fSumw2[n]);
   }

   int Histogram1::GetBinNumber(xbin_t x) const
   {
      if ( x < fXbins[0] ) 
         return 0;
      else if ( x > fXbins[fNbinx]) 
         return fNbinx+1;
      return SearchBin(0,fNbinx-1,x) + 1;  
   }

   Histogram1::xbin_t 
   Histogram1::GetBinSpacing() const
   {
      if (fBinType == kFixedBin) 
         return (fXbins[fNbinx]-fXbins[0])/fNbinx;
      else 
         return 0;
   }

   const char* Histogram1::GetTitle() const {
      if ( fTitle.size() ) 
         return fTitle.c_str();
      else 
         return "";
   }

   const char* Histogram1::GetXLabel() const {
      if ( fXLabel.size() ) 
         return fXLabel.c_str();
      else 
         return "";
   }

   const char* Histogram1::GetNLabel() const {
      if ( fNLabel.size() ) 
         return fNLabel.c_str();
      else 
         return "";
   }

   Histogram1::histdata_t 
   Histogram1::GetMinContent(void) const
   {
      histdata_t min = fArray[1];
      for ( int i = 2 ; i < fNbinx+1 ; i++) {
         if ( fArray[i] < min ) min = fArray[i];
      }
      return min;
   }

   Histogram1::histdata_t 
   Histogram1::GetMaxContent(void) const
   {
      histdata_t max = fArray[1];
      for ( int i = 2 ; i < fNbinx+1 ; i++) {
         if ( fArray[i] > max ) max = fArray[i];
      }
      return max;
   }

   int Histogram1::GetMinContentBin(void) const
   {
      histdata_t min = fArray[1];
      int minIndex = 1;
   
      for ( int i = 2; i < fNbinx+1; i++){
         if ( fArray[i] < min ) {
            min = fArray[i];
            minIndex = i;
         }
      }
   
      return minIndex;
   }

   int Histogram1::GetMaxContentBin(void) const
   {
      histdata_t max = fArray[1];
      int maxIndex = 1;
   
      for ( int i = 2; i < fNbinx+1; i++){
         if ( fArray[i] > max ) {
            max = fArray[i];
            maxIndex = i;
         }
      }
   
      return maxIndex;
   }

   Histogram1::stat_t 
   Histogram1::GetMean(void) const
   {
      stat_t stats[4];
      GetStats(stats);
      if (stats[0] == 0) 
         return 0;
      return stats[2]/stats[0];
   }

   Histogram1::stat_t 
   Histogram1::GetSdev(void) const
   { 
      stat_t stats[4];
      GetStats(stats);
      if (stats[0] == 0) 
         return 0;
      return sqrt(stats[3]/stats[0]-stats[2]*stats[2]/(stats[0]*stats[0]));
   }

   void Histogram1::GetStats(stat_t *stats) const
   {
      int i;
      histdata_t x,w;
   
      if(fTsumw == 0 ){
         for( i=0 ; i<4 ; i++ ) stats[i]=0;
         for ( i=1 ; i<fNbinx+1; i++ ){
            x = GetBinCenter(i);
            w = GetBinContent(i);
            stats[0] += w;
            stats[1] += w*w;
            stats[2] += w*x;
            stats[3] += w*x*x;
         }
      }
      else {
         stats[0] = fTsumw;
         stats[1] = fTsumw2;
         stats[2] = fTsumwx;
         stats[3] = fTsumwx2;
      }
   }

   void Histogram1::PutStats(const stat_t *stats)
   {
      fTsumw   = stats[0];
      fTsumw2  = stats[1];
      fTsumwx  = stats[2];
      fTsumwx2 = stats[3];
   }

   void Histogram1::SetBinLowEdges(int nbinx, xbin_t xmin, xbin_t xmax)
   {
      fNEntries = 0;
   
      Allocate(nbinx);
      if ( !fNbinx ) 
         return;
   
      fXbins[0] = xmin;
      double dx = (xmax - xmin)/nbinx;
      for(int i=0;i<fNbinx+1;i++) fXbins[i] = fXbins[0] + dx * i;
   
      fBinType = kFixedBin;
   
   }

   void Histogram1::SetBinLowEdges(int nbinx, const xbin_t* xbins)
   {  
      fNEntries = 0;
   
      Allocate(nbinx);
      if ( !fNbinx ) 
         return;
   
      memcpy (fXbins, xbins, (fNbinx+1) * sizeof(xbin_t));
   
      fBinType = kVariableBin;
   
   }

   void Histogram1::SetBinContents(const histdata_t* data)
   {
      for (int i=0 ; i<fNbinx+2; i++) { SetBinContent(i,data[i]); }
   }

   void Histogram1::SetBinContent(int n, histdata_t content)
   {
      if ( n<0 || n>fNbinx+1) 
         return;
      fArray[n] = content;
   }

   void Histogram1::SetBinErrors(const stat_t* err)
   {
      Sumw2();
      for (int i=0 ; i<fNbinx+2; i++) { SetBinError(i,err[i]); }
   }

   bool Histogram1::SetBinError(int n, stat_t err)
   { 
      if (fBinErrorFlag){
         if ( n>=0 && n<=fNbinx) { fSumw2[n] = err*err; } 
      }
      return fBinErrorFlag;
   }

   void Histogram1::SetBinType(int n)
   {
      switch(n){
         case 1:
            fBinType = kFixedBin;
            break;
         case 2:
            fBinType = kVariableBin;
            break;
         default:
            fBinType = kUndefinedBin;
      }
   }

   void Histogram1::Sumw2(bool reset)
   {
      if (fSumw2) delete[] fSumw2;
      if ( !fNbinx ) 
         return;
   
      fSumw2 = new stat_t[fNbinx+2];
   
      fBinErrorFlag = 1;
   
      if (reset) memset(fSumw2,0,(fNbinx+2) * sizeof(stat_t));
      else memcpy(fSumw2, fArray,(fNbinx+2) * sizeof(stat_t));
   
   }

   void Histogram1::Reset(void)
   {
      Allocate(0);
   
      fTitle = "";
      fNEntries = 0;
      fTsumw = fTsumw2 = fTsumwx = fTsumwx2 = 0.;
      fXLabel = "";
      fNLabel = "";
      fBinType = kUndefinedBin;
      fTime = Time(0);
      fBinErrorFlag = 0;
   }

   Histogram1& Histogram1::operator = (const Histogram1& h)
   {
      if (this != &h){
         Reset();
      
         fTitle = h.fTitle;
         Allocate(h.fNbinx);
         if ( h.fNbinx ) {
            memcpy(fArray, h.fArray, (fNbinx+2)*sizeof(histdata_t));
            memcpy(fXbins, h.fXbins, (fNbinx+1)*sizeof(xbin_t));
         
            if (h.IsErrorFlagON()) {
               Sumw2();
               memcpy(fSumw2, h.fSumw2, (fNbinx+2)*sizeof(stat_t));
            }
         }
      
         fNEntries = h.fNEntries;
         fTsumw = h.fTsumw;
         fTsumw2 = h.fTsumw2;
         fTsumwx = h.fTsumwx;
         fTsumwx2 = h.fTsumwx2;
         fXLabel = h.fXLabel;
         fNLabel = h.fNLabel;
         fBinType = h.fBinType;
         fTime = h.fTime;
      }
   
      return (*this);
   }

   Histogram1& Histogram1::operator += (const Histogram1& h)
   {
      if (!fBinErrorFlag && h.IsErrorFlagON()) Sumw2();
   
      for (int i=0; i<fNbinx+2; i++) { 
         fArray[i] += h.fArray[i];
         if (fBinErrorFlag) fSumw2[i] += h.fSumw2[i];
      }
   
      stat_t s1[4],s2[4];
      GetStats(s1);
      h.GetStats(s2);
   
      for (int i=0; i<4; i++) { s1[i] += s2[i]; }
   
      fNEntries += h.fNEntries;
   
      PutStats(s1);
   
      return (*this);
   }


   Histogram1& Histogram1::operator += (histdata_t bias)
   {
      for (int i=0; i<fNbinx+2; i++) { fArray[i] += bias; }
   
      fNEntries = fNbinx+2;
      fTsumw = fTsumw2 = fTsumwx = fTsumwx2 = 0;
      stat_t stats[4];
      GetStats(stats);
      PutStats(stats);
   
      return (*this);
   }

   Histogram1& Histogram1::operator -= (const Histogram1& h)
   {
      Histogram1 h0(h);
   
      if (!fBinErrorFlag && h0.IsErrorFlagON()) Sumw2();
   
      for (int i=0; i<fNbinx+2; i++) {
         fArray[i] -= h0.fArray[i];
         if (fBinErrorFlag) fSumw2[i] += h0.fSumw2[i];
      }
   
   
      stat_t s1[4],s2[4];
      GetStats(s1);
      h0.GetStats(s2);
   
      for (int i=0; i<4; i++) s1[i] += s2[i];
   
      fNEntries -= h0.fNEntries;
   
      PutStats(s1);
   
      return (*this);
   }

   Histogram1& Histogram1::operator *= (double scale)
   {
      stat_t stats[4];
      GetStats(stats);
      stats[0] *= scale;
      stats[1] *= ( scale * scale );
      stats[2] *= scale;
      stats[3] *= scale;
   
      PutStats(stats);
   
      for (int i=0; i<fNbinx+2; i++) {
         fArray[i] *= scale;
         if (fBinErrorFlag) fSumw2[i] *= ( scale * scale );
      }
      return (*this);
   }

   Histogram1& Histogram1::operator *= (const Histogram1& h)
   { 
      Histogram1 h0(h);
   
      if (!fBinErrorFlag && h0.IsErrorFlagON()) Sumw2();
   
      for (int i=0; i<fNbinx+2; i++) {
         histdata_t c0 = fArray[i];
         histdata_t c1 = h0.fArray[i];
         stat_t e0 = GetBinError(i);
         stat_t e1 = h0.GetBinError(i);
         if (fBinErrorFlag) fSumw2[i] = e0*e0*c1*c1+e1*e1*c0*c0;
         fArray[i] = c0*c1;
      }
   
      fNEntries = fNbinx+2;
      fTsumw = fTsumw2 = fTsumwx = fTsumwx2 = 0;
      stat_t stats[4];
      GetStats(stats); 
      PutStats(stats);
   
      return (*this);
   }

   Histogram1& Histogram1::operator /= (const Histogram1& h)
   {
      Histogram1 h0(h);
   
      if (!fBinErrorFlag && h0.IsErrorFlagON()) Sumw2();
   
      for (int i=0; i<fNbinx+2; i++) {
         histdata_t c0 = fArray[i];
         histdata_t c1 = h0.fArray[i];
         stat_t e0 = GetBinError(i);
         stat_t e1 = h0.GetBinError(i);
         if (!c1) { 
            if (fBinErrorFlag) fSumw2[i]=0;
            fArray[i]=0;
         }
         else {
            if (fBinErrorFlag) fSumw2[i] = (e0*e0*c1*c1+e1*e1*c0*c0)/(c1*c1*c1*c1);
            fArray[i] = c0/c1;
         }
      }
   
      fNEntries = fNbinx+2;
      fTsumw = fTsumw2 = fTsumwx = fTsumwx2 = 0;
      stat_t stats[4];
      GetStats(stats); 
      PutStats(stats);
   
      return (*this);
   }


//=================External Operator=====================


   Histogram1 operator + (Histogram1::histdata_t bias, const Histogram1& h)
   {
      Histogram1 hbias(h);
      hbias += bias;
   
      return hbias;
   
   }

   Histogram1 operator + (const Histogram1& h, Histogram1::histdata_t bias)
   {
      Histogram1 hbias(h);
      hbias += bias;
   
      return hbias;
   
   }

   Histogram1 operator + (const Histogram1& h1, const Histogram1& h2)
   {
      Histogram1 hsum(h1);
      hsum += h2;
   
      return hsum;
   }

   Histogram1 operator - (const Histogram1& h1, const Histogram1& h2)
   {
      Histogram1 hsub(h1);
      hsub -= h2;
   
      return hsub;
   }

   Histogram1 operator * (double scale, const Histogram1& h)
   {
      Histogram1 hscl(h);
      hscl *= scale;
   
      return hscl;
   }


   Histogram1 operator * (const Histogram1& h, double scale)
   {
      Histogram1 hscl(h);
      hscl *= scale;
   
      return hscl;
   }

   Histogram1 operator * (const Histogram1& h1, const Histogram1& h2)
   {
      Histogram1 hmlt(h1);
      hmlt *= h2;
   
      return hmlt;
   }

   Histogram1 operator / (const Histogram1& h1, const Histogram1& h2)
   {
      Histogram1 hdiv(h1);
      hdiv /= h2;
   
      return hdiv;
   }
