/* -*- mode: c++; c-basic-offset: 3; -*- */
#ifndef SENDS_THREAD_BASE
#define SENDS_THREAD_BASE
#include <pthread.h>


namespace thread {

    /**  Thread_base is a base class for threads. There is a single
      *  entry point for the thread.
      */
    class thread_base {
    public:
	/**  Construct the thread base class.
	  */
	thread_base(void);

	/**  Destroy the thread_base class.
	  */
	virtual ~thread_base(void);
 
	/**  Test whether the thread is currently executing.
	  *  \brief Test the busy flag.
	  *  \return Busy flag state.
	  */
	bool busy(void) const;

        /**  Wait for the thread to complete and store the return value
	  *  pointer in the specified location.
	  *  \brief Join the current thread.
	  *  \param retcd Pointer to a location to receive the exit value.
	  *  \return Zero if the join completed.
	  */
        int join(void** retcd) const;

	/**  Send a signal to this thread.
	  *  \brief Send a signal.
	  *  \param sig Signal number to be sent to the thread.
	  *  \return pthread_kill return code.
	  */
	virtual int kill_thread(int sig);

	/**  Test the thread run-enable flag.
	  *  \brief test the run state.
	  *  \return State of the run-enable flag.
	  */
	bool run(void) const;

	/**  Set the detached flag to the specified state.
	  *  \brief Set the detached flag.
	  *  \param d If tru, the thread is to be run detached.
	  */
	void set_detached(bool d);

	/**  Set the minimum stack size. By default, thread_base uses 
	  *  a stack size of 1MB. If the requested stack size is less 
	  *  than a safe minimum (16kB), the minimum will be used.
	  *  \brief Set the stack size
	  *  \param ssiz Minimum stack size in bytes.
	  */
	void set_stacksize(long ssiz);

	/**  Start the thread. The run flag is set and the busy flag is set 
	  *  if the thread is successfully started.
	  *  \return pthread_create error code.
	  */
	virtual int start_thread(void);

	/**  Stop the thread by clearing its run-enable flag.
	  *  \brief Clear the run-enable flag.
	  *  \return Zero on success.
	  */
	virtual int stop_thread(void);

	/**  Method to be executed when the thread is started.
	  */
	virtual void* thread_entry(void) = 0;
    private:
	/**  thread_base stub method to handle thread status flags.
	  */
	void* thread_stub(void);
	static void* launch(void* p);
    private:
	volatile bool  mRun;
	volatile bool  mBusy;
	pthread_t      mThreadID;
	pthread_attr_t mThreadAttr;
    };

    //==================================  Inline methods
    inline bool
    thread_base::busy(void) const {
	return mBusy;
    }

    inline bool
    thread_base::run(void) const {
	return mRun;
    }
}

#endif // !defined(SENDS_THREAD_BASE)
